!
!    "$Id: ex10.f,v 1.18 1998/04/21 18:33:02 balay Exp bsmith $";
!
!  This example demonstrates basic use of the SNES Fortran interface.
!
!  Note: This example is the same as ex12.F except that it uses the
!        Fortran .f suffix rather than the .F suffix.  Note the different
!        include syntax and that all PETSc objects are defined to be integers.
!
!
!  In this example the application context is a Fortran integer array:
!      ctx(1) = da    - distributed array
!          2  = F     - global vector where the function is stored
!          3  = xl    - local work vector
!          4  = rank  - processor rank
!          5  = size  - number of processors
!          6  = N     - system size
!
!  Note: Any user-defined Fortran routines (such as FormJacobian)
!  MUST be declared as external.

      implicit none

      include '/home/bsmith/petsc/include/foldinclude/petsc.h'
      include '/home/bsmith/petsc/include/foldinclude/vec.h'
      include '/home/bsmith/petsc/include/foldinclude/da.h'
      include '/home/bsmith/petsc/include/foldinclude/mat.h'
      include '/home/bsmith/petsc/include/foldinclude/snes.h'

      integer          ctx(8),ierr,N,start,end,nn,i,ii,its,flg
      integer          snes
      integer          J
      integer          x,r,u,F
      double precision xp,FF,UU,h
      character*(10)   matrixname
      external         FormJacobian, FormFunction

      call PetscInitialize(PETSC_NULL_CHARACTER,ierr)
      N = 10
      call OptionsGetInt(PETSC_NULL_CHARACTER,'-n',N,flg,ierr)
      h = 1.d0/(N-1.d0)
      ctx(6) = N

      call MPI_Comm_rank(PETSC_COMM_WORLD,ctx(4),ierr)
      call MPI_Comm_size(PETSC_COMM_WORLD,ctx(5),ierr)

! Set up data structures
      call DACreate1d(PETSC_COMM_WORLD,DA_NONPERIODIC,N,1,1,            &
     &     PETSC_NULL_INTEGER,ctx(1),ierr)

      call DACreateGlobalVector(ctx(1),x,ierr)
      call DACreateLocalVector(ctx(1),ctx(3),ierr)

      call PetscObjectSetName(x,'Approximate Solution',ierr)
      call VecDuplicate(x,r,ierr)
      call VecDuplicate(x,F,ierr)
      ctx(2) = F
      call VecDuplicate(x,U,ierr)
      call PetscObjectSetName(U,'Exact Solution',ierr)

      call MatCreateMPIAIJ(PETSC_COMM_WORLD,PETSC_DECIDE,PETSC_DECIDE,N, &
     &     N,3,PETSC_NULL_INTEGER,0,PETSC_NULL_INTEGER,J,ierr)

      call MatGetType(J,PETSC_NULL_INTEGER,matrixname,ierr)

! Store right-hand-side of PDE and exact solution
      call VecGetOwnershipRange(x,start,end,ierr)
      xp = h*start
      nn = end - start
      ii = start
      do 10, i=0,nn-1
        FF = 6.d0*xp + (xp+1.d-12)**6.d0
        UU = xp*xp*xp
        call VecSetValues(F,1,ii,FF,INSERT_VALUES,ierr)
        call VecSetValues(U,1,ii,UU,INSERT_VALUES,ierr)
        xp = xp + h
        ii = ii + 1
 10   continue
      call VecAssemblyBegin(F,ierr)
      call VecAssemblyEnd(F,ierr)
      call VecAssemblyBegin(U,ierr)
      call VecAssemblyEnd(U,ierr)

! Create nonlinear solver
      call SNESCreate(PETSC_COMM_WORLD,SNES_NONLINEAR_EQUATIONS,        &
     &     snes,ierr)

! Set various routines and options
      call SNESSetFunction(snes,r,FormFunction,ctx,ierr)
      call SNESSetJacobian(snes,J,J,FormJacobian,ctx,ierr)
      call SNESSetFromOptions(snes,ierr)

! Solve nonlinear system
      call FormInitialGuess(snes,x,ierr)
      call SNESSolve(snes,x,its,ierr)

! Write results if first processor
      if (ctx(4) .eq. 0) then
        write(6,100) its
      endif
  100 format('Number of Newton iterations = ',i5)

! Free data structures
      call VecDestroy(x,ierr)
      call VecDestroy(ctx(3),ierr)
      call VecDestroy(r,ierr)
      call VecDestroy(U,ierr)
      call VecDestroy(F,ierr)
      call MatDestroy(J,ierr)
      call SNESDestroy(snes,ierr)
      call DADestroy(ctx(1),ierr)
      call PetscFinalize(ierr)
      end


! --------------------  Evaluate Function F(x) --------------------- 

      subroutine FormFunction(snes,x,f,ctx)
      integer          snes, x, f, ctx(*)
      integer          rank, size, i, s, n, ierr
!
!   This code is not portable, on 64 bit machines use integer*8
!
      integer      ixx, iff, iF2
      double precision h, d, vf2(1), vxx(1), vff(1)
      include '/home/bsmith/petsc/include/foldinclude/petsc.h'
      include '/home/bsmith/petsc/include/foldinclude/vec.h'
      include '/home/bsmith/petsc/include/foldinclude/da.h'
      include '/home/bsmith/petsc/include/foldinclude/mat.h'
      include '/home/bsmith/petsc/include/foldinclude/snes.h'

      rank  = ctx(4)
      size  = ctx(5)
      h     = 1.d0/(ctx(6) - 1.d0)
      call DAGlobalToLocalBegin(ctx(1),x,INSERT_VALUES,ctx(3),ierr)
      call DAGlobalToLocalEnd(ctx(1),x,INSERT_VALUES,ctx(3),ierr)

      call VecGetLocalSize(ctx(3),n,ierr)
      if (n .gt. 1000) then 
        print*, 'Local work array not big enough'
        call MPI_Abort(PETSC_COMM_WORLD,0,ierr)
      endif

!
! This sets the index ixx so that vxx(ixx+1) is the first local
! element in the vector indicated by ctx(3).
!
      call VecGetArray(ctx(3),vxx,ixx,ierr)
      call VecGetArray(f,vff,iff,ierr)
      call VecGetArray(ctx(2),vF2,iF2,ierr)
!
      d = h*h

      if (rank .eq. 0) then 
        s = 0
        vff(iff+1) = vxx(ixx+1)
      else
        s = 1
      endif

      do 10 i=1,n-2
       vff(iff+i-s+1) = d*(vxx(ixx+i) -                                 &
     &                 2.d0*vxx(ixx+i+1) +                              &
     &                 vxx(ixx+i+2))                                    &
     &                 + vxx(ixx+i+1)*vxx(ixx+i+1) -                    &
     &                 vF2(iF2+i-s+1)
 10   continue

      if (rank .eq. size-1) then
        vff(iff+n-s) = vxx(ixx+n) - 1.d0
      endif

      call VecRestoreArray(f,vff,iff,ierr)
      call VecRestoreArray(ctx(3),vxx,ixx,ierr)
      call VecRestoreArray(ctx(2),vF2,iF2,ierr)
      return
      end

! --------------------  Form initial approximation ----------------- 

      subroutine FormInitialGuess(snes,x,ierr)
      include '/home/bsmith/petsc/include/foldinclude/petsc.h'
      include '/home/bsmith/petsc/include/foldinclude/vec.h'
      include '/home/bsmith/petsc/include/foldinclude/snes.h'
      integer          ierr
      integer          x
      integer          snes
      double precision five 

      five = 5.d-1
      call VecSet(five,x,ierr)
      return
      end

! --------------------  Evaluate Jacobian -------------------- 

      subroutine FormJacobian(snes,x,jac,B,flag,ctx,ierr)
      include '/home/bsmith/petsc/include/foldinclude/petsc.h'
      include '/home/bsmith/petsc/include/foldinclude/vec.h'
      include '/home/bsmith/petsc/include/foldinclude/da.h'
      include '/home/bsmith/petsc/include/foldinclude/mat.h'
      include '/home/bsmith/petsc/include/foldinclude/snes.h'
      integer          snes, x, jac(*), B, flag, ctx(*), ii, istart
!
!   In the .F version of this code it would be called PetscOffset
!
      integer          ixx
      integer          iend, i,j, n, rank, size, end, start, ierr
      double precision d, A, h, vxx(1)

      h = 1.d0/(ctx(6) - 1.d0)
      d = h*h
      rank = ctx(4)
      size = ctx(5)

      call VecGetArray(x,vxx,ixx,ierr)
      call VecGetOwnershipRange(x,start,end,ierr)
      n = end - start

      if (rank .eq. 0) then
        A = 1.0 
        call MatSetValues(jac,1,start,1,start,A,INSERT_VALUES,ierr)
        istart = 1
      else 
        istart = 0
      endif
      if (rank .eq. size-1) then
        i = ctx(6)-1
        A = 1.0 
        call MatSetValues(jac,1,i,1,i,A,INSERT_VALUES,ierr)
        iend = n-1
      else
        iend = n
      endif
      do 10 i=istart,iend-1
        ii = i + start
        j = start + i - 1 
        call MatSetValues(jac,1,ii,1,j,d,INSERT_VALUES,ierr)
        j = start + i + 1 
        call MatSetValues(jac,1,ii,1,j,d,INSERT_VALUES,ierr)
        A = -2.0*d + 2.0*vxx(ixx+i+1)
        call MatSetValues(jac,1,ii,1,ii,A,INSERT_VALUES,ierr)
 10   continue
      call VecRestoreArray(x,vxx,ixx,ierr)
      call MatAssemblyBegin(jac,MAT_FINAL_ASSEMBLY,ierr)
      call MatAssemblyEnd(jac,MAT_FINAL_ASSEMBLY,ierr)
!      call MatView(jac,VIEWER_STDOUT_WORLD)
      return
      end







