#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: grid.c,v 1.23 1998/04/27 14:20:56 curfman Exp $";
#endif

/*
  This file provides routines for grid vectors (vectors that are associated 
  with grids, possibly with multiple degrees of freedom per node). 

  This component is new; changes in the user interface will be occuring in the
  near future!
*/

#include "src/gvec/gvecimpl.h"    /*I "gvec.h" I*/

#undef __FUNC__  
#define __FUNC__ "GridGetGridData" 
/*@
   GridGetGridData - Extracts the grid data from a grid.

   Not Collective

   Input Parameters:
.  grid - the grid context

   Output Parameters:
.  griddata - the grid data 

.keywords: discrete function, vector, get

.seealso: GVecGetGrid()
@*/
int GridGetGridData(Grid grid,GridData *griddata)
{
  PetscFunctionBegin;
  PetscValidHeaderSpecific(grid,GRID_COOKIE);
  PetscValidPointer(griddata);
  *griddata = grid->griddata;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDestroy" 
/*@
    GridDestroy - Destroys a grid. 

    Collective on Grid

    Input Parameters:
.   grid - the grid context to distroy

.seealso: GridView(), GridCreateGVec(), GridSetUp(),GridRefine()
@*/
int GridDestroy(Grid grid)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(grid,GRID_COOKIE);

  if (--grid->refct > 0) PetscFunctionReturn(0);

  if (!grid->setupcalled) {
    ierr = GridSetUp(grid); CHKERRQ(ierr);
  }

  ierr = (*grid->ops->destroy)(grid); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridView" 
/*@
    GridView - Prints data on a grid context.

    Collective on Grid unless viewer is sequential

    Input Parameters:
+   grid - the grid context to distroy
-   viewer - the viewer, generally VIEWER_STDOUT_WORLD

.seealso: GVecView()
@*/
int GridView(Grid grid,Viewer viewer)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(grid,GRID_COOKIE);
  if (!grid->setupcalled) {
    ierr = GridSetUp(grid); CHKERRQ(ierr);
  }
  if (!viewer) {viewer = VIEWER_STDOUT_SELF; }
  else PetscValidHeader(viewer);
  ierr = (*grid->ops->view)(grid,viewer);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateGVec"
/*@
    GridCreateGVec - Creates a grid vector associated with a
    particular discretization context.

    Not Collective (but parallel GVec is returned if Grid is parallel)

    Input Parameter:
.   grid - the discretization context defining the discrete function

    Output Parameter:
.   gvec - the resulting grid vector

@*/
int GridCreateGVec(Grid grid, GVec *gvec)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(grid,GRID_COOKIE);
  if (!grid->setupcalled) {
    ierr = GridSetUp(grid); CHKERRQ(ierr);
  }
  ierr = (*grid->ops->gridcreategvec)(grid,gvec);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


#undef __FUNC__  
#define __FUNC__ "GridSetUp"
/*@
   GridSetUp - Set up any required internal data structures for a discretization
   context. Should be called before GridCreateGVec().

   Collective on Grid

   Input Parameter:
.  g1 - the input grid

.keywords: grid, setup

.seealso: GridDestroy(), GridCreateGVec()
@*/
int GridSetUp(Grid x)
{
  int ierr;

  PetscFunctionBegin;
  if (x->setupcalled) PetscFunctionReturn(0);
  if (x->ops->gridsetup) {ierr = (*x->ops->gridsetup)(x); CHKERRQ(ierr);}
  x->setupcalled = 1;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridRefine"
/*@
   GridRefine - Refine a grid uniformly by bisection.

   Collective on Grid
  
   Input Parameter:
.  grid - the grid

   Output Parameter:
.  newgrid - the refined grid

.keywords: grid, refinement

.seealso: GridDestroy()
@*/
int GridRefine(Grid grid,int l,Grid *newgrid)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(grid,GRID_COOKIE);
  PetscValidPointer(newgrid);
  if (!grid->setupcalled) {
    ierr = GridSetUp(grid); CHKERRQ(ierr);
  }
  if (!grid->ops->gridrefine) SETERRQ(PETSC_ERR_SUP,0,"");

  ierr = (*grid->ops->gridrefine)(grid,l,newgrid); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}







