
/* $Id: common5and6.c,v 1.5 1997/04/10 00:07:55 bsmith Exp $ */


/*
     Demonstrates how one may construct a simple finite difference operator for
   two dimensional Laplacian using PETSc GVecs.
*/

int UserComputeMatrix(GMat J)
{
  int                        ierr, xs,nx,ys,ny,i,gxs,xs1,j;           
  int                        m,n,indices[5],nx1,gnx;
  int                        ny1,gny,gys,ys1,row;
  Scalar                     values[5];
  double                     ht,hb,hr,hl,hx,hy;
  Grid                       grid;
  GridData                   griddata;
  DA                         da;
  double                     *x,*y;

  /*
     Access the data layout and grid coordinate information
  */
  ierr = GMatGetGrid(J,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(griddata,&x,&y,0); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&m,&n,0,0,0,0,0,0,0); CHKERRQ(ierr);
  ierr = DAGetCorners(da,&xs,&ys,0,&nx,&ny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&gxs,&gys,0,&gnx,&gny,0); CHKERRQ(ierr);

  /*
      For each node in the grid: we get the neighbors in the local (on processor ordering
    that includes the ghost points) then insert into the matrix with the local 
    numbering.
  */
  
  nx1 = nx; ny1 = ny;
  if (xs == 0) {xs1 = 1; nx1--;} else xs1 = xs;
  if (ys == 0) {ys1 = 1; ny1--;} else ys1 = ys;
  if (xs + nx == m) nx1--;
  if (ys + ny == n) ny1--;


  if (xs == 0 && ys == 0) {
    /*  Special case, lower left corner of grid */
    row        = 0;
    indices[0] = row; 
    indices[1] = row+1; 
    indices[2] = row+gnx; 
    hr       =  x[1] - x[0];
    ht       =  y[1] - y[0];
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (xs + nx == m && ys == 0) {
    /* Special case, lower right corner of grid */
    row        = gnx-1;
    indices[0] = row-1;
    indices[1] = row;
    indices[2] = row+gnx; 
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (xs == 0 && ys + ny == n) {
    /* Special case, upper left corner of grid */
    row        = gny*gnx-gnx;
    indices[0] = row-gnx;
    indices[1] = row;
    indices[2] = row+1;
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (xs + nx == m && ys + ny == n) {
    /* Special case, upper right corner of grid */
    row        = gny*gnx-1;
    indices[0] = row-gnx;
    indices[1] = row-1;
    indices[2] = row;
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (ys == 0) {
    /* special case boundary along bottom of grid */
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row        = i-gxs;
      indices[0] = row-1;
      indices[1] = row;
      indices[2] = row+1; 
      indices[3] = row+gnx; 
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  if (ys + ny == n) {
    /* special case boundary along top of grid */
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row        = i-gxs+gny*gnx-gnx;
      indices[0] = row-gnx;
      indices[1] = row-1;
      indices[2] = row;
      indices[3] = row+1;
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  if (xs == 0) {
    /* special case boundary along left edge */
    for ( i=ys1; i<ys1+ny1; i++ ) {
      row        = (i-gys)*gnx;
      indices[0] = row-gnx;
      indices[1] = row;
      indices[2] = row+1;
      indices[3] = row+gnx;
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  if (xs + nx == m) {
    /* special case boundary along right edge */
    for ( i=ys1; i<ys1+ny1; i++ ) {
      row        = (i-gys+1)*gnx-1;
      indices[0] = row-gnx;
      indices[1] = row-1;
      indices[2] = row;
      indices[3] = row+gnx;
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  /* do the interior portion of the grid */
  for ( j=ys1; j<ys1+ny1; j++ ) {
    ht = y[j+1] - y[j];
    hb = y[j] - y[j-1];
    hy = hb + ht;
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row = (j-gys)*gnx+i-gxs;
      indices[0] = row-gnx;
      indices[1] = row-1;
      indices[2] = row;
      indices[3] = row+1;
      indices[4] = row+gnx;
      hr = x[i+1] - x[i];
      hl = x[i] - x[i-1];
      hx = hl + hr;
      values[0] = .5*hx/hb;
      values[1] = .5*hy/hl;
      values[2] = -.5*hx*hy/(hr*hl) - .5*hx*hy/(ht*hb);
      values[3] = .5*hy/hr;
      values[4] = .5*hx/ht;
      ierr = MatSetValuesLocal(J,1,&row,5,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  
  ierr = MatAssemblyBegin(J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);  
  ierr = MatAssemblyEnd(J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);  

  return 0;
}

/* ---------------------------------------------------------------------------------*/
/*
     Applies Dirichlet boundary conditions to boundary points of the domain 

*/
int UserComputeDirichlet(GVec xin,GVec bin,GMat J,
                         int (*f)(int,double*,double*,double*,Scalar*,void*),void* fctx)
{
  int                        ierr, xs,nx,ys,ny,i,gxs,xs1;           
  int                        m,n,nx1,gnx;
  int                        ny1,gny,gys,ys1,row,count = 0,*rows;
  Scalar                     value,*fvalues;
  double                     *xc,*yc;
  Grid                       grid;
  GridData                   griddata;
  DA                         da;
  double                     *x,*y;
  IS                         is;

  /*
     Access the data layout and grid coordinate information
  */
  ierr = GMatGetGrid(J,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(griddata,&x,&y,0); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&m,&n,0,0,0,0,0,0,0); CHKERRQ(ierr);
  ierr = DAGetCorners(da,&xs,&ys,0,&nx,&ny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&gxs,&gys,0,&gnx,&gny,0); CHKERRQ(ierr);

  /*
      Generate a list of all the nodes on this processor that are
    on the boundary. We will zero those rows of the matrix and replace 
    with scaled rows of the identity.
  */
  rows    = (int *) PetscMalloc( 2*(nx+1)*(ny+1)*sizeof(int) ); CHKPTRQ(rows);
  xc      = (double *) PetscMalloc(2*2*(nx+1)*(ny+1)*sizeof(double));CHKPTRQ(xc);
  yc      = xc + 2*(nx+1)*(ny+1);

  nx1 = nx; ny1 = ny;
  if (xs == 0) {xs1 = 1; nx1--;} else xs1 = xs;
  if (ys == 0) {ys1 = 1; ny1--;} else ys1 = ys;
  if (xs + nx == m) nx1--;
  if (ys + ny == n) ny1--;


  if (xs == 0 && ys == 0) {
    /*  Special case, lower left corner of grid */
    row         = 0;
    rows[count] = row;
    xc[count]   = x[0];
    yc[count]   = y[0];
    count++;
  }
  if (xs + nx == m && ys == 0) {
    /* Special case, lower right corner of grid */
    row         = gnx-1;
    rows[count] = row;
    xc[count]   = x[m-1];
    yc[count]   = y[0];
    count++;
  }
  if (xs == 0 && ys + ny == n) {
    /* Special case, upper left corner of grid */
    row         = gny*gnx-gnx;
    rows[count] = row;
    xc[count]   = x[0];
    yc[count]   = y[n-1];
    count++;
  }
  if (xs + nx == m && ys + ny == n) {
    /* Special case, upper right corner of grid */
    row         = gny*gnx-1;
    rows[count] = row;
    xc[count]   = x[m-1];
    yc[count]   = y[n-1];
    count++;
  }
  if (ys == 0) {
    /* special case boundary along bottom of grid */
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row         = i-gxs;
      rows[count] = row;
      xc[count]   = x[i];
      yc[count]   = y[0];
      count++;
    }
  }
  if (ys + ny == n) {
    /* special case boundary along top of grid */
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row         = i-gxs+gny*gnx-gnx;
      rows[count] = row;
      xc[count]   = x[i];
      yc[count]   = y[n-1];
      count++;
    }
  }
  if (xs == 0) {
    /* special case boundary along left edge */
    for ( i=ys1; i<ys1+ny1; i++ ) {
      row         = (i-gys)*gnx;
      rows[count] = row;
      xc[count]   = x[0];
      yc[count]   = y[i];
      count++;
    }
  }
  if (xs + nx == m) {
    /* special case boundary along right edge */
    for ( i=ys1; i<ys1+ny1; i++ ) {
      row         = (i-gys+1)*gnx-1;
      rows[count] = row;
      xc[count]   = x[m-1];
      yc[count]   = y[i];
      count++;
    }
  }
  
  /* Change the rows of the matrix to 0 ... 0 1 0 ..... */
  value = 1.0;
  ierr = ISCreateGeneral(PETSC_COMM_SELF,count,rows,&is);CHKERRQ(ierr);
  ierr = MatZeroRowsLocal(J,is,&value);CHKERRQ(ierr);
  ierr = ISDestroy(is); CHKERRQ(ierr);

  /* Evalute the Dirichlet boundary conditions */
  fvalues = (Scalar *) PetscMalloc(count*sizeof(Scalar));CHKPTRQ(fvalues);
  ierr    = (*f)(count,xc,yc,0,fvalues,fctx); CHKERRQ(ierr);
  PetscFree(xc);
  ierr    = VecSetValuesLocal(xin,count,rows,fvalues,INSERT_VALUES);CHKERRQ(ierr);
  ierr    = VecSetValuesLocal(bin,count,rows,fvalues,INSERT_VALUES);CHKERRQ(ierr);
  PetscFree(fvalues);
  PetscFree(rows);
  ierr    = VecAssemblyBegin(xin);CHKERRQ(ierr);
  ierr    = VecAssemblyEnd(xin);CHKERRQ(ierr);
  ierr    = VecAssemblyBegin(bin);CHKERRQ(ierr);
  ierr    = VecAssemblyEnd(bin);CHKERRQ(ierr);

  return 0;
}







