#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: vrmlmesh.c,v 1.13 1998/04/27 20:39:24 curfman Exp $";
#endif

/* 
 * Special code for handling meshes.  
 */ 

#include "src/draw/impls/vrml/vrmlimpl.h"    /*I   "draw.h"  I*/

/*
 * ViInitMesh - Initialize a mesh data structure
 *
 * z is optional for tensor meshes (x(i), y(j), f(i+nx*j))
 */

#undef __FUNC__  
#define __FUNC__ "ViInitMesh"
int ViInitMesh( DrawMesh meshinfo, 
		double *x, double *y, double *z,
		int nx, int ny, int nz, 
		int si, int ei, int sj, int ej, int sk, int ek,
		int i_incr, int j_incr, int k_incr, 
		int is_mapped, 
		double *f,
		int ncolors,
		ColorMapType ctype )
{
    int i, j, k, idx;
    double xmin, xmax, ymin, ymax, zmin, zmax, fmin, fmax, ftmp;

    /* Store basic information */
    meshinfo->x	     = x;
    meshinfo->y	     = y;
    meshinfo->z	     = z;
    meshinfo->nx     = nx;
    meshinfo->ny     = ny;
    meshinfo->nz     = nz;
    meshinfo->f	     = f;
    meshinfo->si     = si;
    meshinfo->ei     = ei;
    meshinfo->sj     = sj;
    meshinfo->ej     = ej;
    meshinfo->sk     = sk;
    meshinfo->ek     = ek;
    meshinfo->i_incr = i_incr;
    meshinfo->j_incr = j_incr;
    meshinfo->k_incr = k_incr;

    /* Compute info: Coordinate mappings */
    if (!is_mapped) {
	if (z) {
	    zmin = zmax = z[sk];
	    for (k=sk + k_incr; k<=ek; k += k_incr) {
		if (zmin > z[k]) zmin = z[k];
		else if (zmax < z[k]) zmax = z[k];
	    }
	}
	ymin = ymax = y[sj];
	for (j=sj + j_incr; j<=ej; j += j_incr) {
	    if (ymin > y[j]) ymin = y[j];
	    else if (ymax < y[j]) ymax = y[j];
	}
	xmin = xmax = x[si];
	for (i=si + i_incr; i<=ei; i += i_incr) {
	    if (xmin > x[i]) xmin = x[i];
	    else if (xmax < x[i]) xmax = x[i];
	}
    }

    /* Get function limits.  If mapped, get coord limits. */
    idx = si+nx*(sj + ny*sk);
    if (f) fmin = fmax = f[idx];
    else   fmin = fmax = 0.0;
    if (is_mapped) {
	xmin = xmax = x[idx];
	ymin = ymax = y[idx];
	if (z) {
	    zmin = zmax = z[idx];
	}
	else 
	    zmin = zmax = 0.0;
    }
    else {
	xmin = x[si];
	ymin = y[sj];
	xmax = x[ei];
	ymax = y[ej];
	if (z) {
	    zmin = z[sk];
	    zmax = z[ek];
	}
	else
	    zmin = zmax = 0.0;
	
    }

    for (k=sk; k<=ek; k += k_incr) {
	for (j=sj; j<=ej; j += j_incr) {
	    for (i=si; i<=ei; i += i_incr) {
		idx = i+nx*(j + ny*k);
		if (f) {
                    ftmp = f[idx]; 
		    if (fmin > ftmp) fmin = ftmp;
		    else if (fmax < ftmp) fmax = ftmp;
		}
		if (is_mapped) {
		    ftmp = x[idx];
		    if (ftmp < xmin) xmin = ftmp;
		    else if (ftmp > xmax) xmax = ftmp;
		    ftmp = y[idx];
		    if (ftmp < ymin) ymin = ftmp;
		    else if (ftmp > ymax) ymax = ftmp;
		    if (z) {
			ftmp = z[idx];
			if (ftmp < zmin) zmin = ftmp;
			else if (ftmp > zmax) zmax = ftmp;
		    }
		}
	    }
	}
    }
    /* Save information on the mesh limits */
    meshinfo->zmin = zmin;
    meshinfo->zmax = zmax;
    meshinfo->xmin = xmin;
    meshinfo->xmax = xmax;
    meshinfo->ymin = ymin;
    meshinfo->ymax = ymax;
    meshinfo->fmin = fmin;
    meshinfo->fmax = fmax;

    /* Set color mapping routines */
    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawMeshCreate"
int DrawMeshCreate( DrawMesh *mesh, 
		    double *x, double *y, double *z,
		    int nx, int ny, int nz, 
		    int si, int ei, int sj, int ej, int sk, int ek,
		    int i_incr, int j_incr, int k_incr, 
		    int is_mapped, 
		    double *f,
		    int ncolors )
{
    DrawMesh meshinfo;
    meshinfo = (DrawMesh) PetscMalloc( sizeof(struct _p_DrawMesh) );
    *mesh = meshinfo;

    return ViInitMesh( meshinfo, x, y, z, nx, ny, nz, 
		       si, ei, sj, ej, sk, ek, 
		       i_incr, j_incr, k_incr, 
		       is_mapped, 
		       f, 
		       ncolors, COLOR_MAP_LINEAR );
}

#undef __FUNC__  
#define __FUNC__ "DrawMeshCreateSimple"
/*@
   DrawMeshCreateSimple - Create a representation of a mesh for drawing

   Input Parameters:
+  x,y,z - X, Y, and Z coordinates of the mesh.  If the mesh is 2-d, use
   '(double *)0' for 'z'
.  nx,ny,nz - Size of the X, Y, and Z arrays.  If the mesh is 2-d, use
   zero for 'nz'
.  is_mapped - If false, the coordinates are '(x[i],y[j])' in 2-d and 
   '(x[i],y[j],z[k])' in 3-d.  If true, the coordinates are 
   '(x[i,j],y[i,j])' in 2-d and '(x[i,j,k],y[i,j,k],z[i,j,k])' in 3-d.
   Here, 'x[i,j]' is shorthand for 'x[i+j*nx]'.
.  f - Array containing values on mesh.  
-  ncolors - Number of colors that may be used in plotting data (the actual 
   number is usually set by the plotting routine).

   Output Parameters:
.  mesh - Structure containing information on the mesh

   Notes:
   To plot a mesh '(x[i],y[j],f[i,j])', code roughly like the following
   can be used
.vb
  DrawMesh meshinfo;
  Draw     draw;
  double x[X_SIZE], y[Y_SIZE], f[X_SIZE*Y_SIZE];

  ierr = DrawOpenVRML(PETSC_COMM_WORLD,0,"Sample Plot",&draw); CHKERRA(ierr);
  ... setup x, y, and f
  DrawMeshCreateSimple( &meshinfo, x, y, (double *)0, X_SIZE,
			      Y_SIZE, 0, 1, f, 64 );
  DrawBOP( draw );
  DrawTensorSurface( draw, meshinfo, DRAW_BLUE );  
.ve

.seealso: DrawMeshCreate, DrawMeshDestroy, DrawTensorSurface, 
          DrawTensorSurfaceContour, DrawTensorMapSurfaceContour, 
	  DrawTensorMapMesh
   
@*/
int DrawMeshCreateSimple( DrawMesh *mesh, 
			  double *x, double *y, double *z,
			  int nx, int ny, int nz, 
			  int is_mapped, 
			  double *f,
			  int ncolors )
{
    return DrawMeshCreate( mesh, x, y, z, nx, ny, nz, 
			   0, nx-1, 0, ny-1, 0, (nz > 0) ? nz-1 : 0,
			   1, 1, 1, is_mapped, 
			   f, 
			   ncolors );
}

#undef __FUNC__  
#define __FUNC__ "DrawMeshDestroy"
int DrawMeshDestroy( DrawMesh *mesh )
{
    PetscFree( *mesh );
    *mesh = 0;
    PetscFunctionReturn(0);
}

/*
 * Routines to draw on meshes
 */
#ifdef FOO
#undef __FUNC__  
#define __FUNC__ "DrawTensorSurface"
int DrawTensorSurface( Draw Win, DrawMesh mesh, int color )
{
    Draw_VRML* VWin = (Draw_VRML*) Win->data;

    return (*VWin->DrawTensorSurface)( Win, mesh, color );
}

#undef __FUNC__  
#define __FUNC__ "DrawTensorSurfaceContour"
int DrawTensorSurfaceContour( Draw Win, DrawMesh mesh, 
			      VRMLGetHue_fcn GetColor, 
			      void *color_context, int ncolor )
{
    Draw_VRML* VWin = (Draw_VRML*) Win->data;

    return (*VWin->DrawTensorSurfaceContour)( Win, mesh, 
					      GetColor, color_context, 
					      ncolor );
}

#undef __FUNC__  
#define __FUNC__ "DrawTensorMapSurfaceContour"
int DrawTensorMapSurfaceContour( Draw Win, DrawMesh mesh, 
				 double ox, double oy, double oz, 
				 int coord_slice, int coord_dim,
				 VRMLGetHue_fcn GetColor, 
				 void *color_context, int ncolor,
				 double transparency )
{
    Draw_VRML* VWin = (Draw_VRML*) Win->data;

    return (*VWin->DrawTensorMapSurfaceContour)( Win, mesh, ox, oy, oz, 
						 coord_slice, coord_dim,
						 GetColor, color_context, 
						 ncolor, transparency );
}

#undef __FUNC__  
#define __FUNC__ "DrawTensorMapMesh"
int DrawTensorMapMesh( Draw Win, DrawMesh mesh, 
		       double ox, double oy, double oz, 
		       int coord_slice, int coord_dim )
{
    Draw_VRML* VWin = (Draw_VRML*) Win->data;

    return (*VWin->DrawTensorMapMesh)( Win, mesh, ox, oy, oz, 
				       coord_slice, coord_dim );
}
#endif
