#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: vrmlcolor.c,v 1.12 1997/10/19 03:27:58 bsmith Exp $";
#endif

/*
 * This algorithm is from Foley and van Dam, page 616
 * given
 *   (0:359, 0:100, 0:100).
 *      h       l      s
 * set
 *   (0:255, 0:255, 0:255)
 *      r       g      b
 */
#include "src/draw/impls/vrml/vrmlimpl.h"

#undef __FUNC__  
#define __FUNC__ "ViHlsHelper"
static int ViHlsHelper( int h, int n1, int n2 )
{
    while (h > 360) h = h - 360;
    while (h < 0)   h = h + 360;
    if (h < 60)     PetscFunctionReturn(n1 + (n2-n1)*h/60);
    if (h < 180)    PetscFunctionReturn(n2);
    if (h < 240)    PetscFunctionReturn(n1 + (n2-n1)*(240-h)/60);
    PetscFunctionReturn(n1);
}

#undef __FUNC__  
#define __FUNC__ "ViHlsToRgb"
int ViHlsToRgb( int h, int l, int s, int *r, int *g, int *b )
{
    int m1, m2;         /* in 0 to 100 */
    if (l <= 50) m2 = l * ( 100 + s ) / 100 ;           /* not sure of "/100" */
    else         m2 = l + s - l*s/100;

    m1  = 2*l - m2;
    if (s == 0) {
	/* ignore h */
	*r  = 255 * l / 100;
	*g  = 255 * l / 100;
	*b  = 255 * l / 100;
    }
    else {
	*r  = (255 * ViHlsHelper( h+120, m1, m2 ) ) / 100;
	*g  = (255 * ViHlsHelper( h, m1, m2 ) )     / 100;
	*b  = (255 * ViHlsHelper( h-120, m1, m2 ) ) / 100;
    }
    PetscFunctionReturn(0);
}
extern int ViHlsToRgb(int,int,int,int*,int*,int*);

/*
 * This is a simple routine that gets uniformly spaced colors between
 * zmin and zmax.
 */
typedef struct { double zmin, zmax; } VRML_GetHue;

#undef __FUNC__  
#define __FUNC__ "VRMLGetHue_setup"
void *VRMLGetHue_setup( DrawMesh mesh, int ncolor )
{
    VRML_GetHue *hue_limits;
    double zmin, zmax;
    int    i, j, k, si, ei, sj, ej, sk, ek, nx, ny, idx;
    double *f  = mesh->f;

    hue_limits = (VRML_GetHue *)PetscMalloc( sizeof(VRML_GetHue) );
    
    si = mesh->si;
    ei = mesh->ei;
    sj = mesh->sj;
    ej = mesh->ej;
    sk = mesh->sk;
    ek = mesh->ek;
    nx = mesh->nx;
    ny = mesh->ny;
    zmin = zmax = f[si + nx * (sj + ny * sk)];
    
    for (k=sk; k<=ek; k += 1) {
	for (j=sj; j<=ej; j += 1) {
	    for (i=si; i<=ei; i += 1) {
		idx = i+nx*(j + ny*k);
		if (f[idx] > zmax)      zmax = f[idx];
		else if (f[idx] < zmin) zmin = f[idx];
	    }
	}
    }
    if (zmin >= zmax) {
	/* No range; make an artificial one */
	zmax = zmax * 1.1 + 1.0;
    }
    hue_limits->zmin = zmin;
    hue_limits->zmax = zmax;

    PetscFunctionReturn((void *)hue_limits);
}

#undef __FUNC__  
#define __FUNC__ "VRMLGetHue_destroy"
void VRMLGetHue_destroy( void *hue_limits )
{
    PetscFree( hue_limits );
}

#undef __FUNC__  
#define __FUNC__ "VRMLGetHue"
void VRMLGetHue( double zval, void * color_context, 
		     int ncolor, double *red, double *green, double *blue )
{
    int r, g, b;
    VRML_GetHue *hue_limits = (VRML_GetHue *)color_context;
    double zmin = hue_limits->zmin;
    double zmax = hue_limits->zmax;

    ViHlsToRgb( (int)(360 * (zval - zmin) / (zmax - zmin)), 
		50, 100, &r, &g, &b );
    *red   = r / 255.0;
    *green = g / 255.0;
    *blue  = b / 255.0;
}

/*
 * This routine uses a predefined table to identify colors; this 
 * can be used to change the color mapping
 */
typedef struct { double v, red, green, blue; } VRML_FindHue;
/*
  Routine commented out because not currently used. Only used
  if qsort() is used below.
static int compare_double( double *z1, double *z2 )
{
    if (*z1 < *z2) PetscFunctionReturn(-1);
    if (*z1 > *z2) PetscFunctionReturn(1);
    PetscFunctionReturn(0);
}
*/
#undef __FUNC__  
#define __FUNC__ "VRMLFindHue_setup"
void *VRMLFindHue_setup( DrawMesh mesh, int ncolor )
{
    VRML_FindHue *hue_table;
    double   *ztmp;
    int      nmodncolor;
    int      i, j, k, si, ei, sj, ej, sk, ek, nx, ny, idx, n, cnt;
    double *f  = mesh->f;
    int      r, g, b;

    hue_table = (VRML_FindHue *) PetscMalloc( sizeof(VRML_FindHue) * ncolor );

    si = mesh->si;
    ei = mesh->ei;
    sj = mesh->sj;
    ej = mesh->ej;
    sk = mesh->sk;
    ek = mesh->ek;
    nx = mesh->nx;
    ny = mesh->ny;
    
    /* First, bin the values into ncolor values.  The easiest way is to
       copy zval, sort, and exactly divide */
    n = (ei - si + 1) * (ej - sj + 1) * (ek - sk + 1);
    ztmp = (double *) PetscMalloc( sizeof(double) * n );
    cnt = 0;
    for (k=sk; k<=ek; k += 1) {
	for (j=sj; j<=ej; j += 1) {
	    for (i=si; i<=ei; i += 1) {
		idx = i+nx*(j + ny*k);
		ztmp[cnt++] = f[idx];
	    }
	}
    }

    PetscSortDouble(n,ztmp);
    /*qsort( ztmp, n, sizeof(double), compare_double ); */
    j = 0;
    nmodncolor = n % ncolor;
    /* To get a better separation of colors, don't run entirly around the
       HLS color wheel.  Instead, finish at about 300 (instead of 360) */
    for (i=0; i<ncolor; i++) {
	hue_table[i].v	   = ztmp[j];
	ViHlsToRgb( (300 * i) / ncolor, 50, 100, &r, &g, &b );
	hue_table[i].red   = r / 255.0;
	hue_table[i].green = g / 255.0;
	hue_table[i].blue  = b / 255.0;
	j += (n/ncolor);
	if (i < nmodncolor) j++;
    }
    PetscFree( ztmp );
    PetscFunctionReturn((void *)hue_table);
}

#undef __FUNC__  
#define __FUNC__ "VRMLFindHue_destroy"
void VRMLFindHue_destroy( void *hue_table )
{
    PetscFree( hue_table );
}

#undef __FUNC__  
#define __FUNC__ "VRMLFindHue"
void VRMLFindHue( double zval, void *color_context,
		     int ncolor, double *red, double *green, double *blue )
{
    int low, high, j;
    VRML_FindHue *hue_table = (VRML_FindHue *)color_context;

    /* Find v in table by bisection */
    low  = 0; 
    high = ncolor - 1;
    /* 
       We can switch to linear search by changing (low + 1 < high) to
       (low + 6 < high), and adding a linear search at the end.
     */
    while (low + 1 < high) {
	j = ( low + high ) / 2;
	if (hue_table[j].v > zval) {
	    high = j;
	}
	else if (hue_table[j].v < zval) {
	    low = j;
	}
	else {
	    /* exact match */
	    *red   = hue_table[j].red;
	    *green = hue_table[j].green;
	    *blue  = hue_table[j].blue;
	    return;
	}
    }
    j = ( low + high ) / 2;
    *red   = hue_table[j].red;
    *green = hue_table[j].green;
    *blue  = hue_table[j].blue;
}

