#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex1.c,v 1.30 1997/11/28 16:21:17 bsmith Exp $";
#endif

static char help[] = "Tests some utilities for vectors with the extra property\n\
of being discrete functions (case of regular grid using a single processor).\n\n";

#include "dfvec.h"

int ViewCoordinates(MPI_Comm,double*,double*,double*,int,int,int);

int main(int argc,char **args)
{
  DFVec  v, rv;          /* discrete function vectors (initial and refined) */
  DF     df, rdf;        /* discrete function shell (initial and refined) */
  Vec    *vsub;          /* standard vectors */
  Viewer viewer;
  Scalar q[2], *va;
  double *x, *y, *z, *xx, *yy, *zz;
  int    i, j, k, nc = 2, mx = 3, my = 4, mz = 1, neq, I[2], contours = 0;
  int    flg, width = 300, height = 300, ierr, dim = 2, info_view = 0, nsub;
  int    surface = 0;

  PetscInitialize(&argc,&args,0,help);
  ierr = OptionsGetInt(PETSC_NULL,"-mx",&mx,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-my",&my,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-mz",&mz,&flg); CHKERRA(ierr);
  if (mx<1 || my<1 || mz<1) SETERRA(1,0,"Only mx, my, mz > 0 supported");
  ierr = OptionsHasName(PETSC_NULL,"-info_view",&info_view); CHKERRA(ierr);
  ierr = OptionsHasName(PETSC_NULL,"-contours",&contours); CHKERRA(ierr);
  ierr = OptionsHasName(PETSC_NULL,"-surface",&surface); CHKERRA(ierr);
  x = (double *) PetscMalloc( mx*sizeof(double) ); CHKPTRA(x);
  y = (double *) PetscMalloc( my*sizeof(double) ); CHKPTRA(y);
  z = (double *) PetscMalloc( mz*sizeof(double) ); CHKPTRA(z);
  neq = mx*my*mz;

  /* Create multicomponent vector and set grid coordinates */
  ierr = VecCreate(PETSC_COMM_SELF,PETSC_DECIDE,nc*neq,&v); CHKERRA(ierr);
  for ( k=0; k<mz; k++ ) {
    z[k] = k*0.5;
    for ( j=0; j<my; j++ ) {
      for ( i=0; i<mx; i++ ) {
        I[0] = nc*(i + j*mx + k*mx*my); I[1] = I[0]+1;
        q[0] = k*100 + j*10 + i;        q[1] = k*10000 + j*1000 + i;
        ierr = VecSetValues(v,2,I,q,INSERT_VALUES); CHKERRA(ierr);
      }
    }
  }
  for ( j=0; j<my; j++ ) y[j] = j*0.4;
  for ( i=0; i<mx; i++ ) x[i] = i*0.3;
  ierr = VecAssemblyBegin(v); CHKERRA(ierr);
  ierr = VecAssemblyEnd(v); CHKERRA(ierr);

  /* Create vectors for separate components */
  if (my == 1 && mz == 1) dim = 1;
  if (mz != 1) dim = 3;
  ierr = DFShellCreate(PETSC_COMM_SELF,DF_SEQREG,dim,nc,ORDER_1,
         PETSC_NULL,mx,my,mz,&df); CHKERRA(ierr);
  ierr = DFVecShellAssociate(df,v); CHKERRA(ierr);
  if (info_view) {
    ierr = ViewerPushFormat(VIEWER_STDOUT_SELF,VIEWER_FORMAT_ASCII_INFO,
           PETSC_NULL); CHKERRA(ierr);
    ierr = DFVecView(v,VIEWER_STDOUT_SELF); CHKERRA(ierr);
    ierr = ViewerPopFormat(VIEWER_STDOUT_SELF); CHKERRA(ierr);
  }
  ierr = DFVecGetComponentVectors(v,&nsub,&vsub); CHKERRA(ierr);

  /* Apply some user-defined function to vector vsub[0] */
  ierr = VecGetArray(vsub[0],&va); CHKERRA(ierr);
  for (i=0; i<neq; i++) {
     va[i] += 80000;
  }
  ierr = VecRestoreArray(vsub[0],&va); CHKERRA(ierr);
  ierr = DFVecAssembleFullVector(vsub,v); CHKERRA(ierr);

  /* View vectors */
  PetscPrintf(PETSC_COMM_SELF,"multicomponent vector\n");
  ierr = DFVecView(v,VIEWER_STDOUT_SELF); CHKERRA(ierr);
  /* for (i=0; i<nc; i++) {
    PetscPrintf(PETSC_COMM_SELF,"component %d\n",i);
    ierr = VecView(vsub[i],VIEWER_STDOUT_SELF); CHKERRA(ierr);
  } */
  if (surface) {
    ierr = ViewerDrawOpenVRML(PETSC_COMM_WORLD,0,"Title",&viewer); CHKERRA(ierr);
    ierr = DFVecView(v,viewer); CHKERRA(ierr);
  }

  /* Draw contour plot for each component */
  ierr = DFShellSetCoordinates(df,mx,my,mz,x,y,z); CHKERRA(ierr);
  ierr = ViewCoordinates(PETSC_COMM_SELF,x,y,z,mx,my,mz); CHKERRA(ierr);
  if (contours) {ierr = DFVecDrawTensorContoursX(v,width,height); CHKERRA(ierr);}

  /* Refine vector */
  ierr = DFVecRefineVector(v,&rv); CHKERRA(ierr);
  if (info_view) {
    ierr = ViewerPushFormat(VIEWER_STDOUT_SELF,VIEWER_FORMAT_ASCII_INFO,
           PETSC_NULL); CHKERRA(ierr);
    ierr = DFVecView(rv,VIEWER_STDOUT_SELF); CHKERRA(ierr);
    ierr = ViewerPopFormat(VIEWER_STDOUT_SELF); CHKERRA(ierr);
  }

  /* View refined vectors */
  PetscPrintf(PETSC_COMM_SELF,"refined multicomponent vector\n");
  ierr = DFVecView(rv,VIEWER_STDOUT_SELF); CHKERRA(ierr);
  ierr = VecDestroyVecs(vsub,nc); CHKERRA(ierr);
  ierr = DFVecGetComponentVectors(rv,&nsub,&vsub); CHKERRA(ierr);
  /* for (i=0; i<nc; i++) {
    PetscPrintf(PETSC_COMM_SELF,"refined vector component %d\n",i);
    ierr = VecView(vsub[i],VIEWER_STDOUT_SELF); CHKERRA(ierr);
  } */

  /* View refined grid coordinates */
  ierr = DFVecGetDFShell(rv,&rdf); CHKERRA(ierr);
  ierr = DFShellGetCoordinates(rdf,&mx,&my,&mz,&xx,&yy,&zz); CHKERRA(ierr);
  ierr = ViewCoordinates(PETSC_COMM_SELF,xx,yy,zz,mx,my,mz); CHKERRA(ierr);
  if (contours) {ierr = DFVecDrawTensorContoursX(rv,width,height); CHKERRA(ierr);}

  /* Destroy data structures */
  PetscFree(x); PetscFree(y); PetscFree(z);
  ierr = VecDestroy(v); CHKERRA(ierr);
  ierr = VecDestroy(rv); CHKERRA(ierr);
  ierr = VecDestroyVecs(vsub,nc); CHKERRA(ierr);
  ierr = DFShellDestroy(df); CHKERRA(ierr);
  ierr = DFShellDestroy(rdf); CHKERRA(ierr);
  PLogFlops(1);

  PetscFinalize();
  return 0;
}
/* --------------------------------------------------------------- */
/* 
    ViewCoordinates - Prints grid coordinates.
 */
int ViewCoordinates(MPI_Comm comm,double *x,double *y,double *z,
                    int mx,int my, int mz)
{
  int i;

  PetscPrintf(comm,"x coordinates:\n");
  for (i=0; i<mx; i++) 
    PetscPrintf(comm,"x[%d] = %g\n",i,x[i]);
  PetscPrintf(comm,"y coordinates:\n");
  for (i=0; i<my; i++)
    PetscPrintf(comm,"y[%d] = %g\n",i,y[i]);
  PetscPrintf(comm,"z coordinates:\n");
  for (i=0; i<mz; i++)
    PetscPrintf(comm,"z[%d] = %g\n",i,z[i]);

  return 0;
}

