      SUBROUTINE PBZTRMM( ICONTXT, MATBLK, SIDE, UPLO, TRANSA, DIAG, M,
     $                    N, NB, ALPHA, A, LDA, B, LDB, IAROW, IACOL,
     $                    IBPOS, ABCOMM, ABWORK, MULLEN, WORK )
*
*  -- PB-BLAS routine (version 2.1) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory.
*     April 28, 1996
*
*     Jaeyoung Choi, Oak Ridge National Laboratory
*     Jack Dongarra, University of Tennessee and Oak Ridge National Lab.
*     David Walker,  Oak Ridge National Laboratory
*
*     .. Scalar Arguments ..
      CHARACTER*1        ABCOMM, ABWORK, DIAG, MATBLK, SIDE, TRANSA,
     $                   UPLO
      INTEGER            IACOL, IAROW, IBPOS, ICONTXT, LDA, LDB, M,
     $                   MULLEN, N, NB
      COMPLEX*16         ALPHA
*     ..
*     .. Array Arguments ..
      COMPLEX*16         A( LDA, * ), B( LDB, * ), WORK( * )
*     ..
*
*  Purpose
*  =======
*
*  PBZTRMM is a parallel blocked version of the Level 3 BLAS routine
*  ZTRMM.
*  PBZTRMM performs  one of the matrix-matrix operations based on block
*  cyclic distribution.
*
*     B := alpha*op( A )*B,   or   B := alpha*B*op( A ),
*
*     op( A )*X = alpha*B,   or   X*op( A ) = alpha*B,
*
*  where alpha is a scalar, X and B are M-bt-N matrices, A is a unit, or
*  non-unit,  upper or lower triangular matrix.  op( A ) is one of
*
*     op( A ) = A,  A**T,  or  A**H
*
*  where the size of the matrix op( A ) is M-by-M if SIDE = 'L',  and N-
*  by-N  otherwise.  The M-by-N  matrix B  is a column block  (only one
*  column of processors have B) if SIDE = 'L', and a row block otherwise
*  (only one row of processors have B).  The matrix X is overwritten  on
*  B.
*
*  The first elements  of the matrices A, and B  should  be  located  at
*  the beginnings of their first blocks. (not the middle of the blocks.
*
*  B can be broadcast columnwise or rowwise, or transposed if necessary.
*  The communication scheme can be selected.
*
*  Parameters
*  ==========
*
*  ICONTXT (input) INTEGER
*          ICONTXT is the BLACS mechanism for partitioning communication
*          space.  A defining property of a context is that a message in
*          a context cannot be sent or received in another context.  The
*          BLACS context includes the definition of a grid, and each
*          process' coordinates in it.
*
*  MATBLK  (input) CHARACTER*1
*          MATBLK specifies whether op( A ) is a (full) block matrix or
*          a single block as follows:
*
*             MATBLK = 'M',  op( A ) is a (full) block matrix
*             MATBLK = 'B',  op( A ) is a single block
*
*  SIDE    (input) CHARACTER*1
*          SIDE specifies whether op( A ) multiplies B from the left or
*          right as follows:
*
*             SIDE = 'L',   B := alpha*op( A )*B.
*             SIDE = 'R',   B := alpha*B*op( A ).
*
*  UPLO    (input) CHARACTER*1
*          UPLO specifies whether the matrix A is an upper or lower
*          triangular matrix as follows:
*
*             UPLO = 'U',   A is an upper triangular matrix.
*             UPLO = 'L',   A is a lower triangular matrix.
*
*  TRANSA  (input) CHARACTER*1
*          TRANSA specifies the form of op( A ) to be used in the
*          matrix multiplication as follows:
*
*             TRANSA = 'N',   op( A ) = A.
*             TRANSA = 'T',   op( A ) = A**T.
*             TRANSA = 'C',   op( A ) = A**H.
*
*  DIAG    (input) CHARACTER*1
*          DIAG specifies whether or not A is unit triangular
*          as follows:
*
*             DIAG = 'U',  A is assumed to be unit triangular.
*             DIAG = 'N',  A is not assumed to be unit triangular.
*
*  M       (input) INTEGER
*          M specifies the (global) number of rows of B. M >= 0.
*
*  N       (input) INTEGER
*          N specifies the (global) number of columns of B. N >= 0.
*
*  NB      (input) INTEGER
*          NB specifies the row and column block size of matrix A.
*          It also specifies the row block size of the matrix B if
*          MATBLK = 'M' and SIDE = 'L', or MATBLK = 'B' and SIDE = 'R';
*          and the column block size of the matrix B if MATBLK = 'M'
*          and SIDE = 'R', or MATBLK = 'B' and SIDE = 'L'.  NB >= 1.
*
*  ALPHA   (input) COMPLEX*16
*          ALPHA specifies the scalar  alpha.  When alpha is zero then
*          A is not referenced and  B need not be set before entry.
*
*  A       (input) COMPLEX*16 array of DIMENSION ( LDA, kq ),
*          where kq is Mq (Kp is Mp) when SIDE = 'L' and is Nq (Kp is
*          Np) when SIDE = 'R'.
*          If SIDE = `L', the M-by-M part of the array A must contain
*          the (global) triangular matrix, such that when UPLO = 'U',
*          the leading M-by-M upper triangular part of the array A must
*          contain the upper triangular part of the (global) matrix and
*          the strictly  lower triangular part of A is not referenced,
*          and when  UPLO = 'L', the leading M-by-M lower triangular
*          part of the array A must  contain the lower triangular part
*          of the (global) matrix and the strictly upper triangular
*          part of A is not referenced.
*          And if SIDE = 'R', the N-by-N part of the (global) array A
*          must contain the (global) matrix, such that when UPLO = 'U',
*          the leading N-by-N upper triangular part of the array A must
*          contain the upper triangular part of the (global) matrix and
*          the strictly lower triangular part of A is not referenced,
*          and when UPLO = 'L', the leading N-by-N lower triangular
*          part of the array A must contain the lower triangular part
*          of the (global) matrix and the strictly upper triangular
*          part of A is not referenced.
*          Note that when DIAG = `U', the diagonal elements of A are
*          not referenced either, but are assumed to be unity.
*
*  LDA     (input) INTEGER
*          LDA specifies the first dimension of A as declared in the
*          calling (sub) program.  LDA >= MAX(1,Mp) if SIDE = 'L', and
*          LDA >= MAX(1,Np) otherwise.
*
*  B       (input/output) COMPLEX*16 array of DIMENSION ( LDB, Nq )
*          On entry, the leading Mp-by-Nq part of the array B must
*          contain the matrix B  when SIDE = 'R', or the leading Mp-by-
*          Nq part of the array B  must contain the (local) matrix B
*          otherwise.
*          On exit B is overwritten by the transformed matrix.  Input
*          values of B would be changed after the computation in the
*          processors which don't have the resultant column block or
*          row block of B if MATBLK = 'M'.
*
*  LDB     (input) INTEGER
*          LDB specifies the first dimension of (local) B as declared
*          in the calling (sub) program.  LDB >= MAX(1,Mp).
*
*  IAROW   (input) INTEGER
*          It specifies a row of processor template which has the
*          first block of A.  When MATBLK = 'B', SIDE = 'R', and all
*          rows of processors have their own copies of A, set IAROW
*          = -1.
*
*  IACOL   (input) INTEGER
*          It specifies a column of processor template which has the
*          first block of A.  When MATBLK = 'B', SIDE = 'L', and all
*          columns of processors have their own copies of A, set
*          IACOL = -1.
*
*  IBPOS   (input) INTEGER
*          When MATBLK = 'M', if SIDE = 'L', IBPOS specifies a column of
*          the process template, which holds the column of blocks of B
*          (-1 <= IBPOS < NPCOL).  And if SIDE = 'R', it specifies a row
*          of the template, which holds the row of blocks of B (-1 <=
*          IBPOS < NPROW).  If all columns or rows of the template have
*          their own copies of B, set IBPOS = -1.
*          When MATBLK = 'B', if SIDE = 'L', it specifies a column of
*          the template which has the first block of B (0 <= IBPOS
*          < NPCOL), and if SIDE = 'R', it specifies a row of the
*          template, which has the first block of B (0 <=IBPOS <NPROW).
*
*  ABCOMM  (input) CHARACTER*1
*          When MATBLK = 'M', ABCOMM specifies the communication scheme
*          of a row or column block of B if TRANSA <> 'N', and it is
*          ignored if TRANSA = 'N'.
*          When MATBLK = 'B', ABCOMM specifies the communication scheme
*          of a block of A.
*          It follows topology definition of BLACS.
*
*  ABWORK  (input) CHARACTER*1
*          When MATBLK = 'M', ABWORK determines whether B is a
*          workspace or not.
*
*            ABWORK = 'Y':  B is workspace in other processors.
*                           It is assumed that processors have
*                           sufficient space to store (local) B.
*            ABWORK = 'N':  Data of B in other processors will be
*                           untouched (unchanged).
*
*          And MATBLK = 'B', ABWORK determines whether A is a
*          workspace or not.
*
*            ABWORK = 'Y':  A is workspace in other processors.
*                           A is sent to A position in other processors.
*                           It is assumed that processors have
*                           sufficient space to store a single block A.
*            ABWORK = 'N':  Data of A in other processors will be
*                           untouched (unchanged).
*
*  MULLEN  (input) INTEGER
*          When MATBLK = 'M', it specifies multiplication length of the
*          optimum column number of A  for multiplying A with B.  The
*          value depends on machine characteristics. When MATBLK = 'B',
*          the argument is ignored.
*
*  WORK    (workspace) COMPLEX*16 array
*          It will store copy of B, resultant B, and portion of A
*          if necessary.
*
*  Parameters Details
*  ==================
*
*  Lx      It is  a local portion  of L  owned  by  a processor,  (L is
*          replaced by M, or N,  and x  is replaced  by  either  p
*          (=NPROW) or q (=NPCOL)).  The value is determined by  L, LB,
*          x, and MI,  where  LB is  a block size  and MI is a  row  or
*          column position in a processor template.  Lx is equal to  or
*          less than  Lx0 = CEIL( L, LB*x ) * LB.
*
*  Memory Requirement of WORK
*  ==========================
*
*  Mqb    = CEIL( M, NB*NPCOL )
*  Npb    = CEIL( N, NB*NPROW )
*  Mq0    = NUMROC( M, NB, 0, 0, NPCOL ) ~= Mqb * NB
*  Np0    = NUMROC( N, NB, 0, 0, NPROW ) ~= Npb * NB
*  LCMQ   = LCM / NPCOL
*  LCMP   = LCM / NPROW
*  ISZCMP = CEIL(MULLEN, LCMQ*NB)
*  SZCMP  = ISZCMP * ISZCMP * LCMQ*NB * LCMP*NB
*
*  (1) MATBLK = 'M'
*    (a) SIDE = 'Left'
*      (i)  TRANSA = 'N'
*         Size(WORK) = N * Mq0
*                    + Mp0 * N     (if IBPOS <> -1 and ABWORK <> 'Y') ]
*                    + MAX[ SZCMP,
*                           N*CEIL(Mqb,LCMQ)*NB    ( if IBPOS <> -1 ),
*                           N*CEIL(Mqb,LCMQ)*NB*MIN(LCMQ,CEIL(M,NB))
*                                                  ( if IBPOS =  -1 ) ]
*      (ii) TRANSA = 'T'/'C'
*         Size(WORK) = N * Mq0 + SZCMP
*                    + Mp0 * N     (if IBPOS <> -1 and ABWORK <> 'Y') ]
*    (b) SIDE = 'Right'
*      (i) TRANSA = 'N'
*         Size(WORK) = Np0 * M
*                    + M * Nq0     (if IBPOS <> -1 and ABWORK <> 'Y') ]
*                    + MAX[ SZCMP,
*                           M*CEIL(Npb,LCMP)*NB    ( if IBPOS <> -1 ),
*                           M*CEIL(Npb,LCMP)*NB*MIN(LCMP,CEIL(N,NB))
*                                                  ( if IBPOS =  -1 ) ]
*      (ii) TRANSA = 'T'/'C'
*         Size(WORK) = Np0 * M + SZCMP
*                    + M * Nq0     (if IBPOS <> -1 and ABWORK <> 'Y') ]
*
*  (2) MATBLK = 'B'
*    (a) SIDE = 'Left'
*       Size(WORK) = M * M  (in IAROW; if IACOL <> -1 and ABWORK <> 'Y')
*    (b) SIDE = 'Right'
*       Size(WORK) = N * N  (in IACOL; if IAROW <> -1 and ABWORK <> 'Y')
*
*  Notes
*  -----
*  More precise space can be computed as
*
*  CEIL(Mqb,LCMQ)*NB => NUMROC( NUMROC(M,NB,0,0,NPCOL), NB, 0, 0, LCMQ )
*                    = NUMROC( Mq0, NB, 0, 0, LCMQ )
*  CEIL(Npb,LCMP)*NB => NUMROC( NUMROC(N,NB,0,0,NPROW), NB, 0, 0, LCMP )
*                    = NUMROC( Np0, NB, 0, 0, LCMP )
*
*  =====================================================================
*
*     ..
*     .. Parameters ..
      COMPLEX*16         ONE, ZERO
      PARAMETER          ( ONE  = ( 1.0D+0, 0.0D+0 ),
     $                   ZERO = ( 0.0D+0, 0.0D+0 ) )
*     ..
*     .. Local Scalars ..
      CHARACTER*1        COMMA, COMMB, FORM
      LOGICAL            ADATA, AMAT, ASPACE, BDATA, BSPACE, LSIDE,
     $                   NOTRAN, NOUNIT, RSIDE, UPPER
      INTEGER            INFO, IPB, IPBZ, IPR, IPW, IQBZ, ISZCMP, ITER,
     $                   JJ, JNPBZ, JNQBZ, JPBZ, JQBZ, KI, KIZ, KJ, KJZ,
     $                   LCM, LCMP, LCMQ, LMW, LNW, LPBZ, LQBZ, MP,
     $                   MRCOL, MRROW, MYCOL, MYROW, MZCOL, MZROW, NDIM,
     $                   NP, NPCOL, NPROW, NQ, NS
      COMPLEX*16         DUMMY, TALPHA, TBETA
*     ..
*     .. External Functions ..
      LOGICAL            LSAME
      INTEGER            ICEIL, ILCM, NUMROC
      EXTERNAL           ICEIL, ILCM, LSAME, NUMROC
*     ..
*     .. External Subroutines ..
      EXTERNAL           BLACS_GRIDINFO, PBZLACPZ, PBZMATADD, PBZTRAN,
     $                   PXERBLA, ZGEBR2D, ZGEBS2D, ZGEMM, ZGSUM2D,
     $                   ZTRBR2D, ZTRBS2D, ZTRMM
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          DCONJG, MAX
*     ..
*     .. Executable Statements ..
*
*     Quick return if possible.
*
      IF( M.EQ.0 .OR. N.EQ.0 ) RETURN
*
      CALL BLACS_GRIDINFO( ICONTXT, NPROW, NPCOL, MYROW, MYCOL )
*
*     Test the input parameters.
*
      AMAT   = LSAME( MATBLK, 'M' )
      LSIDE  = LSAME( SIDE,   'L' )
      RSIDE  = LSAME( SIDE,   'R' )
      UPPER  = LSAME( UPLO,   'U' )
      NOTRAN = LSAME( TRANSA, 'N' )
      NOUNIT = LSAME( DIAG,   'N' )
*
      INFO = 0
      IF(      ( .NOT.AMAT                 ).AND.
     $         ( .NOT.LSAME( MATBLK, 'B' ) )        ) THEN
        INFO = 2
      ELSE IF( ( .NOT.LSIDE ) .AND. ( .NOT.RSIDE )  ) THEN
        INFO = 3
      ELSE IF( ( .NOT.UPPER                ).AND.
     $         ( .NOT.LSAME( UPLO  , 'L' ) )        ) THEN
        INFO = 4
      ELSE IF( ( .NOT.NOTRAN               ).AND.
     $         ( .NOT.LSAME( TRANSA, 'T' ) ).AND.
     $         ( .NOT.LSAME( TRANSA, 'C' ) )        ) THEN
        INFO = 5
      ELSE IF( ( .NOT.NOUNIT               ).AND.
     $         ( .NOT.LSAME( DIAG  , 'U' ) )        ) THEN
        INFO = 6
      ELSE IF( M  .LT. 0                            ) THEN
        INFO = 7
      ELSE IF( N  .LT. 0                            ) THEN
        INFO = 8
      ELSE IF( NB .LT. 1                            ) THEN
        INFO = 9
      END IF
*
   10 CONTINUE
      IF( INFO .NE. 0 ) THEN
        CALL PXERBLA( ICONTXT, 'PBZTRMM ', INFO )
        RETURN
      END IF
*
*     Start the operations.
*
* === If A is a matrix ===
*
      IF( AMAT ) THEN
        IF( LSIDE ) THEN
          NDIM = M
          NS   = N
        ELSE
          NDIM = N
          NS   = M
        END IF
        NP = NUMROC( NDIM, NB, MYROW, IAROW, NPROW )
        NQ = NUMROC( NDIM, NB, MYCOL, IACOL, NPCOL )
*
        IF( LDA.LT.MAX(1,NP)                    ) THEN
          INFO = 12
        ELSE IF( IAROW.LT.0 .OR. IAROW.GE.NPROW ) THEN
          INFO = 15
        ELSE IF( IACOL.LT.0 .OR. IACOL.GE.NPCOL ) THEN
          INFO = 16
        END IF
*
*       LCM : the least common multiple of NPROW and NPCOL
*
        LCM  = ILCM( NPROW, NPCOL )
        LCMP = LCM  / NPROW
        LCMQ = LCM  / NPCOL
        LPBZ = LCMP * NB
        LQBZ = LCMQ * NB
*
        MRROW = MOD( NPROW+MYROW-IAROW, NPROW )
        MRCOL = MOD( NPCOL+MYCOL-IACOL, NPCOL )
        BDATA = .FALSE.
        BSPACE = LSAME( ABWORK, 'Y' )
        COMMB = ABCOMM
        IF( LSAME( COMMB, ' ' ) ) COMMB = '1'
        TALPHA = ALPHA
        IF( LSAME( TRANSA, 'C' ) ) TALPHA = DCONJG( ALPHA )
*
*       PART 1: Distribute a column (or row) block B or its transpose B'
*       ================================================================
*
        IF( LSIDE ) THEN
*
*         Form  B := alpha*op( A )*B.
*            _               _____________       _
*           | |             |\_           |     | |
*           | |             |  \_         |     | |
*           | |             |    \_       |     | |
*           |B|  =  alpha * |    op(A)    |  *  |B|
*           | |             |        \_   |     | |
*           | |             |          \_ |     | |
*           |_|             |____________\|     |_|
*
          IF( LDB.LT.MAX(1,NP) .AND. ( BSPACE .OR.
     $      IBPOS.EQ.MYCOL .OR. IBPOS.EQ.-1 )           ) THEN
            INFO = 14
          ELSE IF( IBPOS.LT.-1 .OR. IBPOS.GE.NPCOL      ) THEN
            INFO = 17
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
          IF( NOTRAN ) THEN
*
*           Transpose a column block of B to WORK(IPB)
*
            IPB = 1
            IPR = N * NQ + IPB
            CALL PBZTRAN( ICONTXT, 'Col', 'T', M, N, NB, B, LDB, ZERO,
     $                    WORK(IPB), N, IAROW, IBPOS, -1, IACOL,
     $                    WORK(IPR) )
*
            IF( BSPACE ) THEN
              CALL PBZMATADD( ICONTXT, 'G', NP, N, ZERO, DUMMY, 1, ZERO,
     $                        B, LDB )
              BDATA = .TRUE.
              IPW = IPR
            ELSE
              CALL PBZMATADD( ICONTXT, 'G', NP, N, ZERO, DUMMY, 1, ZERO,
     $                        WORK(IPR), NP )
              IPW = NP * N + IPR
            END IF
*
*         if TRANSA = 'Transpose' or 'Conjugate'
*
          ELSE
*
*           Broadcast B if necessary
*
            IPR = 1
            IPB = N * NQ + IPR
            IPW = IPB
*
            IF( IBPOS.EQ.-1 ) THEN
              BDATA = .TRUE.
            ELSE
              IF( BSPACE ) THEN
                IF( MYCOL.EQ.IBPOS ) THEN
                  CALL ZGEBS2D( ICONTXT, 'Row', COMMB, NP, N, B, LDB )
                ELSE
                  CALL ZGEBR2D( ICONTXT, 'Row', COMMB, NP, N, B, LDB,
     $                           MYROW, IBPOS )
                END IF
                BDATA = .TRUE.
              ELSE
                IF( MYCOL.EQ.IBPOS ) THEN
                  CALL PBZMATADD( ICONTXT, 'V', NP, N, ONE, B, LDB,
     $                            ZERO, WORK(IPB), NP )
                  CALL ZGEBS2D( ICONTXT, 'Row', COMMB, NP, N,
     $                          WORK(IPB), NP )
                ELSE
                  CALL ZGEBR2D( ICONTXT, 'Row', COMMB, NP, N,
     $                          WORK(IPB), NP, MYROW, IBPOS )
                END IF
                IPW = NP * N + IPB
              END IF
            END IF
*
            CALL PBZMATADD( ICONTXT, 'G', N, NQ, ZERO, DUMMY, 1, ZERO,
     $                      WORK(IPR), N )
          END IF
*
*       If SIDE = 'Right'
*
        ELSE
*
*         Form  B := alpha*B*op( A ).
*                                                     _____________
*                                                    |\_           |
*                                                    |  \_         |
*         _____________             _____________    |    \_       |
*        |______B______| = alpha * |______B______| * |    op(A)    |
*                                                    |        \_   |
*                                                    |          \_ |
*                                                    |____________\|
*
          IF( LDB.LT.MAX(1,M) .AND. ( BSPACE .OR.
     $      IBPOS.EQ.MYROW .OR. IBPOS.EQ.-1 )          ) THEN
            INFO = 14
          ELSE IF( IBPOS.LT.-1 .OR. IBPOS.GE.NPROW     ) THEN
            INFO = 17
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
          IF( NOTRAN ) THEN
*
*           Transpose a column block of B to WORK(IPB),
*
            IPB = 1
            IPR = NP * M + IPB
            CALL PBZTRAN( ICONTXT, 'Row', 'T', M, N, NB, B, LDB, ZERO,
     $                    WORK(IPB), NP, IBPOS, IACOL, IAROW, -1,
     $                    WORK(IPR) )
*
            IF( BSPACE ) THEN
              CALL PBZMATADD( ICONTXT, 'G', M, NQ, ZERO, DUMMY, 1, ZERO,
     $                        B, LDB )
              BDATA = .TRUE.
              IPW = IPR
            ELSE
              CALL PBZMATADD( ICONTXT, 'G', M, NQ, ZERO, DUMMY, 1, ZERO,
     $                        WORK(IPR), M )
              IPW = M * NQ + IPR
            END IF
*
*         If TRANSA = 'Transpose' or 'Conjugate'
*
          ELSE
*
*           Broadcast B if necessary
*
            IPR = 1
            IPB = NP * M + IPR
            IPW = IPB
*
            IF( IBPOS.EQ.-1 ) THEN
              BDATA = .TRUE.
            ELSE
              IF( BSPACE ) THEN
                IF( MYROW.EQ.IBPOS ) THEN
                  CALL ZGEBS2D( ICONTXT, 'Col', COMMB, M, NQ, B, LDB )
                ELSE
                  CALL ZGEBR2D( ICONTXT, 'Col', COMMB, M, NQ, B, LDB,
     $                          IBPOS, MYCOL )
                END IF
                BDATA = .TRUE.
              ELSE
                IF( MYROW.EQ.IBPOS ) THEN
                  CALL PBZMATADD( ICONTXT, 'G', M, NQ, ONE, B, LDB,
     $                            ZERO,  WORK(IPB), M )
                  CALL ZGEBS2D( ICONTXT, 'Col', COMMB, M, NQ,
     $                          WORK(IPB), M )
                ELSE
                  CALL ZGEBR2D( ICONTXT, 'Col', COMMB, M, NQ,
     $                          WORK(IPB), M, IBPOS, MYCOL )
                END IF
                IPW = M * NQ + IPB
              END IF
            END IF
*
            CALL PBZMATADD( ICONTXT, 'G', NP, M, ZERO, DUMMY, 1, ZERO,
     $                      WORK(IPR), NP )
          END IF
        END IF
*
*       PART 2: Compute B locally
*       =========================
*
        IF( NP.EQ.0 .OR. NQ.EQ.0 ) GO TO 100
*
        IF( UPPER ) THEN
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          ITER = ICEIL( NQ, IQBZ )
          JPBZ = 0
          JQBZ = 0
*
          DO 50 JJ = 0, ITER-1
            LMW   = MIN( IPBZ, NP-JPBZ )
            LNW   = MIN( IQBZ, NQ-JQBZ )
            JNPBZ = JPBZ + LMW
            JNQBZ = JQBZ + LNW
*
*           Copy the upper triangular matrix A to WORK(IPW)
*
            MZROW = MRROW
            MZCOL = MRCOL
            KI = 0
*
            DO 30 KJ = 0, LCMQ-1
   20         CONTINUE
              IF( MZROW.LT.MZCOL ) THEN
                MZROW = MZROW + NPROW
                KI = KI + 1
                GO TO 20
              END IF
              KIZ = KI * NB
              KJZ = KJ * NB
              IF( KJZ.GE.LNW ) GO TO 40
              FORM = 'G'
              IF( MZROW.EQ.MZCOL ) FORM = 'T'
              MZCOL = MZCOL + NPCOL
*
              CALL PBZLACPZ( ICONTXT, 'Upper', FORM, DIAG, KIZ, NB,
     $                       A(JPBZ+1,JQBZ+KJZ+1), LDA,
     $                       WORK(KJZ*LMW+IPW), LMW,
     $                       LPBZ, LQBZ, LMW, LNW-KJZ )
   30       CONTINUE
   40       CONTINUE
*
*           If SIDE = 'Left',
*
            IF( LSIDE ) THEN
*
*             Compute B if A is not transposed
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL ZGEMM( 'No', 'Trans', LMW, NS, LNW, ALPHA,
     $                        WORK(IPW), MAX(1,LMW), WORK(JQBZ*NS+IPB),
     $                        NS, ZERO, B(JPBZ+1,1), LDB )
                  CALL ZGEMM( 'No', 'Trans', JPBZ, NS, LNW, ALPHA,
     $                        A(1,JQBZ+1), LDA, WORK(JQBZ*NS+IPB), NS,
     $                        ONE, B, LDB )
                ELSE
                  CALL ZGEMM( 'No', 'Trans', LMW, NS, LNW, ALPHA,
     $                        WORK(IPW), MAX(1,LMW), WORK(JQBZ*NS+IPB),
     $                        NS, ZERO, WORK(JPBZ+IPR), NP )
                  CALL ZGEMM( 'No', 'Trans', JPBZ, NS, LNW, ALPHA,
     $                        A(1,JQBZ+1), LDA, WORK(JQBZ*NS+IPB), NS,
     $                        ONE, WORK(IPR), NP )
                END IF
*
*             Compute B if A is (conjugate) transposed
*
              ELSE
                IF( BDATA ) THEN
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, LMW, TALPHA,
     $                        B(JPBZ+1,1), LDB, WORK(IPW), MAX(1,LMW),
     $                        ZERO, WORK(NS*JQBZ+IPR), NS )
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, JPBZ, TALPHA,
     $                        B, LDB, A(1,JQBZ+1), LDA, ONE,
     $                        WORK(NS*JQBZ+IPR), NS )
                ELSE
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, LMW, TALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(IPW),
     $                        MAX(1,LMW), ZERO, WORK(NS*JQBZ+IPR), NS )
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, JPBZ, TALPHA,
     $                        WORK(IPB), NP, A(1,JQBZ+1), LDA, ONE,
     $                        WORK(NS*JQBZ+IPR), NS )
                END IF
              END IF
*
*           If SIDE = 'Right',
*
            ELSE
*
*             Compute B if A is not transposed
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, LMW, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(IPW),
     $                        MAX(1,LMW), ZERO, B(1,JQBZ+1), NS )
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, JPBZ, ALPHA,
     $                        WORK(IPB), NP, A(1,JQBZ+1), LDA, ONE,
     $                        B(1,JQBZ+1), NS )
                ELSE
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, LMW, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(IPW),
     $                        MAX(1,LMW), ZERO, WORK(NS*JQBZ+IPR), NS )
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, JPBZ, ALPHA,
     $                        WORK(IPB), NP, A(1,JQBZ+1), LDA, ONE,
     $                        WORK(NS*JQBZ+IPR), NS )
                END IF
*
*             Compute B if A is (conjugate) transposed
*
              ELSE
                IF( BDATA ) THEN
                  CALL ZGEMM( 'No', TRANSA, LMW, NS, LNW, TALPHA,
     $                        WORK(IPW), MAX(1,LMW), B(1,JQBZ+1), LDB,
     $                        ZERO, WORK(JPBZ+IPR), NP )
                  CALL ZGEMM( 'No', TRANSA, JPBZ, NS, LNW, TALPHA,
     $                        A(1,JQBZ+1), LDA, B(1,JQBZ+1), LDB,
     $                        ONE, WORK(IPR), NP )
                ELSE
                  CALL ZGEMM( 'No', TRANSA, LMW, NS, LNW, TALPHA,
     $                        WORK(IPW), MAX(1,LMW), WORK(JQBZ*NS+IPB),
     $                        NS, ZERO, WORK(JPBZ+IPR), NP )
                  CALL ZGEMM( 'No', TRANSA, JPBZ, NS, LNW, TALPHA,
     $                        A(1,JQBZ+1), LDA, WORK(JQBZ*NS+IPB), NS,
     $                        ONE, WORK(IPR), NP )
                END IF
              END IF
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JNQBZ
   50     CONTINUE
*
*       If A is a lower triangular matrix,
*
        ELSE
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ  = ISZCMP * LPBZ
          IQBZ  = ISZCMP * LQBZ
          ITER  = ICEIL( NQ, IQBZ )
          JPBZ  = 0
          JQBZ  = 0
          TBETA = ZERO
*
          DO 90 JJ = 0, ITER-1
            LMW   = MIN( IPBZ, NP-JPBZ )
            LNW   = MIN( IQBZ, NQ-JQBZ )
            JNPBZ = JPBZ + LMW
            JNQBZ = JQBZ + LNW
*
*           Copy the lower triangular matrix A to WORK(IPW)
*
            MZROW = MRROW
            MZCOL = MRCOL
            KI = 0
*
            DO 70 KJ = 0, LCMQ-1
   60         CONTINUE
              IF( MZROW.LT.MZCOL ) THEN
                MZROW = MZROW + NPROW
                KI = KI + 1
                GO TO 60
              END IF
              KIZ = KI * NB
              KJZ = KJ * NB
              IF( KJZ.GE.LNW ) GO TO 80
              FORM = 'G'
              IF( MZROW.EQ.MZCOL ) FORM = 'T'
              MZCOL = MZCOL + NPCOL
*
              CALL PBZLACPZ( ICONTXT, 'Lower', FORM, DIAG, KIZ, NB,
     $                       A(JPBZ+1,JQBZ+KJZ+1), LDA,
     $                       WORK(KJZ*LMW+IPW), LMW,
     $                       LPBZ, LQBZ, LMW, LNW-KJZ )
   70       CONTINUE
   80       CONTINUE
*
*           If SIDE = 'Left',
*
            IF( LSIDE ) THEN
*
*             Compute B if A is not transposed
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL ZGEMM( 'No', 'Trans', LMW, NS, LNW, ALPHA,
     $                        WORK(IPW), MAX(1,LMW), WORK(JQBZ*NS+IPB),
     $                        NS, TBETA, B(JPBZ+1,1), LDB )
                  CALL ZGEMM( 'No', 'Trans', NP-JNPBZ, NS, LNW, ALPHA,
     $                        A(JNPBZ+1,JQBZ+1), LDA, WORK(JQBZ*NS+IPB),
     $                        NS, TBETA, B(JNPBZ+1,1), LDB )
                ELSE
                  CALL ZGEMM( 'No', 'Trans', LMW, NS, LNW, ALPHA,
     $                        WORK(IPW), MAX(1,LMW), WORK(JQBZ*NS+IPB),
     $                        NS, TBETA, WORK(JPBZ+IPR), NP )
                  CALL ZGEMM( 'No', 'Trans', NP-JNPBZ, NS, LNW, ALPHA,
     $                        A(JNPBZ+1,JQBZ+1), LDA, WORK(JQBZ*NS+IPB),
     $                        NS, TBETA, WORK(JNPBZ+IPR), NP )
              END IF
*
*             Compute B if A is (conjugate) transposed
*
              ELSE
                IF( BDATA ) THEN
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, LMW, TALPHA,
     $                        B(JPBZ+1,1), LDB, WORK(IPW), MAX(1,LMW),
     $                        ZERO, WORK(NS*JQBZ+IPR), NS )
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, NP-JNPBZ, TALPHA,
     $                        B(JNPBZ+1,1), LDB, A(JNPBZ+1,JQBZ+1),
     $                        LDA, ONE, WORK(NS*JQBZ+IPR), NS )
                ELSE
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, LMW, TALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(IPW),
     $                        MAX(1,LMW), ZERO, WORK(NS*JQBZ+IPR), NS )
                  CALL ZGEMM( TRANSA, 'No', NS, LNW, NP-JNPBZ, TALPHA,
     $                        WORK(JNPBZ+IPB), NP, A(JNPBZ+1,JQBZ+1),
     $                        LDA, ONE, WORK(NS*JQBZ+IPR), NS )
                END IF
              END IF
*
*           If SIDE = 'Right',
*
            ELSE
*
*             Compute B if A is not transposed
*
              IF( NOTRAN ) THEN
                IF( BDATA ) THEN
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, LMW, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(IPW),
     $                        MAX(1,LMW), ZERO, B(1,JQBZ+1), LDB )
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, NP-JNPBZ, ALPHA,
     $                        WORK(JNPBZ+IPB), NP, A(JNPBZ+1,JQBZ+1),
     $                        LDA, ONE, B(1,JQBZ+1), LDB )
                ELSE
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, LMW, ALPHA,
     $                        WORK(JPBZ+IPB), NP, WORK(IPW),
     $                        MAX(1,LMW), ZERO, WORK(NS*JQBZ+IPR), NS )
                  CALL ZGEMM( 'Trans', 'No', NS, LNW, NP-JNPBZ, ALPHA,
     $                        WORK(JNPBZ+IPB), NP, A(JNPBZ+1,JQBZ+1),
     $                        LDA, ONE, WORK(NS*JQBZ+IPR), NS )
                END IF
*
*             Compute B if A is (conjugate) transposed
*
              ELSE
                IF( BDATA ) THEN
                  CALL ZGEMM( 'No', TRANSA, LMW, NS, LNW, TALPHA,
     $                        WORK(IPW), MAX(1,LMW), B(1,JQBZ+1), LDB,
     $                        TBETA, WORK(JPBZ+IPR), NP )
                  CALL ZGEMM( 'No', TRANSA, NP-JNPBZ, NS, LNW, TALPHA,
     $                        A(JNPBZ+1,JQBZ+1), LDA, B(1,JQBZ+1), LDB,
     $                        TBETA, WORK(JNPBZ+IPR), NP )
                ELSE
                  CALL ZGEMM( 'No', TRANSA, LMW, NS, LNW, TALPHA,
     $                        WORK(IPW), MAX(1,LMW), WORK(JQBZ*NS+IPB),
     $                        NS, TBETA, WORK(JPBZ+IPR), NP )
                  CALL ZGEMM( 'No', TRANSA, NP-JNPBZ, NS, LNW, TALPHA,
     $                        A(JNPBZ+1,JQBZ+1),LDA, WORK(JQBZ*NS+IPB),
     $                        NS, TBETA, WORK(JNPBZ+IPR), NP )
                END IF
              END IF
            END IF
*
            TBETA = ONE
            JPBZ = JNPBZ
            JQBZ = JNQBZ
   90     CONTINUE
        END IF
*
  100   CONTINUE
*
*       PART 3: Collect B, and transpose it if necessary
*       ================================================
*
        IF( LSIDE ) THEN
          IF( NOTRAN ) THEN
            IF( BDATA ) THEN
              CALL ZGSUM2D( ICONTXT, 'Row', '1-tree', NP, NS, B, LDB,
     $                      MYROW, IBPOS )
            ELSE
              IF( MYCOL.EQ.IBPOS ) THEN
                CALL PBZMATADD( ICONTXT, 'V', NP, NS, ONE, WORK(IPR),
     $                          NP, ZERO, B, LDB )
                CALL ZGSUM2D( ICONTXT, 'Row', '1-tree', NP, NS, B, LDB,
     $                        MYROW, IBPOS )
              ELSE
                CALL ZGSUM2D( ICONTXT, 'Row', '1-tree', NP, NS,
     $                        WORK(IPR), NP, MYROW, IBPOS )
                IF( IBPOS.EQ.-1 )
     $            CALL PBZMATADD( ICONTXT, 'V', NP, NS, ONE, WORK(IPR),
     $                            NP, ZERO, B, LDB )
              END IF
            END IF
*
          ELSE
            CALL ZGSUM2D( ICONTXT, 'Col', '1-tree', NS, NQ, WORK(IPR),
     $                    NS,  IAROW, MYCOL)
            CALL PBZTRAN( ICONTXT, 'Row', TRANSA, N, M, NB, WORK(IPR),
     $                    NS, ZERO, B, LDB, IAROW, IACOL, IAROW, IBPOS,
     $                    WORK(IPB) )
          END IF
*
        ELSE
          IF( NOTRAN ) THEN
            IF( BDATA ) THEN
              CALL ZGSUM2D( ICONTXT, 'Col', '1-tree', NS, NQ, B, LDB,
     $                      IBPOS, MYCOL )
            ELSE
              IF( MYROW.EQ.IBPOS ) THEN
                CALL PBZMATADD( ICONTXT, 'G', NS, NQ, ONE, WORK(IPR),
     $                          NS, ZERO, B, LDB )
                CALL ZGSUM2D( ICONTXT, 'Col', '1-tree', NS,NQ, B, LDB,
     $                        IBPOS, MYCOL )
              ELSE
                CALL ZGSUM2D( ICONTXT, 'Col', '1-tree', NS, NQ,
     $                        WORK(IPR), NS, IBPOS, MYCOL )
                IF( IBPOS.EQ.-1 )
     $            CALL PBZMATADD( ICONTXT, 'G', NS, NQ, ONE, WORK(IPR),
     $                            NS, ZERO, B, LDB )
              END IF
            END IF
*
          ELSE
            CALL ZGSUM2D( ICONTXT, 'Row', '1-tree', NP, NS, WORK(IPR),
     $                    NP, MYROW, IACOL)
            CALL PBZTRAN( ICONTXT, 'Col', TRANSA, N, M, NB, WORK(IPR),
     $                    NP, ZERO, B, LDB, IAROW, IACOL, IBPOS, IACOL,
     $                    WORK(IPB) )
          END IF
        END IF
*
* === If A is just a block ===
*
      ELSE
        ADATA = .FALSE.
        ASPACE = LSAME( ABWORK, 'Y' )
        COMMA = ABCOMM
        IF( LSAME( COMMA, ' ' ) ) COMMA = '1'
*
        IF( LSIDE .AND. MYROW.EQ.IAROW ) THEN
*
*         Form  B := alpha*op( A )*B.
*            _____________       _       _____________
*           |______B______|  =  |_|  *  |______B______|
*                              op(A)
*
          IF( IACOL.EQ.-1 )  ADATA = .TRUE.
          NQ = NUMROC( N, NB, MYCOL, IBPOS, NPCOL )
*
          IF( LDA.LT.MAX(1,M) .AND. ( ASPACE .OR.
     $             IACOL.EQ.MYCOL .OR. IACOL.EQ.-1 ) ) THEN
            INFO = 12
          ELSE IF( LDB  .LT. M                       ) THEN
            INFO = 14
          ELSE IF( IAROW.LT. 0 .OR. IAROW.GE.NPROW   ) THEN
            INFO = 15
          ELSE IF( IACOL.LT.-1 .OR. IACOL.GE.NPCOL   ) THEN
            INFO = 16
          ELSE IF( IBPOS.LT. 0 .OR. IBPOS.GE.NPCOL   ) THEN
            INFO = 17
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Broadcast A if necessary
*
          IF( .NOT. ADATA ) THEN
            IF( ASPACE ) THEN
              IF( MYCOL.EQ.IACOL ) THEN
                CALL ZTRBS2D( ICONTXT, 'Row', COMMA, UPLO, DIAG, M, M,
     $                        A, LDA )
              ELSE
                CALL ZTRBR2D( ICONTXT, 'Row', COMMA, UPLO, DIAG, M, M,
     $                        A, LDA, MYROW, IACOL )
              END IF
              ADATA = .TRUE.
            ELSE
              IF( MYCOL.EQ.IACOL ) THEN
                CALL ZTRBS2D( ICONTXT, 'Row', COMMA, UPLO, DIAG, M, M,
     $                        A, LDA )
                CALL PBZMATADD( ICONTXT, UPLO, M, M, ONE, A, LDA, ZERO,
     $                          WORK, M )
              ELSE
                CALL ZTRBR2D( ICONTXT, 'Row', COMMA, UPLO, DIAG, M, M,
     $                        WORK, M, MYROW, IACOL )
              END IF
            END IF
          END IF
*
*         Compute ZTRMM
*
          IF( ADATA ) THEN
            CALL ZTRMM( 'Left', UPLO, TRANSA, DIAG, M, NQ, ALPHA,
     $                  A, LDA, B, LDB )
          ELSE
            CALL ZTRMM( 'Left', UPLO, TRANSA, DIAG, M, NQ, ALPHA,
     $                  WORK, M, B, LDB )
          END IF
*
        ELSE IF( LSAME( SIDE, 'R' ) .AND. MYCOL.EQ.IACOL ) THEN
*
*         Form  B := alpha*B*op( A ).
*                _         _
*               | |       | |
*               | |       | |
*               | |       | |        _
*               |B|    =  |B|   *   |_|
*               | |       | |       op(A)
*               | |       | |
*               |_|       |_|
*
          IF( IAROW.EQ.-1 )  ADATA = .TRUE.
          MP = NUMROC( M, NB, MYROW, IBPOS, NPROW )
*
          IF( LDA.LT.MAX(1,N) .AND. ( ASPACE .OR.
     $             IAROW.EQ.MYROW .OR. IAROW.EQ.-1 ) ) THEN
            INFO = 12
          ELSE IF( LDB .LT.MAX(1,MP)                 ) THEN
            INFO = 14
          ELSE IF( IAROW.LT.-1 .OR. IAROW.GE.NPROW   ) THEN
            INFO = 15
          ELSE IF( IACOL.LT. 0 .OR. IACOL.GE.NPCOL   ) THEN
            INFO = 16
          ELSE IF( IBPOS.LT. 0 .OR. IBPOS.GE.NPROW   ) THEN
            INFO = 17
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Broadcast B if necessary
*
          IF( .NOT.ADATA ) THEN
            IF( ASPACE ) THEN
              IF( MYROW.EQ.IAROW ) THEN
                CALL ZTRBS2D( ICONTXT, 'Col', COMMA, UPLO, DIAG, N, N,
     $                        A, LDA )
              ELSE
                CALL ZTRBR2D( ICONTXT, 'Col', COMMA, UPLO, DIAG, N, N,
     $                        A, LDA, IAROW, MYCOL )
              END IF
              ADATA = .TRUE.
            ELSE
              IF( MYROW.EQ.IAROW ) THEN
                CALL ZTRBS2D( ICONTXT, 'Col', COMMA, UPLO, DIAG, N, N,
     $                        A, LDA )
                CALL PBZMATADD( ICONTXT, UPLO, N, N, ONE, A, LDA, ZERO,
     $                          WORK, N )
              ELSE
                CALL ZTRBR2D( ICONTXT, 'Col', COMMA, UPLO, DIAG, N, N,
     $                        WORK, N, IAROW, MYCOL )
              END IF
            END IF
          END IF
*
*         Compute ZTRMM
*
          IF( ADATA ) THEN
            CALL ZTRMM( 'Right', UPLO, TRANSA, DIAG, MP, N, ALPHA,
     $                  A, LDA, B, LDB )
          ELSE
            CALL ZTRMM( 'Right', UPLO, TRANSA, DIAG, MP, N, ALPHA,
     $                  WORK, N, B, LDB )
          END IF
        END IF
      END IF
*
      RETURN
*
*     End of PBZTRMM
*
      END
