#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: common8and9.c,v 1.4 1998/03/31 17:17:59 balay Exp $";
#endif

/* ------------------------------------------------------------------------

    This file contains routines for initial guess and nonlinear function 
    evaluation that are used by both two example programs:
         ex8.c - supports solution on a single grid
         ex9.c - incorporates mesh sequencing

    We thank David E. Keyes for contributing the driven cavity discretization
    within this example code.

    This problem is modeled by the partial differential equation system
  
	- Lap( U ) - Grad_y( Omega ) = 0
	- Lap( V ) + Grad_x( Omega ) = 0
	- Lap( Omega ) + Div( [U*Omega,V*Omega] ) - GR*Grad_x( T ) = 0
	- Lap( T ) + PR*Div( [U*T,V*T] ) = 0

    in the unit square, which is uniformly discretized in each of x and
    y in this simple encoding.

    No-slip, rigid-wall Dirichlet conditions are used for [U,V].
    Dirichlet conditions are used for Omega, based on the definition of
    vorticity: Omega = - Grad_y( U ) + Grad_x( V ), where along each
    constant coordinate boundary, the tangential derivative is zero.
    Dirichlet conditions are used for T on the left and right walls,
    and insulation homogeneous Neumann conditions are used for T on
    the top and bottom walls. 

    A finite difference approximation with the usual 5-point stencil 
    is used to discretize the boundary value problem to obtain a 
    nonlinear system of equations.  Upwinding is used for the divergence
    (convective) terms and central for the gradient (source) terms.
    
  ------------------------------------------------------------------------- */

/* 
   Include "da.h" so that we can use distributed arrays (DAs).
   Include "snes.h" so that we can use SNES solvers.  Note that this
   file automatically includes:
     petsc.h  - base PETSc routines   vec.h - vectors
     sys.h    - system routines       mat.h - matrices
     is.h     - index sets            ksp.h - Krylov subspace methods
     viewer.h - viewers               pc.h  - preconditioners
     sles.h   - linear solvers        dfvec.h - for visualization of solution
                                                and refinement
*/
#include "snes.h"
#include "da.h"
#include "dfvec.h"
#include "ex8and9.h"
#include <math.h>
#include <stdio.h>

/* ------------------------------------------------------------------- */

/*
   Define macros to allow us to easily access the components of the PDE
   solution and nonlinear residual vectors.
      Note: the "4" below is a hardcoding of "user.mc" 
*/
#define U(i)     4*(i)
#define V(i)     4*(i)+1
#define Omega(i) 4*(i)+2
#define Temp(i)  4*(i)+3

/* ------------------------------------------------------------------- */
/* 
   FormInitialGuess - Forms initial approximation.

   Input Parameters:
   user - user-defined application context
   X - vector

   Output Parameter:
   X - vector
 */
int FormInitialGuess(AppCtx *user,Vec X)
{
  int     i, j, row, mx, ierr, xs, ys, xm, ym, gxm, gym, gxs, gys;
  double  grashof;
  Scalar  *x;
  Vec     localX = user->localX;

  mx = user->mx;
  grashof = user->grashof;

  /*
     Get a pointer to vector data.
       - For default PETSc vectors, VecGetArray() returns a pointer to
         the data array.  Otherwise, the routine is implementation dependent.
       - You MUST call VecRestoreArray() when you no longer need access to
         the array.
  */
  ierr = VecGetArray(localX,&x); CHKERRQ(ierr);

  /*
     Get local grid boundaries (for 2-dimensional DA):
       xs, ys   - starting grid indices (no ghost points)
       xm, ym   - widths of local grid (no ghost points)
       gxs, gys - starting grid indices (including ghost points)
       gxm, gym - widths of local grid (including ghost points)
  */
  ierr = DAGetCorners(user->da,&xs,&ys,PETSC_NULL,&xm,&ym,PETSC_NULL); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(user->da,&gxs,&gys,PETSC_NULL,&gxm,&gym,PETSC_NULL); CHKERRQ(ierr);

  /*
     Compute initial guess over the locally owned part of the grid
     Initial condition is motionless fluid and equilibrium temperature
  */
  for (j=ys; j<ys+ym; j++) {
    for (i=xs; i<xs+xm; i++) {
      row = i - gxs + (j - gys)*gxm; 
      x[U(row)]     = 0.0;
      x[V(row)]     = 0.0;
      x[Omega(row)] = 0.0;
      x[Temp(row)]  = (grashof>0)*(double)(i)/(double)(mx-1);
    }
  }

  /*
     Restore vector
  */
  ierr = VecRestoreArray(localX,&x); CHKERRQ(ierr);

  /*
     Insert values into global vector
  */
  ierr = DALocalToGlobal(user->da,localX,INSERT_VALUES,X); CHKERRQ(ierr);
  return 0;
} 
/* ------------------------------------------------------------------- */
/* 
   FormFunction - Evaluates the nonlinear function, F(x).

   Input Parameters:
.  snes - the SNES context
.  X - input vector
.  ptr - optional user-defined context, as set by SNESSetFunction()

   Output Parameter:
.  F - function vector

   Notes:
   We process the boundary nodes before handling the interior
   nodes, so that no conditional statements are needed within the
   double loop over the local grid indices. 
 */
int FormFunction(SNES snes,Vec X,Vec F,void *ptr)
{
  AppCtx  *user = (AppCtx *) ptr;
  int     ierr, i, j, row, mx, my, xs, ys, xm, ym, gxs, gys, gxm, gym;
  int     xints, xinte, yints, yinte;
  double  two = 2.0, one = 1.0, p5 = 0.5, hx, hy, dhx, dhy, hxdhy, hydhx;
  double  grashof, prandtl, lid;
  Scalar  u, uxx, uyy, vx, vy, avx, avy, vxp, vxm, vyp, vym;
  Scalar  *x, *f;
  Vec     localX = user->localX, localF = user->localF; 

  mx = user->mx;            my = user->my;            
  grashof = user->grashof;  prandtl = user->prandtl;
  lid = user->lidvelocity;

  /* 
     Define mesh intervals ratios for uniform grid.
     [Note: FD formulae below are normalized by multiplying through by
     local volume element to obtain coefficients O(1) in two dimensions.]
  */
  dhx = (double)(mx-1);     dhy = (double)(my-1);
  hx = one/dhx;             hy = one/dhy;
  hxdhy = hx*dhy;           hydhx = hy*dhx;

  /*
     Scatter ghost points to local vector, using the 2-step process
        DAGlobalToLocalBegin(), DAGlobalToLocalEnd().
     By placing code between these two statements, computations can be
     done while messages are in transition.
  */
  ierr = DAGlobalToLocalBegin(user->da,X,INSERT_VALUES,localX); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(user->da,X,INSERT_VALUES,localX); CHKERRQ(ierr);

  /*
     Get pointers to vector data
  */
  ierr = VecGetArray(localX,&x); CHKERRQ(ierr);
  ierr = VecGetArray(localF,&f); CHKERRQ(ierr);

  /*
     Get local grid boundaries
  */
  ierr = DAGetCorners(user->da,&xs,&ys,PETSC_NULL,&xm,&ym,PETSC_NULL); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(user->da,&gxs,&gys,PETSC_NULL,&gxm,&gym,PETSC_NULL); CHKERRQ(ierr);

  /*
     Compute function over the locally owned part of the grid
     (physical corner points are set twice to avoid more conditionals).
  */
  xints = xs; xinte = xs+xm; yints = ys; yinte = ys+ym;

  /* Test whether we are on the bottom edge of the global array */
  if (yints == 0) {
    yints = yints + 1;
    /* bottom edge */
    row = xs - gxs - 1; 
    for (i=xs; i<xs+xm; i++) {
      row++;
        f[U(row)]     = x[U(row)];
        f[V(row)]     = x[V(row)];
        f[Omega(row)] = x[Omega(row)] + (x[U(row+gxm)] - x[U(row)])*dhy; 
	f[Temp(row)]  = x[Temp(row)]-x[Temp(row+gxm)];
    }
  }

  /* Test whether we are on the top edge of the global array */
  if (yinte == my) {
    yinte = yinte - 1;
    /* top edge */
    row = (ys + ym - 1 - gys)*gxm + xs - gxs - 1; 
    for (i=xs; i<xs+xm; i++) {
      row++;
        f[U(row)]     = x[U(row)] - lid;
        f[V(row)]     = x[V(row)];
        f[Omega(row)] = x[Omega(row)] + (x[U(row)] - x[U(row-gxm)])*dhy; 
	f[Temp(row)]  = x[Temp(row)]-x[Temp(row-gxm)];
    }
  }

  /* Test whether we are on the left edge of the global array */
  if (xints == 0) {
    xints = xints + 1;
    /* left edge */
    for (j=ys; j<ys+ym; j++) {
      row = (j - gys)*gxm + xs - gxs; 
      f[U(row)]     = x[U(row)];
      f[V(row)]     = x[V(row)];
      f[Omega(row)] = x[Omega(row)] - (x[V(row+1)] - x[V(row)])*dhx; 
      f[Temp(row)]  = x[Temp(row)];
    }
  }

  /* Test whether we are on the right edge of the global array */
  if (xinte == mx) {
    xinte = xinte - 1;
    /* right edge */ 
    for (j=ys; j<ys+ym; j++) {
      row = (j - gys)*gxm + xs + xm - gxs - 1; 
      f[U(row)]     = x[U(row)];
      f[V(row)]     = x[V(row)];
      f[Omega(row)] = x[Omega(row)] - (x[V(row)] - x[V(row-1)])*dhx; 
      f[Temp(row)]  = x[Temp(row)] - (grashof>0);
    }
  }

  /* Compute over the interior points */
  for (j=yints; j<yinte; j++) {
    row = (j - gys)*gxm + xints - gxs - 1; 
    for (i=xints; i<xinte; i++) {
      row++;

	/*
	  convective coefficients for upwinding
        */
	vx = x[U(row)]; avx = PetscAbsScalar(vx);
        vxp = p5*(vx+avx); vxm = p5*(vx-avx);
	vy = x[V(row)]; avy = PetscAbsScalar(vy);
        vyp = p5*(vy+avy); vym = p5*(vy-avy);

	/* U velocity */
        u          = x[U(row)];
        uxx        = (two*u - x[U(row-1)] - x[U(row+1)])*hydhx;
        uyy        = (two*u - x[U(row-gxm)] - x[U(row+gxm)])*hxdhy;
        f[U(row)]  = uxx + uyy - p5*(x[Omega(row+gxm)]-x[Omega(row-gxm)])*hx;

	/* V velocity */
        u          = x[V(row)];
        uxx        = (two*u - x[V(row-1)] - x[V(row+1)])*hydhx;
        uyy        = (two*u - x[V(row-gxm)] - x[V(row+gxm)])*hxdhy;
        f[V(row)]  = uxx + uyy + p5*(x[Omega(row+1)]-x[Omega(row-1)])*hy;

	/* Omega */
        u          = x[Omega(row)];
        uxx        = (two*u - x[Omega(row-1)] - x[Omega(row+1)])*hydhx;
        uyy        = (two*u - x[Omega(row-gxm)] - x[Omega(row+gxm)])*hxdhy;
	f[Omega(row)] = uxx + uyy + 
			( vxp*(u - x[Omega(row-1)]) +
			  vxm*(x[Omega(row+1)] - u) ) * hy +
			( vyp*(u - x[Omega(row-gxm)]) +
			  vym*(x[Omega(row+gxm)] - u) ) * hx -
			p5 * grashof * (x[Temp(row+1)] - x[Temp(row-1)]) * hy;

        /* Temperature */
        u             = x[Temp(row)];
        uxx           = (two*u - x[Temp(row-1)] - x[Temp(row+1)])*hydhx;
        uyy           = (two*u - x[Temp(row-gxm)] - x[Temp(row+gxm)])*hxdhy;
	f[Temp(row)] =  uxx + uyy  + prandtl * (
			( vxp*(u - x[Temp(row-1)]) +
			  vxm*(x[Temp(row+1)] - u) ) * hy +
		        ( vyp*(u - x[Temp(row-gxm)]) +
		       	  vym*(x[Temp(row+gxm)] - u) ) * hx );
    }
  }

  /*
     Restore vectors
  */
  ierr = VecRestoreArray(localX,&x); CHKERRQ(ierr);
  ierr = VecRestoreArray(localF,&f); CHKERRQ(ierr);

  /*
     Insert values into global vector
  */
  ierr = DALocalToGlobal(user->da,localF,INSERT_VALUES,F); CHKERRQ(ierr);

  /*
     Flop count (multiply-adds are counted as 2 operations)
  */
  PLogFlops(84*ym*xm);

  return 0; 
} 
