      subroutine dgl2fg(nx,ny,x,f,fgrad,task,w,vornum)
      character*(*) task
      integer nx, ny, vornum
      double precision f
      double precision x(4*nx*ny), fgrad(4*nx*ny), w(4*(nx+1)*(ny+1))
!     **********
!
!     Subroutine dgl2fg
!
!     This subroutine computes the function and gradient of the
!     Ginzburg-Landau (2-dimensional) superconductivity problem.
!
!     The subroutine statement is
!
!       subroutine dgl2fg(nx,ny,x,f,fgrad,task,w,vornum)
!
!     where
!
!       nx is an integer variable.
!         On entry nx is the number of grid points in the first
!            coordinate direction.
!         On exit nx is unchanged.
!
!       ny is an integer variable.
!         On entry ny is the number of grid points in the second
!            coordinate direction.
!         On exit ny is unchanged.
!
!       x is a double precision array of dimension 4*nx*ny.
!         On entry x specifies the vector x if task = 'F', 'G', or 'FG'.
!            Otherwise x need not be specified.
!         On exit x is unchanged if task = 'F', 'G', or 'FG'. Otherwise
!            x is set according to task.
!
!       f is a double precision variable.
!         On entry f need not be specified.
!         On exit f is set to the function evaluated at x if task = 'F'
!            or 'FG'.
!
!       fgrad is a double precision array of dimension 4*nx*ny.
!         On entry fgrad need not be specified.
!         On exit fgrad contains the gradient evaluated at x if
!            task = 'G' or 'FG'.
!
!       task is a character variable.
!         On entry task specifies the action of the subroutine:
!
!            task               action
!            ----               ------
!             'F'     Evaluate the function at x.
!             'G'     Evaluate the gradient at x.
!             'FG'    Evaluate the function and the gradient at x.
!             'XS'    Set x to the standard starting point xs.
!
!         On exit task is unchanged.
!
!       w is a double precision work array of dimension 4*(nx+1)(ny+1).
!
!       vornum is an integer variable.
!         On entry vornum specifies the number of vortices.
!         On exit vornum is unchanged.
!
!     Subprograms called
!
!       MINPACK-supplied   ...   dgl2fc
!
!     MINPACK-2 Project. November 1993.
!     Argonne National Laboratory and University of Minnesota.
!     Brett M. Averick, Paul L. Plassmann, and Stephen J. Wright.
!
!     **********
      double precision zero
      parameter (zero=0.0d0)

      integer ctr, i, itemp, j, k

      external dgl2fc

      itemp = (nx+1)*(ny+1)

!     Pack work array.

      ctr = 1
      do 20 j = 1, ny
         do 10 i = 1, nx
            k = (j-1)*nx + i
            w(ctr) = x(k)
            w(itemp+ctr) = x(nx*ny+k)
            w(2*itemp+ctr) = x(2*nx*ny+k)
            w(3*itemp+ctr) = x(3*nx*ny+k)
            ctr = ctr + 1
   10    continue
         w(ctr) = zero
         w(itemp+ctr) = zero
         w(2*itemp+ctr) = zero
         w(3*itemp+ctr) = zero
         ctr = ctr + 1
   20 continue

      call dgl2fc(nx,ny,w(1),w(itemp+1),w(2*itemp+1),w(3*itemp+1),f,    &
     &            fgrad(1),fgrad(nx*ny+1),fgrad(2*nx*ny+1),             &
     &            fgrad(3*nx*ny+1),task,vornum)

!     Unpack work array

      ctr = 1
      do 40 j = 1, ny
         do 30 i = 1, nx
            k = (j-1)*nx + i
            x(k) = w(ctr)
            x(nx*ny+k) = w(itemp+ctr)
            x(2*nx*ny+k) = w(2*itemp+ctr)
            x(3*nx*ny+k) = w(3*itemp+ctr)
            ctr = ctr + 1
   30    continue
         ctr = ctr + 1
   40 continue

      end

      subroutine dgl2fc(nx,ny,x,y,vpotx,vpoty,f,gradx,grady,gradax,     &
     &                 graday,task,vornum)
      character*(*) task
      integer nx, ny, vornum
      double precision x(nx+1,ny+1), y(nx+1,ny+1), vpotx(nx+1,ny+1)
      double precision vpoty(nx+1,ny+1), gradx(nx,ny), grady(nx,ny)
      double precision gradax(nx,ny), graday(nx,ny)
!     **********
!
!     Subroutine dgl2fc
!
!     This subroutine computes the function and gradient of the
!     Ginzburg-Landau (2-dimensional) superconductivity problem.
!
!     The subroutine statement is
!
!       subroutine dgl2fc(nx,ny,x,y,vpotx,vpoty,f,                       &
!    &                  gradx,grady,gradax,graday,task,vornum)
!
!     where
!
!       nx is an integer variable.
!         On entry nx is the number of grid points in the first
!            coordinate direction.
!         On exit nx is unchanged.
!
!       ny is an integer variable.
!         On entry ny is the number of grid points in the second
!            coordinate direction.
!         On exit ny is unchanged.
!
!       x is a double precision array of dimension nx*ny.
!         On entry x specifies the real part of the order parameter
!            if task = 'F', 'G', or 'FG'.
!            Otherwise x need not be specified.
!         On exit x is unchanged if task = 'F', 'G', or 'FG'. Otherwise
!            x is set according to task.
!
!       y is a double precision array of dimension nx*ny.
!         On entry y specifies the imaginary part of the order parameter
!            if task = 'F', 'G', or 'FG'.
!            Otherwise y need not be specified.
!         On exit y is unchanged if task = 'F', 'G', or 'FG'. Otherwise
!            y is set according to task.
!
!       vpotx is a double precision array of dimension nx*ny.
!         On entry vpotx specifies the x component of the vector
!            potential if task = 'F', 'G', or 'FG'.
!            Otherwise vpotx need not be specified.
!         On exit vpotx is unchanged if task = 'F', 'G', or 'FG'.
!            Otherwise vpotx is set according to task.
!
!       vpoty is a double precision array of dimension nx*ny.
!         On entry vpoty specifies the y component of the vector
!            potential if task = 'F', 'G', or 'FG'.
!            Otherwise vpoty need not be specified.
!         On exit vpoty is unchanged if task = 'F', 'G', or 'FG'.
!            Otherwise vpoty is set according to task.
!
!       f is a double precision variable.
!         On entry f need not be specified.
!         On exit f is set to the function evaluated at x if task = 'F'
!            or 'FG'.
!
!       gradx is a double precision array of dimension nx*ny.
!         On entry gradx need not be specified.
!         On exit gradx contains the gradient with respect to x
!            of f evaluated at (x,y,vpotx,vpoty) if task = 'G' or 'FG'.
!
!       grady is a double precision array of dimension nx*ny.
!         On entry grady need not be specified.
!         On exit grady contains the gradient with respect to y
!            of f evaluated at (x,y,vpotx,vpoty) if task = 'G' or 'FG'.
!            if task = 'G' or 'FG'.
!
!       gradax is a double precision array of dimension nx*ny.
!         On entry gradax need not be specified.
!         On exit gradax contains the gradient with respect to vpotx
!            of f evaluated at (x,y,vpotx,vpoty) if task = 'G' or 'FG'.
!            if task = 'G' or 'FG'.
!
!       graday is a double precision array of dimension nx*ny.
!         On entry graday need not be specified.
!         On exit graday contains the gradient with respect to vpoty
!            of f evaluated at (x,y,vpotx,vpoty) if task = 'G' or 'FG'.
!            if task = 'G' or 'FG'.
!
!       task is a character variable.
!         On entry task specifies the action of the subroutine:
!
!            task               action
!            ----               ------
!             'F'     Evaluate the function at (x,y,vpotx,vpoty).
!             'G'     Evaluate the gradient at (x,y,vpotx,vpoty).
!             'FG'    Evaluate the function and the gradient at
!                         (x,y,vpotx,vpoty).
!             'XS'    Set (x,y,vpotx,vpoty) to the standard starting
!                         point xs.
!
!         On exit task is unchanged.
!
!       vornum is an integer variable.
!         On entry vornum is the number of vortices.
!         On exit vornum is unchanged.
!
!     MINPACK-2 Project. November 1993.
!     Argonne National Laboratory and University of Minnesota.
!     Brett M. Averick, Paul L. Plassmann, and Stephen J. Wright.
!
!     **********
      double precision five, four, one, three, two, zero
      parameter (zero=0.0d0,one=1.0d0,two=2.0d0,three=3.0d0,four=4.0d0, &
     &          five=5.0d0)

      integer i, j
      double precision arg, bave, cfac, delsq, f, fcond, ffield, fkin
      double precision fkinx1, fkinx2, fkiny1, fkiny2, hx, hy, pi, sfac
      double precision sqn, sqrtv, tkappa, x1, x2, xpt, xy, ypt

!     Initialize.

      tkappa = five
      hx = sqrt(vornum/two)*three/dble(nx)
      hy = sqrt(vornum/two)*three*sqrt(three)/dble(ny)
      sqn = dble(nx*ny)
      pi = four*atan(one)
      bave = two*pi*vornum*tkappa/(sqn*hx*hy)
      sqrtv = sqrt(dble(vornum))*pi

      if (task .eq. 'XS') then

!        Initial Order Parameter.

         do 20 j = 1, ny + 1
            ypt = (dble(j)-one)*hy
            do 10 i = 1, nx + 1
               xpt = (dble(i)-one)*hx
               x(i,j) = one - (sin(sqrtv*xpt/(two*three))*              &
     &                  sin(sqrtv*ypt/(two*sqrt(three)*three)))**2
               y(i,j) = zero
   10       continue
   20    continue

!        Initial Vector Potential.

         do 40 j = 1, ny + 1
            do 30 i = 1, nx + 1
               xpt = (dble(i)-one)*hx
               vpotx(i,j) = zero
               vpoty(i,j) = bave*xpt/tkappa
   30       continue
   40    continue

         return

      end if

!     Enforce vortex constraint and boundary conditions.

!     Right face for order parameter and vector potential.

      do 50 j = 1, ny + 1
         arg = two*pi*vornum*(dble(j)-one)/dble(ny)
         x(nx+1,j) = x(1,j)*cos(arg) - y(1,j)*sin(arg)
         y(nx+1,j) = x(1,j)*sin(arg) + y(1,j)*cos(arg)
         vpotx(nx+1,j) = vpotx(1,j)
         vpoty(nx+1,j) = vpoty(1,j) + two*pi*vornum/(dble(ny)*hy)
   50 continue

!     Top face for order parameter and vector potential.

      do 60 i = 1, nx + 1
         x(i,ny+1) = x(i,1)
         y(i,ny+1) = y(i,1)
         vpotx(i,ny+1) = vpotx(i,1)
         vpoty(i,ny+1) = vpoty(i,1)
   60 continue

      if (task .eq. 'F' .or. task .eq. 'FG') then

!        Compute the Condensation Energy Density

         fcond = zero
         do 80 i = 1, nx
            do 70 j = 1, ny
               delsq = x(i,j)**2 + y(i,j)**2
               fcond = fcond - delsq + (delsq**2)/two
   70       continue
   80    continue
         fcond = fcond/sqn

!        Compute the Kinetic Energy Density.

         fkin = zero
         do 100 i = 1, nx
            do 90 j = 1, ny
               x1 = x(i+1,j) - x(i,j)*cos(hx*vpotx(i,j)) +              &
     &              y(i,j)*sin(hx*vpotx(i,j))
               x2 = y(i+1,j) - y(i,j)*cos(hx*vpotx(i,j)) -              &
     &              x(i,j)*sin(hx*vpotx(i,j))
               fkin = fkin + (x1**2+x2**2)/(hx**2)
               x1 = x(i,j+1) - x(i,j)*cos(hy*vpoty(i,j)) +              &
     &              y(i,j)*sin(hy*vpoty(i,j))
               x2 = y(i,j+1) - y(i,j)*cos(hy*vpoty(i,j)) -              &
     &              x(i,j)*sin(hy*vpoty(i,j))
               fkin = fkin + (x1**2+x2**2)/(hy**2)
   90       continue
  100    continue
         fkin = fkin/sqn

!        Compute the Magnetic Field Energy Density.

         ffield = zero
         do 120 i = 1, nx
            do 110 j = 1, ny
               xy = (vpoty(i+1,j)-vpoty(i,j))/hx -                      &
     &              (vpotx(i,j+1)-vpotx(i,j))/hy
               ffield = ffield + xy**2
  110       continue
  120    continue
         ffield = ffield*(tkappa**2)/sqn
         f = fcond + fkin + ffield
      end if

      if (task .eq. 'G' .or. task .eq. 'FG') then

         do 140 j = 1, ny
            do 130 i = 1, nx
               gradx(i,j) = x(i,j)*(-one+x(i,j)**2+y(i,j)**2)
               gradx(i,j) = gradx(i,j)*two/sqn
               grady(i,j) = y(i,j)*(-one+x(i,j)**2+y(i,j)**2)
               grady(i,j) = grady(i,j)*two/sqn
               gradax(i,j) = zero
               graday(i,j) = zero
  130       continue
  140    continue

!        Kinetic Energy Part, Interior Points

         do 160 i = 2, nx
            do 150 j = 2, ny
               fkinx1 = (two/(hx*hx*sqn))*(x(i+1,j)-                    &
     &                  x(i,j)*cos(hx*vpotx(i,j))+                      &
     &                  y(i,j)*sin(hx*vpotx(i,j)))
               fkinx2 = (two/(hx*hx*sqn))*(y(i+1,j)-                    &
     &                  y(i,j)*cos(hx*vpotx(i,j))-                      &
     &                  x(i,j)*sin(hx*vpotx(i,j)))
               fkiny1 = (two/(hy*hy*sqn))*(x(i,j+1)-                    &
     &                  x(i,j)*cos(hy*vpoty(i,j))+                      &
     &                  y(i,j)*sin(hy*vpoty(i,j)))
               fkiny2 = (two/(hy*hy*sqn))*(y(i,j+1)-                    &
     &                  y(i,j)*cos(hy*vpoty(i,j))-                      &
     &                  x(i,j)*sin(hy*vpoty(i,j)))
               ffield = (vpotx(i,j)-vpotx(i,j+1))/hy +                  &
     &                  (vpoty(i+1,j)-vpoty(i,j))/hx
               ffield = (two*(tkappa**2)/sqn)*ffield
               gradx(i,j) = gradx(i,j) - cos(hx*vpotx(i,j))*fkinx1 -    &
     &                      sin(hx*vpotx(i,j))*fkinx2 -                 &
     &                      cos(hy*vpoty(i,j))*fkiny1 -                 &
     &                      sin(hy*vpoty(i,j))*fkiny2
               grady(i,j) = grady(i,j) + sin(hx*vpotx(i,j))*fkinx1 -    &
     &                      cos(hx*vpotx(i,j))*fkinx2 +                 &
     &                      sin(hy*vpoty(i,j))*fkiny1 -                 &
     &                      cos(hy*vpoty(i,j))*fkiny2
               gradax(i,j) = gradax(i,j) + ffield/hy +                  &
     &                       fkinx1*(hx*x(i,j)*sin(hx*vpotx(i,j))+      &
     &                       hx*y(i,j)*cos(hx*vpotx(i,j))) +            &
     &                       fkinx2*(hx*y(i,j)*sin(hx*vpotx(i,j))-      &
     &                       hx*x(i,j)*cos(hx*vpotx(i,j)))
               graday(i,j) = graday(i,j) - ffield/hx +                  &
     &                       fkiny1*(hy*x(i,j)*sin(hy*vpoty(i,j))+      &
     &                       hy*y(i,j)*cos(hy*vpoty(i,j))) +            &
     &                       fkiny2*(hy*y(i,j)*sin(hy*vpoty(i,j))-      &
     &                       hy*x(i,j)*cos(hy*vpoty(i,j)))
               fkinx1 = (two/(hx*hx*sqn))*(x(i,j)-                      &
     &                  x(i-1,j)*cos(hx*vpotx(i-1,j))+                  &
     &                  y(i-1,j)*sin(hx*vpotx(i-1,j)))
               fkinx2 = (two/(hx*hx*sqn))*(y(i,j)-                      &
     &                  y(i-1,j)*cos(hx*vpotx(i-1,j))-                  &
     &                  x(i-1,j)*sin(hx*vpotx(i-1,j)))
               fkiny1 = (two/(hy*hy*sqn))*(x(i,j)-                      &
     &                  x(i,j-1)*cos(hy*vpoty(i,j-1))+                  &
     &                  y(i,j-1)*sin(hy*vpoty(i,j-1)))
               fkiny2 = (two/(hy*hy*sqn))*(y(i,j)-                      &
     &                  y(i,j-1)*cos(hy*vpoty(i,j-1))-                  &
     &                  x(i,j-1)*sin(hy*vpoty(i,j-1)))
               gradx(i,j) = gradx(i,j) + fkinx1 + fkiny1
               grady(i,j) = grady(i,j) + fkinx2 + fkiny2
               ffield = (vpotx(i,j-1)-vpotx(i,j))/hy +                  &
     &                  (vpoty(i+1,j-1)-vpoty(i,j-1))/hx
               ffield = (two*(tkappa**2)/sqn)*ffield
               gradax(i,j) = gradax(i,j) - ffield/hy
               ffield = (vpotx(i-1,j)-vpotx(i-1,j+1))/hy +              &
     &                  (vpoty(i,j)-vpoty(i-1,j))/hx
               ffield = (two*(tkappa**2)/sqn)*ffield
               graday(i,j) = graday(i,j) + ffield/hx
  150       continue
  160    continue

!        Kinetic Energy Part, Boundary Points.

!        Bottom J = 1

         do 170 i = 2, nx
            fkinx1 = (two/(hx*hx*sqn))*(x(i+1,1)-                       &
     &               x(i,1)*cos(hx*vpotx(i,1))+                         &
     &               y(i,1)*sin(hx*vpotx(i,1)))
            fkinx2 = (two/(hx*hx*sqn))*(y(i+1,1)-                       &
     &               y(i,1)*cos(hx*vpotx(i,1))-                         &
     &               x(i,1)*sin(hx*vpotx(i,1)))
            fkiny1 = (two/(hy*hy*sqn))*(x(i,2)-                         &
     &               x(i,1)*cos(hy*vpoty(i,1))+                         &
     &               y(i,1)*sin(hy*vpoty(i,1)))
            fkiny2 = (two/(hy*hy*sqn))*(y(i,2)-                         &
     &               y(i,1)*cos(hy*vpoty(i,1))-                         &
     &               x(i,1)*sin(hy*vpoty(i,1)))
            ffield = (vpotx(i,1)-vpotx(i,2))/hy +                       &
     &               (vpoty(i+1,1)-vpoty(i,1))/hx
            ffield = (two*(tkappa**2)/sqn)*ffield
            gradx(i,1) = gradx(i,1) - cos(hx*vpotx(i,1))*fkinx1 -       &
     &                   sin(hx*vpotx(i,1))*fkinx2 -                    &
     &                   cos(hy*vpoty(i,1))*fkiny1 -                    &
     &                   sin(hy*vpoty(i,1))*fkiny2
            grady(i,1) = grady(i,1) + sin(hx*vpotx(i,1))*fkinx1 -       &
     &                   cos(hx*vpotx(i,1))*fkinx2 +                    &
     &                   sin(hy*vpoty(i,1))*fkiny1 -                    &
     &                   cos(hy*vpoty(i,1))*fkiny2
            gradax(i,1) = gradax(i,1) + ffield/hy +                     &
     &                    fkinx1*(hx*x(i,1)*sin(hx*vpotx(i,1))+         &
     &                    hx*y(i,1)*cos(hx*vpotx(i,1))) +               &
     &                    fkinx2*(hx*y(i,1)*sin(hx*vpotx(i,1))-         &
     &                    hx*x(i,1)*cos(hx*vpotx(i,1)))
            graday(i,1) = graday(i,1) - ffield/hx +                     &
     &                    fkiny1*(hy*x(i,1)*sin(hy*vpoty(i,1))+         &
     &                    hy*y(i,1)*cos(hy*vpoty(i,1))) +               &
     &                    fkiny2*(hy*y(i,1)*sin(hy*vpoty(i,1))-         &
     &                    hy*x(i,1)*cos(hy*vpoty(i,1)))
            fkinx1 = (two/(hx*hx*sqn))*(x(i,1)-                         &
     &               x(i-1,1)*cos(hx*vpotx(i-1,1))+                     &
     &               y(i-1,1)*sin(hx*vpotx(i-1,1)))
            fkinx2 = (two/(hx*hx*sqn))*(y(i,1)-                         &
     &               y(i-1,1)*cos(hx*vpotx(i-1,1))-                     &
     &               x(i-1,1)*sin(hx*vpotx(i-1,1)))
            fkiny1 = (two/(hy*hy*sqn))*(x(i,ny+1)-                      &
     &               x(i,ny)*cos(hy*vpoty(i,ny))+                       &
     &               y(i,ny)*sin(hy*vpoty(i,ny)))
            fkiny2 = (two/(hy*hy*sqn))*(y(i,ny+1)-                      &
     &               y(i,ny)*cos(hy*vpoty(i,ny))-                       &
     &               x(i,ny)*sin(hy*vpoty(i,ny)))
            gradx(i,1) = gradx(i,1) + fkinx1 + fkiny1
            grady(i,1) = grady(i,1) + fkinx2 + fkiny2
            ffield = (vpotx(i,ny)-vpotx(i,ny+1))/hy +                   &
     &               (vpoty(i+1,ny)-vpoty(i,ny))/hx
            ffield = (two*(tkappa**2)/sqn)*ffield
            gradax(i,1) = gradax(i,1) - ffield/hy
            ffield = (vpotx(i-1,1)-vpotx(i-1,2))/hy +                   &
     &               (vpoty(i,1)-vpoty(i-1,1))/hx
            ffield = (two*(tkappa**2)/sqn)*ffield
            graday(i,1) = graday(i,1) + ffield/hx
  170    continue

!        Left I = 1.

         do 180 j = 2, ny
            fkinx1 = (two/(hx*hx*sqn))*(x(2,j)-                         &
     &               x(1,j)*cos(hx*vpotx(1,j))+                         &
     &               y(1,j)*sin(hx*vpotx(1,j)))
            fkinx2 = (two/(hx*hx*sqn))*(y(2,j)-                         &
     &               y(1,j)*cos(hx*vpotx(1,j))-                         &
     &               x(1,j)*sin(hx*vpotx(1,j)))
            fkiny1 = (two/(hy*hy*sqn))*(x(1,j+1)-                       &
     &               x(1,j)*cos(hy*vpoty(1,j))+                         &
     &               y(1,j)*sin(hy*vpoty(1,j)))
            fkiny2 = (two/(hy*hy*sqn))*(y(1,j+1)-                       &
     &               y(1,j)*cos(hy*vpoty(1,j))-                         &
     &               x(1,j)*sin(hy*vpoty(1,j)))
            ffield = (vpotx(1,j)-vpotx(1,j+1))/hy +                     &
     &               (vpoty(2,j)-vpoty(1,j))/hx
            ffield = (two*(tkappa**2)/sqn)*ffield
            gradx(1,j) = gradx(1,j) - cos(hx*vpotx(1,j))*fkinx1 -       &
     &                   sin(hx*vpotx(1,j))*fkinx2 -                    &
     &                   cos(hy*vpoty(1,j))*fkiny1 -                    &
     &                   sin(hy*vpoty(1,j))*fkiny2
            grady(1,j) = grady(1,j) + sin(hx*vpotx(1,j))*fkinx1 -       &
     &                   cos(hx*vpotx(1,j))*fkinx2 +                    &
     &                   sin(hy*vpoty(1,j))*fkiny1 -                    &
     &                   cos(hy*vpoty(1,j))*fkiny2
            gradax(1,j) = gradax(1,j) + ffield/hy +                     &
     &                    fkinx1*(hx*x(1,j)*sin(hx*vpotx(1,j))+         &
     &                    hx*y(1,j)*cos(hx*vpotx(1,j))) +               &
     &                    fkinx2*(hx*y(1,j)*sin(hx*vpotx(1,j))-         &
     &                    hx*x(1,j)*cos(hx*vpotx(1,j)))
            graday(1,j) = graday(1,j) - ffield/hx +                     &
     &                    fkiny1*(hy*x(1,j)*sin(hy*vpoty(1,j))+         &
     &                    hy*y(1,j)*cos(hy*vpoty(1,j))) +               &
     &                    fkiny2*(hy*y(1,j)*sin(hy*vpoty(1,j))-         &
     &                    hy*x(1,j)*cos(hy*vpoty(1,j)))
            fkinx1 = (two/(hx*hx*sqn))*(x(nx+1,j)-                      &
     &               x(nx,j)*cos(hx*vpotx(nx,j))+                       &
     &               y(nx,j)*sin(hx*vpotx(nx,j)))
            fkinx2 = (two/(hx*hx*sqn))*(y(nx+1,j)-                      &
     &               y(nx,j)*cos(hx*vpotx(nx,j))-                       &
     &               x(nx,j)*sin(hx*vpotx(nx,j)))
            fkiny1 = (two/(hy*hy*sqn))*(x(1,j)-                         &
     &               x(1,j-1)*cos(hy*vpoty(1,j-1))+                     &
     &               y(1,j-1)*sin(hy*vpoty(1,j-1)))
            fkiny2 = (two/(hy*hy*sqn))*(y(1,j)-                         &
     &               y(1,j-1)*cos(hy*vpoty(1,j-1))-                     &
     &               x(1,j-1)*sin(hy*vpoty(1,j-1)))
            sfac = sin(two*pi*vornum*(j-one)/dble(ny))
            cfac = cos(two*pi*vornum*(j-one)/dble(ny))
            gradx(1,j) = gradx(1,j) + cfac*fkinx1 + sfac*fkinx2 + fkiny1
            grady(1,j) = grady(1,j) - sfac*fkinx1 + cfac*fkinx2 + fkiny2
            ffield = (vpotx(1,j-1)-vpotx(1,j))/hy +                     &
     &               (vpoty(2,j-1)-vpoty(1,j-1))/hx
            ffield = (two*(tkappa**2)/sqn)*ffield
            gradax(1,j) = gradax(1,j) - ffield/hy
            ffield = (vpotx(nx,j)-vpotx(nx,j+1))/hy +                   &
     &               (vpoty(nx+1,j)-vpoty(nx,j))/hx
            ffield = (two*(tkappa**2)/sqn)*ffield
            graday(1,j) = graday(1,j) + ffield/hx
  180    continue

!        Kinetic Energy Part, at origin (only needed in zero field).

         fkinx1 = (two/(hx*hx*sqn))*(x(2,1)-x(1,1)*cos(hx*vpotx(1,1))+  &
     &            y(1,1)*sin(hx*vpotx(1,1)))
         fkinx2 = (two/(hx*hx*sqn))*(y(2,1)-y(1,1)*cos(hx*vpotx(1,1))-  &
     &            x(1,1)*sin(hx*vpotx(1,1)))
         fkiny1 = (two/(hy*hy*sqn))*(x(1,2)-x(1,1)*cos(hy*vpoty(1,1))+  &
     &            y(1,1)*sin(hy*vpoty(1,1)))
         fkiny2 = (two/(hy*hy*sqn))*(y(1,2)-y(1,1)*cos(hy*vpoty(1,1))-  &
     &            x(1,1)*sin(hy*vpoty(1,1)))
         ffield = (vpotx(1,1)-vpotx(1,2))/hy +                          &
     &            (vpoty(2,1)-vpoty(1,1))/hx
         ffield = (two*(tkappa**2)/sqn)*ffield
         gradx(1,1) = gradx(1,1) - cos(hx*vpotx(1,1))*fkinx1 -          &
     &                sin(hx*vpotx(1,1))*fkinx2 -                       &
     &                cos(hy*vpoty(1,1))*fkiny1 -                       &
     &                sin(hy*vpoty(1,1))*fkiny2
         grady(1,1) = grady(1,1) + sin(hx*vpotx(1,1))*fkinx1 -          &
     &                cos(hx*vpotx(1,1))*fkinx2 +                       &
     &                sin(hy*vpoty(1,1))*fkiny1 -                       &
     &                cos(hy*vpoty(1,1))*fkiny2
         gradax(1,1) = gradax(1,1) + ffield/hy +                        &
     &                 fkinx1*(hx*x(1,1)*sin(hx*vpotx(1,1))+            &
     &                 hx*y(1,1)*cos(hx*vpotx(1,1))) +                  &
     &                 fkinx2*(hx*y(1,1)*sin(hx*vpotx(1,1))-            &
     &                 hx*x(1,1)*cos(hx*vpotx(1,1)))
         graday(1,1) = graday(1,1) - ffield/hx +                        &
     &                 fkiny1*(hy*x(1,1)*sin(hy*vpoty(1,1))+            &
     &                 hy*y(1,1)*cos(hy*vpoty(1,1))) +                  &
     &                 fkiny2*(hy*y(1,1)*sin(hy*vpoty(1,1))-            &
     &                 hy*x(1,1)*cos(hy*vpoty(1,1)))
         fkinx1 = (two/(hx*hx*sqn))*(x(nx+1,1)-                         &
     &            x(nx,1)*cos(hx*vpotx(nx,1))+                          &
     &            y(nx,1)*sin(hx*vpotx(nx,1)))
         fkinx2 = (two/(hx*hx*sqn))*(y(nx+1,1)-                         &
     &            y(nx,1)*cos(hx*vpotx(nx,1))-                          &
     &            x(nx,1)*sin(hx*vpotx(nx,1)))
         fkiny1 = (two/(hy*hy*sqn))*(x(1,ny+1)-                         &
     &            x(1,ny)*cos(hy*vpoty(1,ny))+                          &
     &            y(1,ny)*sin(hy*vpoty(1,ny)))
         fkiny2 = (two/(hy*hy*sqn))*(y(1,ny+1)-                         &
     &            y(1,ny)*cos(hy*vpoty(1,ny))-                          &
     &            x(1,ny)*sin(hy*vpoty(1,ny)))
         gradx(1,1) = gradx(1,1) + fkinx1 + fkiny1
         grady(1,1) = grady(1,1) + fkinx2 + fkiny2
         ffield = (vpotx(1,ny)-vpotx(1,ny+1))/hy +                      &
     &            (vpoty(2,ny)-vpoty(1,ny))/hx
         ffield = (two*(tkappa**2)/sqn)*ffield
         gradax(1,1) = gradax(1,1) - ffield/hy
         ffield = (vpotx(nx,1)-vpotx(nx,2))/hy +                        &
     &            (vpoty(nx+1,1)-vpoty(nx,1))/hx
         ffield = (two*(tkappa**2)/sqn)*ffield
         graday(1,1) = graday(1,1) + ffield/hx
      end if

      end
