      subroutine deptfg(nx,ny,x,f,fgrad,task,c)
      character*(*) task
      integer nx, ny
      double precision f, c
      double precision x(nx*ny), fgrad(nx*ny)
!     **********
!
!     Subroutine deptfg
!
!     This subroutine computes the function and gradient of the
!     elastic-plastic torsion problem.
!
!     The subroutine statement is
!
!       subroutine deptfg(nx,ny,x,f,fgrad,task,c)
!
!     where
!
!       nx is an integer variable.
!         On entry nx is the number of grid points in the first
!            coordinate direction.
!         On exit nx is unchanged.
!
!       ny is an integer variable.
!         On entry ny is the number of grid points in the second
!            coordinate direction.
!         On exit ny is unchanged.
!
!       x is a double precision array of dimension nx*ny.
!         On entry x specifies the vector x if task = 'F', 'G', or 'FG'.
!            Otherwise x need not be specified.
!         On exit x is unchanged if task = 'F', 'G', or 'FG'. Otherwise
!            x is set according to task.
!
!       f is a double precision variable.
!         On entry f need not be specified.
!         On exit f is set to the function evaluated at x if task = 'F'
!            or 'FG'.
!
!       fgrad is a double precision array of dimension nx*ny.
!         On entry fgrad need not be specified.
!         On exit fgrad contains the gradient evaluated at x if
!            task = 'G' or 'FG'.
!
!       task is a character variable.
!         On entry task specifies the action of the subroutine:
!
!            task               action
!            ----               ------
!             'F'     Evaluate the function at x.
!             'G'     Evaluate the gradient at x.
!             'FG'    Evaluate the function and the gradient at x.
!             'XS'    Set x to the standard starting point xs.
!             'XL'    Set x to the lower bound xl.
!             'XU'    Set x to the upper bound xu.
!
!         On exit task is unchanged.
!
!       c is a double precision variable.
!         On entry c is the angle of twist per unit length.
!         On exit c is unchanged.
!
!     MINPACK-2 Project. November 1993.
!     Argonne National Laboratory and University of Minnesota.
!     Brett M. Averick and Jorge J. More'.
!
!     **********
      double precision one, p5, three, zero
      parameter (zero=0.0d0,p5=0.5d0,one=1.0d0,three=3.0d0)

      logical feval, geval
      integer i, j, k
      double precision area, cdiv3, dvdx, dvdy, flin, fquad, hx, hy,    &
     &                 temp, temp1, v, vb, vl, vr, vt

      hx = one/dble(nx+1)
      hy = one/dble(ny+1)
      area = p5*hx*hy
      cdiv3 = c/three

!     Compute a lower bound for x if task = 'XL' or an upper bound if
!     task = 'XU'.

      if (task .eq. 'XL' .or. task .eq. 'XU') then
         if (task .eq. 'XL') temp1 = -one
         if (task .eq. 'XU') temp1 = one
         do 20 j = 1, ny
            temp = dble(min(j,ny-j+1))*hy
            do 10 i = 1, nx
               k = nx*(j-1) + i
               x(k) = sign(min(dble(min(i,nx-i+1))*hx,temp),temp1)
   10       continue
   20    continue

         return

      end if

!     Compute the standard starting point if task = 'XS'.

      if (task .eq. 'XS') then
         do 40 j = 1, ny
            temp = dble(min(j,ny-j+1))*hy
            do 30 i = 1, nx
               k = nx*(j-1) + i
               x(k) = min(dble(min(i,nx-i+1))*hx,temp)
   30       continue
   40    continue

         return

      end if

      if (task .eq. 'F' .or. task .eq. 'FG') then
         feval = .true.
      else
         feval = .false.
      end if
      if (task .eq. 'G' .or. task .eq. 'FG') then
         geval = .true.
      else
         geval = .false.
      end if

!     Evaluate the function if task = 'F', the gradient if task = 'G',
!     or both if task = 'FG'.

      if (feval) then
         fquad = zero
         flin = zero
      end if
      if (geval) then
         do 50 k = 1, nx*ny
            fgrad(k) = zero
   50    continue
      end if

!     Computation of the function and the gradient over the lower
!     triangular elements.

      do 70 j = 0, ny
         do 60 i = 0, nx
            k = nx*(j-1) + i
            v = zero
            vr = zero
            vt = zero
            if (i .ge. 1 .and. j .ge. 1) v = x(k)
            if (i .lt. nx .and. j .gt. 0) vr = x(k+1)
            if (i .gt. 0 .and. j .lt. ny) vt = x(k+nx)
            dvdx = (vr-v)/hx
            dvdy = (vt-v)/hy
            if (feval) then
               fquad = fquad + dvdx**2 + dvdy**2
               flin = flin - cdiv3*(v+vr+vt)
            end if
            if (geval) then
               if (i .ne. 0 .and. j .ne. 0)                             &
     &             fgrad(k) = fgrad(k) - dvdx/hx - dvdy/hy - cdiv3
               if (i .ne. nx .and. j .ne. 0)                            &
     &             fgrad(k+1) = fgrad(k+1) + dvdx/hx - cdiv3
               if (i .ne. 0 .and. j .ne. ny)                            &
     &             fgrad(k+nx) = fgrad(k+nx) + dvdy/hy - cdiv3
            end if
   60    continue
   70 continue

!     Computation of the function and the gradient over the upper
!     triangular elements.

      do 90 j = 1, ny + 1
         do 80 i = 1, nx + 1
            k = nx*(j-1) + i
            vb = zero
            vl = zero
            v = zero
            if (i .le. nx .and. j .gt. 1) vb = x(k-nx)
            if (i .gt. 1 .and. j .le. ny) vl = x(k-1)
            if (i .le. nx .and. j .le. ny) v = x(k)
            dvdx = (v-vl)/hx
            dvdy = (v-vb)/hy
            if (feval) then
               fquad = fquad + dvdx**2 + dvdy**2
               flin = flin - cdiv3*(vb+vl+v)
            end if
            if (geval) then
               if (i .ne. nx+1 .and. j .ne. 1)                          &
     &             fgrad(k-nx) = fgrad(k-nx) - dvdy/hy - cdiv3
               if (i .ne. 1 .and. j .ne. ny+1)                          &
     &             fgrad(k-1) = fgrad(k-1) - dvdx/hx - cdiv3
               if (i .ne. nx+1 .and. j .ne. ny+1)                       &
     &             fgrad(k) = fgrad(k) + dvdx/hx + dvdy/hy - cdiv3
            end if
   80    continue
   90 continue

!     Scale the result.

      if (feval) f = area*(p5*fquad+flin)
      if (geval) then
         do 100 k = 1, nx*ny
            fgrad(k) = area*fgrad(k)
  100    continue
      end if

      end
