!
!    "$Id: ex15f.F,v 1.9 1998/04/27 20:01:41 balay Exp $";
!
!   Solves a linear system in parallel with SLES.  Also indicates
!   use of a user-provided preconditioner.  Input parameters include:
!      -user_defined_pc : Activate a user-defined preconditioner
!
!  Program usage: mpirun ex15f [-help] [all PETSc options]
!
!/*T
!   Concepts: SLES^Solving a system of linear equations (basic parallel example);
!   Concepts: PC^Setting a user-defined shell preconditioner
!   Routines: SLESCreate(); SLESSetOperators(); SLESSetFromOptions();
!   Routines: SLESSolve(); SLESGetKSP(); SLESGetPC(); KSPSetTolerances(); 
!   Routines: PCSetType(); PCShellSetApply(); 
!   Processors: n
!T*/
!
!  ------------------------------------------------------------------------- 

      program main
      implicit none

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!                    Include files
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!
!     petsc.h  - base PETSc routines      vec.h - vectors
!     sys.h    - system routines          mat.h - matrices
!     ksp.h    - Krylov subspace methods  pc.h  - preconditioners
!     sles.h   - SLES interface

#include "include/finclude/petsc.h"
#include "include/finclude/vec.h"
#include "include/finclude/mat.h"
#include "include/finclude/pc.h"
#include "include/finclude/ksp.h"
#include "include/finclude/sles.h"

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!                   Variable declarations
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!
!  Variables:
!     sles     - linear solver context
!     ksp      - Krylov subspace method context
!     pc       - preconditioner context
!     x, b, u  - approx solution, right-hand-side, exact solution vectors
!     A        - matrix that defines linear system
!     its      - iterations for convergence
!     norm     - norm of solution error

      Vec              x, b, u
      Mat              A      
      SLES             sles   
      PC               pc     
      KSP              ksp   
      Scalar           v, one, neg_one
      double precision norm, tol
      integer          i, j, II, JJ, Istart, Iend, ierr, m, n
      integer          user_defined_pc, its, flg, rank

!  Note: Any user-defined Fortran routines MUST be declared as external.

      external SampleShellPCSetUp, SampleShellPCApply

!  Common block to store data for user-provided preconditioner 
      common /myshellpc/ diag
      Vec    diag

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!                 Beginning of program
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

      call PetscInitialize(PETSC_NULL_CHARACTER,ierr)
      one     = 1.0
      neg_one = -1.0
      m       = 8
      n       = 7
      call OptionsGetInt(PETSC_NULL_CHARACTER,'-m',m,flg,ierr)
      call OptionsGetInt(PETSC_NULL_CHARACTER,'-n',n,flg,ierr)
      call MPI_Comm_rank(PETSC_COMM_WORLD,rank,ierr)

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
!      Compute the matrix and right-hand-side vector that define
!      the linear system, Ax = b.
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

!  Create parallel matrix, specifying only its global dimensions.
!  When using MatCreate(), the matrix format can be specified at
!  runtime. Also, the parallel partitioning of the matrix is
!  determined by PETSc at runtime.

      call MatCreate(PETSC_COMM_WORLD,m*n,m*n,A,ierr)

!  Currently, all PETSc parallel matrix formats are partitioned by
!  contiguous chunks of rows across the processors.  Determine which
!  rows of the matrix are locally owned. 

      call MatGetOwnershipRange(A,Istart,Iend,ierr)

!  Set matrix elements for the 2-D, five-point stencil in parallel.
!   - Each processor needs to insert only elements that it owns
!     locally (but any non-local elements will be sent to the
!     appropriate processor during matrix assembly). 
!   - Always specify global row and columns of matrix entries.
!   - Note that MatSetValues() uses 0-based row and column numbers
!     in Fortran as well as in C.

      do 10, II=Istart,Iend-1
        v = -1.0
        i = II/n
        j = II - i*n  
        if ( i.gt.0 ) then
          JJ = II - n
          call MatSetValues(A,1,II,1,JJ,v,ADD_VALUES,ierr)
        endif
        if ( i.lt.m-1 ) then
          JJ = II + n
          call MatSetValues(A,1,II,1,JJ,v,ADD_VALUES,ierr)
        endif
        if ( j.gt.0 ) then
          JJ = II - 1
          call MatSetValues(A,1,II,1,JJ,v,ADD_VALUES,ierr)
        endif
        if ( j.lt.n-1 ) then
          JJ = II + 1
          call MatSetValues(A,1,II,1,JJ,v,ADD_VALUES,ierr)
        endif
        v = 4.0
        call  MatSetValues(A,1,II,1,II,v,ADD_VALUES,ierr)
 10   continue

!  Assemble matrix, using the 2-step process:
!       MatAssemblyBegin(), MatAssemblyEnd()
!  Computations can be done while messages are in transition,
!  by placing code between these two statements.

      call MatAssemblyBegin(A,MAT_FINAL_ASSEMBLY,ierr)
      call MatAssemblyEnd(A,MAT_FINAL_ASSEMBLY,ierr)

!  Create parallel vectors.
!   - Here, the parallel partitioning of the vector is determined by
!     PETSc at runtime.  We could also specify the local dimensions
!     if desired -- or use the more general routine VecCreate().
!   - When solving a linear system, the vectors and matrices MUST
!     be partitioned accordingly.  PETSc automatically generates
!     appropriately partitioned matrices and vectors when MatCreate()
!     and VecCreate() are used with the same communicator. 
!   - Note: We form 1 vector from scratch and then duplicate as needed.

      call VecCreateMPI(PETSC_COMM_WORLD,PETSC_DECIDE,m*n,u,ierr)
      call VecDuplicate(u,b,ierr)
      call VecDuplicate(b,x,ierr)

!  Set exact solution; then compute right-hand-side vector.

      call VecSet(one,u,ierr)
      call MatMult(A,u,b,ierr)

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
!         Create the linear solver and set various options
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

!  Create linear solver context

      call SLESCreate(PETSC_COMM_WORLD,sles,ierr)

!  Set operators. Here the matrix that defines the linear system
!  also serves as the preconditioning matrix.

      call SLESSetOperators(sles,A,A,DIFFERENT_NONZERO_PATTERN,ierr)

!  Set linear solver defaults for this problem (optional).
!   - By extracting the KSP and PC contexts from the SLES context,
!     we can then directly directly call any KSP and PC routines
!     to set various options.

      call SLESGetKSP(sles,ksp,ierr)
      call SLESGetPC(sles,pc,ierr)
      tol = 1.e-7
      call KSPSetTolerances(ksp,tol,PETSC_DEFAULT_DOUBLE_PRECISION,     &
     &     PETSC_DEFAULT_DOUBLE_PRECISION,PETSC_DEFAULT_INTEGER,ierr)

!
!  Set a user-defined shell preconditioner if desired
!
      call OptionsHasName(PETSC_NULL_CHARACTER,'-user_defined_pc',      &
     &                    user_defined_pc,ierr)

      if (user_defined_pc .eq. 1) then

!  (Required) Indicate to PETSc that we are using a shell preconditioner 
         call PCSetType(pc,PCSHELL,ierr)

!  (Required) Set the user-defined routine for applying the preconditioner
         call PCShellSetApply(pc,SampleShellPCApply,PETSC_NULL_OBJECT,   &
     &        ierr)

!  (Optional) Do any setup required for the preconditioner
         call SampleShellPCSetUp(A,x,ierr)

      else 
         call PCSetType(pc,PCJACOBI,ierr)
      endif

!  Set runtime options, e.g.,
!      -ksp_type <type> -pc_type <type> -ksp_monitor -ksp_rtol <rtol>
!  These options will override those specified above as long as
!  SLESSetFromOptions() is called _after_ any other customization
!  routines.

      call SLESSetFromOptions(sles,ierr)

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
!                      Solve the linear system
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

      call SLESSolve(sles,b,x,its,ierr)

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
!                     Check solution and clean up
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

!  Check the error

      call VecAXPY(neg_one,u,x,ierr)
      call VecNorm(x,NORM_2,norm,ierr)
      if (rank .eq. 0) then
        if (norm .gt. 1.e-12) then
           write(6,100) norm, its
        else
           write(6,110) its
        endif
      endif
  100 format('Norm of error ',e10.4,' iterations ',i5)
  110 format('Norm of error < 1.e-12, iterations ',i5)

!  Free work space.  All PETSc objects should be destroyed when they
!  are no longer needed.

      call SLESDestroy(sles,ierr)
      call VecDestroy(u,ierr)
      call VecDestroy(x,ierr)
      call VecDestroy(b,ierr)
      call MatDestroy(A,ierr)
      if (user_defined_pc .eq. 1) then
         call VecDestroy(diag,ierr)
      endif


!  Always call PetscFinalize() before exiting a program. 

      call PetscFinalize(ierr)
      end

!/***********************************************************************/
!/*          Routines for a user-defined shell preconditioner           */
!/***********************************************************************/

!
!   SampleShellPCSetUp - This routine sets up a user-defined
!   preconditioner context.  
!
!   Input Parameters:
!   pmat  - preconditioner matrix
!   x     - vector
!
!   Output Parameter:
!   ierr  - error code (nonzero if error has been detected)
!
!   Notes:
!   In this example, we define the shell preconditioner to be Jacobi
!   method.  Thus, here we create a work vector for storing the reciprocal
!   of the diagonal of the preconditioner matrix; this vector is then
!   used within the routine SampleShellPCApply().
!
      subroutine SampleShellPCSetUp(pmat,x,ierr)

      implicit none

#include "include/finclude/petsc.h"
#include "include/finclude/vec.h"
#include "include/finclude/mat.h"

      Vec     x
      Mat     pmat
      integer ierr

!  Common block to store data for user-provided preconditioner 
      common /myshellpc/ diag
      Vec    diag

      call VecDuplicate(x,diag,ierr)
      call MatGetDiagonal(pmat,diag,ierr)
      call VecReciprocal(diag,ierr)

      end

! ------------------------------------------------------------------- 
!
!   SampleShellPCApply - This routine demonstrates the use of a
!   user-provided preconditioner.
!
!   Input Parameters:
!   dummy - optional user-defined context, not used here
!   x - input vector
!
!   Output Parameters:
!   y - preconditioned vector
!   ierr  - error code (nonzero if error has been detected)
!
!   Notes:
!   This code implements the Jacobi preconditioner, merely as an
!   example of working with a PCSHELL.  Note that the Jacobi method
!   is already provided within PETSc.
!
      subroutine SampleShellPCApply(dummy,x,y,ierr)

      implicit none

#include "include/finclude/petsc.h"
#include "include/finclude/vec.h"

      Vec     x, y
      integer dummy, ierr

!  Common block to store data for user-provided preconditioner 
      common /myshellpc/ diag
      Vec    diag

      call VecPointwiseMult(x,diag,y,ierr)

      end

