#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: mgtype.c,v 1.13 1998/04/27 04:03:39 curfman Exp $";
#endif

#include "sys.h"
#include "mat.h"       /*I "mat.h"  I*/

#undef __FUNC__  
#define __FUNC__ "MatGetTypeFromOptions"
/*@C
   MatGetTypeFromOptions - Determines from the options database what matrix
   format the user has specified.

   Input Parameter:
+  comm - the MPI communicator
.  type - the type of matrix desired, for example MATSEQAIJ, MATMPIAIJ
-  pre - optional string to prepend to the name

   Output Parameters:
.  set - flag indicating whether user set matrix type option.

   Basic Options Database Keys:
   These options return MATSEQxxx or MATMPIxxx,
   depending on the communicator, comm.
+    -mat_aij      - AIJ type
.    -mat_baij     - block AIJ type
.    -mat_dense    - dense type
-    -mat_bdiag    - block diagonal type

   More Options Database Keys:
+    -mat_seqaij   - MATSEQAIJ
.    -mat_mpiaij   - MATMPIAIJ
.    -mat_seqbaij  - MATSEQBAIJ
.    -mat_mpibaij  - MATMPIBAIJ
.    -mat_seqbdiag - MATSEQBDAIG
.    -mat_mpibdiag - MATMPIBDIAG
.    -mat_mpirowbs - MATMPIROWBS
.    -mat_seqdense - MATSEQDENSE
-    -mat_mpidense - MATMPIDENSE

   Notes:
   This routine is collective over all processes in the communicator, comm.  

   This routine is automatically called within MatCreate() and MatLoad().

.keywords: matrix, get, format, from, options

.seealso: MatCreate(), MatLoad()
@*/

int MatGetTypeFromOptions(MPI_Comm comm,char *pre,MatType *type,PetscTruth *set)
{
  int         size,flg1,flg2,flg3,flg4,flg5,flg8,flg9,flg10,flg12,flg13,ierr,flg11,flg14,flg15,flg16;
  static int  helpprinted = 0;
  char        p[64];

  PetscFunctionBegin;
  PetscValidIntPointer(type); PetscValidIntPointer(set);

  PetscStrcpy(p,"-");
  if (pre) PetscStrcat(p,pre);

  MPI_Comm_size(comm,&size);
  ierr = OptionsHasName(PETSC_NULL,"-help",&flg1); CHKERRQ(ierr);
  if (flg1 && !helpprinted) {
    (*PetscHelpPrintf)(comm,"Matrix format options:\n");
    (*PetscHelpPrintf)(comm,"  %smat_aij, %smat_seqaij, %smat_mpiaij\n",p,p,p);
    (*PetscHelpPrintf)(comm,"  %smat_baij, %smat_seqbaij, %smat_mpibaij\n",p,p,p);
    (*PetscHelpPrintf)(comm,"  %smat_dense, %smat_seqdense, %smat_mpidense\n",p,p,p);
    (*PetscHelpPrintf)(comm,"  %smat_mpirowbs, %smat_bdiag, %smat_seqbdiag, %smat_mpibdiag\n",p,p,p,p); 
    helpprinted = 1;
  }
  ierr = OptionsHasName(pre,"-mat_seqdense",&flg1); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_mpidense",&flg2); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_seqbdiag",&flg3); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_mpibdiag",&flg4); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_mpirowbs",&flg5); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_seqbaij",&flg11); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_mpibaij",&flg14); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_mpiaij",&flg8); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_seqaij",&flg9); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_aij",&flg10); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_bdiag",&flg12); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_dense",&flg13); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_baij",&flg15); CHKERRQ(ierr);
  ierr = OptionsHasName(pre,"-mat_adj",&flg16); CHKERRQ(ierr);
  if (flg1) {
    *type = MATSEQDENSE;
    *set = PETSC_TRUE;
  }
  else if (flg2) {
    *type = MATMPIDENSE;
    *set = PETSC_TRUE;
  }
  else if (flg3) {
    *type = MATSEQBDIAG;
    *set = PETSC_TRUE;
  }
  else if (flg4) {
    *type = MATMPIBDIAG;
    *set = PETSC_TRUE;
  }
  else if (flg5) {
    *type = MATMPIROWBS;
    *set = PETSC_TRUE;
  }
  else if (flg8) {
    *type = MATMPIAIJ;
    *set = PETSC_TRUE;
  }
  else if (flg9){
    *type = MATSEQAIJ;
    *set = PETSC_TRUE;
  }
  else if (flg10) {
    if (size == 1) *type = MATSEQAIJ;
    else *type = MATMPIAIJ;
    *set = PETSC_TRUE;
  }  
  else if (flg11) {
    *type = MATSEQBAIJ;
    *set = PETSC_TRUE;
  }
  else if (flg12) {
    if (size == 1) *type = MATSEQBDIAG;
    else *type = MATMPIBDIAG;
    *set = PETSC_TRUE;
  }  
  else if (flg13) {
    if (size == 1) *type = MATSEQDENSE;
    else *type = MATMPIDENSE;
    *set = PETSC_TRUE;
  } 
  else if (flg14) {
    *type = MATMPIBAIJ;
    *set = PETSC_TRUE;
  }  
  else if (flg15) {
    if (size == 1) *type = MATSEQBAIJ;
    else *type = MATMPIBAIJ;
    *set = PETSC_TRUE;
  }  
  else if (flg16) {
    if (size == 1) *type = MATSEQADJ;
    else *type = MATMPIADJ;
    *set = PETSC_TRUE;
  }  
  else {
    if (size == 1) *type = MATSEQAIJ;
    else *type = MATMPIAIJ;
    *set = PETSC_FALSE;
  }
  PetscFunctionReturn(0);
}
