#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: gvec.c,v 1.23 1998/04/27 14:21:04 curfman Exp $";
#endif
/*
  This file provides routines for grid vectors (vectors that are associated with grids,
  possibly with multiple degrees of freedom per node). 

  This component is new; changes in the user interface will be occuring in the
  near future!
*/

#include "src/gvec/gvecimpl.h"    /*I "gvec.h" I*/

#undef __FUNC__  
#define __FUNC__ "GVecGetGrid" /* ADIC Ignore */
/*@
   GVecGetGrid - Gets the grid from a grid vector.

   Collective on GridData

   Input Parameters:
.  v - the vector

   Output Parameters:
.  grid - the grid context

.keywords: discrete function, vector, get

.seealso: GridGetGridData(), GMatGetGrid()
@*/
int GVecGetGrid(GVec v,Grid *grid)
{
  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidPointer(grid);

  PetscObjectQuery((PetscObject)v,"grid",(PetscObject*)grid);
  PetscValidHeaderSpecific(*grid,GRID_COOKIE);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GMatGetGrid" /* ADIC Ignore */
/*@
   GMatGetGrid - Gets the grid from a matrix vector.

   Not Collective (but if GMat is parallel then Grid is parallel)
   
   Input Parameters:
.  v - the matrix

   Output Parameters:
.  grid - the associated grid object

.keywords: discrete operator, vector, get

.seealso: GridGetGridData(), GVecGetGrid()
@*/
int GMatGetGrid(GMat v,Grid *grid)
{
  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,MAT_COOKIE);
  PetscValidPointer(grid);

  PetscObjectQuery((PetscObject)v,"grid",(PetscObject*)grid);
  PetscValidHeaderSpecific(*grid,GRID_COOKIE);
  PetscFunctionReturn(0);
}


#undef __FUNC__  
#define __FUNC__ "GridCreateRestriction"
/*@
    GridCreateRestriction - Generates restriction matrix between two grids.

    Collective on Grids

    Input Parameters:
+   vf - grid 
-   vc - grid one is restricting too

    Output Parameters:
.   gmat - sparse matrix containing restriction

@*/
int GridCreateRestriction(Grid vf,Grid vc,GMat *gmat)
{
  int   ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(vf,GRID_COOKIE);
  PetscValidHeaderSpecific(vc,GRID_COOKIE);
  PetscValidPointer(gmat);

  if (!vf->setupcalled) {
    ierr = GridSetUp(vf); CHKERRQ(ierr);
  }
  if (!vc->setupcalled) {
    ierr = GridSetUp(vc); CHKERRQ(ierr);
  }
  if (!vf->ops->gridcreaterestriction) SETERRQ(PETSC_ERR_SUP,0,"");
  ierr = (*vf->ops->gridcreaterestriction)(vf,vc,gmat); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


#undef __FUNC__  
#define __FUNC__ "GVecGetComponentGVecs" /* ADIC Ignore */
/*@
   GVecGetComponentGVecs - Extracts the component vectors from a larger
   multicomponent vector.  

   Not Collective (but GVec * is parallel if GVec is parallel)

   Input Parameter:
+  v - the grid vector
-  comp - COMPONENT_1, COMPONENT_2, etc or COMPONENT_ALL

   Output Parameters:
.  vcomp - array of component vectors

.seealso: GVecCreateComponentGVecs(), GVecAssembleComponents()
@*/
int GVecGetComponentGVecs(GVec v,GVecComponents com,GVec *vcomp)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecgetcomponentgvecs)(v,com,vcomp); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecAssembleComponentGVecs"
/*@
   GVecAssembleComponentGVecs - Assembles the component vectors into a 
   multicomponent vector.

   Collective on GVec and GVec*

   Input Parameter:
+  v - the grid vector
-  comp - COMPONENT_1, COMPONENT_2, etc or COMPONENT_ALL

   Output Parameters:
.  vcomp - array of component vectors

.seealso: GVecCreateComponentGVecs(), GVecGetComponentGVecs()
@*/
int GVecAssembleComponentGVecs(GVec v,GVecComponents comp,GVec *vcomp)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecassemblecomponentgvecs)(v,comp,vcomp); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecCreateComponentGVecs"
/*@
   GVecCreateComponentGVecs - Creates vectors to hold the components
   from a larger multicomponent vector.  

   Collective on GVec

   Input Parameter:
+  v - the grid vector
-  com - COMPONENT_1, COMPONENT_2, etc or COMPONENT_ALL

   Output Parameters:
+  ncomp - number of component vectors
-  vcomp - array of component vectors

.seealso: GVecGetComponentGVecs(), GVecAssembleComponents()
@*/
int GVecCreateComponentGVecs(GVec v,GVecComponents com,int *ncomp,GVec **vcomp)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gveccreatecomponentgvecs)(v,com,ncomp,vcomp); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecView" /* ADIC Ignore */
/*@ 
   GVecView - Views a grid vector.

   Input Parameters:
+  v - the grid vector
-  viewer - an optional visualization context

   Collective on GVec unless Viewer is sequential

   Notes:
   GVecView() supports the same viewers as VecView().  The only difference
   is that for all multiprocessor cases, the output vector employs the natural
   ordering of the grid, so it many cases this corresponds to the ordering 
   that would have been used for the uniprocessor case.

   The available visualization contexts include
+     VIEWER_STDOUT_SELF - standard output (default)
-     VIEWER_STDOUT_WORLD - synchronized standard
         output where only the first processor opens
         the file.  All other processors send their 
         data to the first processor to print. 

   The user can open alternative vistualization contexts with
+    ViewerFileOpenASCII() - Outputs vector to a specified file
.    ViewerFileOpenBinary() - Outputs vector in binary to a
         specified file; corresponding input uses VecLoad()
.    ViewerDrawOpenX() - Outputs vector to an X window display
-    ViewerMatlabOpen() - Outputs vector to Matlab viewer

.keywords: view, visualize, output, print, write, draw

.seealso: VecView()
@*/
int GVecView(GVec v,Viewer viewer)
{  
  int        ierr;
  Grid       grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  if (!viewer) {viewer = VIEWER_STDOUT_SELF;}
  else PetscValidHeaderSpecific(viewer,VIEWER_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecview)(v,viewer);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGetLocalWorkGVec" /* ADIC Ignore */
/*@
   GVecGetLocalWorkGVec - Creates a new GVec to contain the local + ghost
   values portion of a GVec.

   Not Collective

   Input Parameter:
.  v - the grid vector

   Output Parameters:
.  lv - the local vector

.seealso: GVecLocalToGlobal(), GVecGlobalToLocal(), GVecRestoreLocalWorkGVec(),
          GVecGetWorkGVec(), GVecRestoreWorkGVec()
@*/
int GVecGetLocalWorkGVec(GVec v,GVec *lv)
{
  int  ierr;
  Grid grid;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidPointer(lv);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecgetlocalworkgvec)(v,lv); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecRestoreLocalWorkGVec" /* ADIC Ignore */
/*@
   GVecRestoreLocalWorkGVec - Restores local working grid vector.

   Not Collective

   Input Parameter:
.  v - the grid vector

   Output Parameters:
.  lv - the local vector

.seealso: GVecLocalToGlobal(), GVecGlobalToLocal(), GVecGetLocalWorkGVec(),
          GVecGetWorkGVec(), GVecRestoreWorkGVec()
@*/
int GVecRestoreLocalWorkGVec(GVec v,GVec *lv)
{
  int  ierr;
  Grid grid;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidPointer(lv);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecrestorelocalworkgvec)(v,lv); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGetWorkGVec" /* ADIC Ignore */
/*@
   GVecGetWorkGVec - Gets work gvec.

   Collective on GVec

   Input Parameter:
.  v - the grid vector

   Output Parameters:
.  lv - the local vector

.seealso: GVecLocalToGlobal(), GVecGlobalToLocal(), GVecRestoreWorkGVec(),
          GVecGetLocalWorkGVec(), GVecRestoreLocalWorkGVec()
@*/
int GVecGetWorkGVec(GVec v,GVec *lv)
{
  int ierr;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidPointer(lv);

  ierr = VecDuplicate(v,lv);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecRestoreWorkGVec" /* ADIC Ignore */
/*@
   GVecRestoreWorkGVec - Restores work gvec.

   Collective on GVec

   Input Parameter:
.  v - the grid vector

   Output Parameters:
.  lv - the local vector

.seealso: GVecLocalToGlobal(), GVecGlobalToLocal(), GVecGetWorkGVec(),
          GVecGetLocalWorkGVec(), GVecRestoreLocalWorkGVec()
@*/
int GVecRestoreWorkGVec(GVec v,GVec *lv)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidHeaderSpecific(*lv,VEC_COOKIE);

  ierr = VecDestroy(*lv);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGlobalToLocal"
/*@
   GVecGlobalToLocal - Copies a global vector including ghost points to 
   a local work vector.

   Collective on GVec

   Input Parameter:
+  v - the grid vector
-  mode - either INSERT_VALUES or ADD_VALUES

   Output Parameters:
.  lv - the local vector

.seealso: GVecLocalToGlobal(), GVecCreateLocalGVec()
@*/
int GVecGlobalToLocal(GVec v,InsertMode mode,GVec lv)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidHeaderSpecific(lv,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecglobaltolocal)(v,mode,lv); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecLocalToGlobal"
/*@
   GVecLocalToGlobal - Copies a local vector including ghost points to its
   global representation.

   Collective on GVec

   Input Parameter:
+  v - the local grid vector
-  mode - either SET_VALUES or ADD_VALUES

   Output Parameters:
.  lv - the global vector

.seealso: GVecGlobalToLocal(), GVecCreateLocalGVec()
@*/
int GVecLocalToGlobal(GVec v,InsertMode mode,GVec lv)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  PetscValidHeaderSpecific(lv,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gveclocaltoglobal)(v,mode,lv); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateGMat"
/*@
   GridCreateGMat - Creates a sparse matrix with the appropriate preallocation
   of nonzeros for a discretization of an operator on the grid.

   Collective on GVec

   Input Parameter:
.  v - the grid 

   Output Parameters:
.  gmat - the grid matrix

.seealso: GridCreateGVec()
@*/
int GridCreateGMat(Grid v,GMat *gmat)
{
  int  ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,GRID_COOKIE);
  PetscValidPointer(gmat);

  ierr = (*v->ops->gridcreategmat)(v,gmat); CHKERRQ(ierr);
  PetscObjectCompose((PetscObject) *gmat, "grid",(PetscObject) v);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecEvaluateFunction"
/*@
   GVecEvaluateFunction - Evaluates a function on the locations defined by
   the underlying grid and its discretization.

   Collective on GVec

   Input Parameter:
+  v - the  grid vector
.  f - the user provided function
-  ctx - optional user provided context for the function

@*/
int GVecEvaluateFunction(GVec v,PointFunction f,void* ctx)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecevaluatefunction)(v,f,ctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecEvaluateFunctionGalerkin"
/*@
   GVecEvaluateFunctionGalerkin - Evaluates a function on the locations defined by
   the underlying grid and its discretization.

   Collective on GVec

   Input Parameter:
+  v - the  grid vector
.  f - the user provided function
-  ctx - optional user provided context for the function

@*/
int GVecEvaluateFunctionGalerkin(GVec v,PointFunction f,void* ctx)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);

  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gvecevaluatefunctiongalerkin)(v,f,ctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GMatFDColoringCreate"
/*@
   GMatFDColoringCreate - Sets up the data structure in a grid matrix to 
   compute a Jacobian efficiently using finite differences and coloring.

   Collective on GMat

   Input Parameter:
.  J - the matrix

@*/
int GMatFDColoringCreate(GMat J)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(J,MAT_COOKIE);

  ierr = GMatGetGrid(J,&grid); CHKERRQ(ierr);
  ierr = (*grid->ops->gmatfdcoloringcreate)(J); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GMatFDColoringApply"
int GMatFDColoringApply(GMat J,GVec x,MatStructure *str,void *sctx)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  ierr = GMatGetGrid(J,&grid); CHKERRQ(ierr);
  if (!grid->fdcoloring) {
    ierr = GMatFDColoringCreate(J); CHKERRQ(ierr);
  }
  ierr = MatFDColoringApply(J,grid->fdcoloring,x,str,sctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#include "snes.h"

#undef __FUNC__  
#define __FUNC__ "GVecSNESDefaultComputeJacobianWithColoring"
int GVecSNESDefaultComputeJacobianWithColoring(SNES snes,GVec x1,GMat *JJ,GMat *B,
                                               MatStructure *flag,void *ctx)
{
  int  ierr;
  Grid grid;

  PetscFunctionBegin;
  ierr = GMatGetGrid(*B,&grid); CHKERRQ(ierr);
  if (!grid->fdcoloring) {
    ierr = GMatFDColoringCreate(*B); CHKERRQ(ierr);
  }
  ierr = SNESDefaultComputeJacobianWithColoring(snes,x1,JJ,B,flag,grid->fdcoloring);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "PointFunctionOne"
/*@
   PointFunctionOne - A PointFunction that is one at all locations.
   Currently only for scalar problems.

   Not Collective

   Input Parameters:
+  n - number of points
.  x,y,z, coordinates of points
-  dummy - unneeded context variable

   Output Parameter:
.  values - location where 1.0 is stored (n times)

.seealso: PointFunctionZero()
@*/
int PointFunctionOne(int n,double *x,double *y,double *z,Scalar *values,void *dummy)
{
  int i;
  
  PetscFunctionBegin;
  for ( i=0; i<n; i++ ) { values[i] = 1.0;}
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "PointFunctionZero"
/*@
    PointFunctionZero - A PointFunction that is zero at all locations.
    Currently only for scalar problems.

    Not Collective

    Input Parameters:
+   n - number of points
.   x,y,z, coordinates of points
-   dummy - unneeded context variable

    Output Parameter:
.   values - location where 0.0 is stored (n times)

.seealso: PointFunctionOne()
@*/
int PointFunctionZero(int n,double *x,double *y,double *z,Scalar *values,void *dummy)
{
  int i;
  
  PetscFunctionBegin;
  for ( i=0; i<n; i++ ) { values[i] = 0.0;}
  PetscFunctionReturn(0);
}


