#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: lin2d.c,v 1.24 1998/04/21 19:47:37 bsmith Exp $";
#endif

/*
   Defines piecewise bilinear function space on a two dimensional 
 grid. Suitable for finite difference type discretization of a PDE.
*/

#include "src/gvec/impls/rectangular/rectimpl.h"         /*I "gvec.h" I*/
#include "pinclude/pviewer.h"

typedef struct  {
  int            dof;
  DAPeriodicType per;
} Grid_Rectangular_2D_Linear;

#undef __FUNC__  
#define __FUNC__ "GridDestroy_Rectangular_2D_Linear"
static int GridDestroy_Rectangular_2D_Linear(Grid grid)
{
  Grid_Rectangular_2D_Linear *s = (Grid_Rectangular_2D_Linear*) grid->data;
  int                        ierr;

  PetscFunctionBegin;
  PetscFree(s);
  ierr = GridDataDestroy(grid->griddata); CHKERRQ(ierr);
  if (grid->iscoloring) {
    ierr = ISColoringDestroy(grid->iscoloring); CHKERRQ(ierr);
  }
  if (grid->fdcoloring) {
    ierr = MatFDColoringDestroy(grid->fdcoloring); CHKERRQ(ierr);
  }
  PLogObjectDestroy(grid);
  PetscHeaderDestroy(grid);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateGMat_Rectangular_2D_Linear"
static int GridCreateGMat_Rectangular_2D_Linear(Grid grid,GMat *gmat)
{
  Grid_Rectangular_2D_Linear *s = (Grid_Rectangular_2D_Linear *)grid->data;
  GridData_Rectangular       *gridr = (GridData_Rectangular*)grid->griddata->data;
  int                        m,ierr;
  MPI_Comm                   comm;
  Vec                        w;
  ISLocalToGlobalMapping     ltog;

  PetscFunctionBegin;
  ierr = DACreateGlobalVector(gridr->da,&w); CHKERRQ(ierr);
  ierr = VecGetLocalSize(w,&m); CHKERRQ(ierr);
  ierr = PetscObjectGetComm((PetscObject)grid,&comm);CHKERRQ(ierr);
  ierr = MatCreateMPIAIJ(comm,m,m,PETSC_DECIDE,PETSC_DECIDE,5*s->dof,0,0,0,gmat);CHKERRQ(ierr);

  ierr = DAGetISLocalToGlobalMapping(gridr->da,&ltog); CHKERRQ(ierr);
  ierr = MatSetLocalToGlobalMapping(*gmat,ltog); CHKERRQ(ierr);

  PetscObjectCompose((PetscObject) *gmat, "grid",(PetscObject) grid);CHKERRQ(ierr);
  
  PetscFunctionReturn(0);
}

/* -------------------------------------------------------------------*/

#undef __FUNC__  
#define __FUNC__ "GridView_Rectangular_2D_Linear"
static int GridView_Rectangular_2D_Linear(Grid grid,Viewer viewer)
{
  ViewerType  vtype;
  int         rank,ierr;
  FILE        *fd;

  PetscFunctionBegin;
  MPI_Comm_rank(grid->comm,&rank); if (rank > 0) PetscFunctionReturn(0);
  ierr = ViewerGetType(viewer,&vtype);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER){
    ierr = ViewerASCIIGetPointer(viewer,&fd); CHKERRQ(ierr);
    fprintf(fd,"Grid Object:\n");
    fprintf(fd,"  Rectangular 2D bilinear discretization\n");
  } else {
    SETERRQ(1,1,"Viewer type not supported by PETSc object");
  }
  ierr = GridDataView(grid->griddata,viewer); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecView_Rectangular_2D_Linear_Draw"
static int GVecView_Rectangular_2D_Linear_Draw(GVec gvec,Viewer v)
{
  Grid          grid;
  GridData      griddata;
  DA            da;
  Vec           lvec;
  int           i,rank,size,ierr,m,n,w,showgrid,done = 0,pause;
  int           mlocal,nlocal,gxs,gys;
  double        min,max,*x,*y,coor[4];
  double        xl = 0.0, yl = 0.0, xr = 1.0, yr = .1,scale = 1.0;
  Scalar        *va;
  Draw          draw;
  PetscTruth    isnull;
  MPI_Comm      comm;
  Draw          popup;

  PetscFunctionBegin;
  ierr = ViewerDrawGetDraw(v,&draw);CHKERRQ(ierr); 
  ierr = DrawIsNull(draw,&isnull); CHKERRQ(ierr); if (isnull) PetscFunctionReturn(0);
  ierr = DrawCheckResizedWindow(draw);CHKERRQ(ierr);
  ierr = PetscObjectGetComm((PetscObject) gvec,&comm); CHKERRQ(ierr);
  /*
     Access the data layout and grid coordinate information --------------
  */
  ierr = GVecGetGrid(gvec,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(griddata,&x,&y,0); CHKERRQ(ierr); 
  ierr = DAGetInfo(da,0,&m,&n,0,0,0,0,&w,0,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&gxs,&gys,0,&mlocal,&nlocal,0); CHKERRQ(ierr);

  /* Check for easy case when entire vector is on one processor */
  MPI_Comm_size(comm,&size); 
  if (size == 1 && w == 1) {
    ierr = DrawTensorContour(draw,m,n,x,y,gvec); CHKERRQ(ierr);
    PetscFunctionReturn(0);
  }
  MPI_Comm_rank(comm,&rank);

  /* create the PopUp scale window */
  ierr = VecMax(gvec,0,&max); CHKERRQ(ierr); ierr = VecMin(gvec,0,&min); CHKERRQ(ierr);
  if (max - min < 1.e-7) {min -= 5.e-8; max += 5.e-8;}
  ierr = DrawGetPopup(draw,&popup); CHKERRQ(ierr);
  ierr = DrawScalePopup(popup,min,max); CHKERRQ(ierr);

  ierr = GVecGetLocalWorkGVec(gvec,&lvec); CHKERRQ(ierr);
  ierr = GVecGlobalToLocal(gvec,ADD_VALUES,lvec); CHKERRQ(ierr);
  ierr = VecGetArray(lvec,&va); CHKERRQ(ierr);

  ierr = OptionsHasName(PETSC_NULL,"-draw_contour_grid",&showgrid);CHKERRQ(ierr);

  /* set the extreme window coordinates */
  xl = x[0]; yl = y[0]; xr = x[m-1]; yr = y[n-1];
  ierr = DrawSetCoordinates(draw,xl,yl,xr,yr); CHKERRQ(ierr);

  /* Draw the contour plot */
  ierr = DrawGetPause(draw,&pause); CHKERRA(ierr);
  while (1) {
    ierr = DrawTensorContourPatch(draw,mlocal,nlocal,x+gxs,y+gys,max,min,va);CHKERRQ(ierr);

    if (showgrid && !rank) {
      for ( i=0; i<m; i++ ) {
        DrawLine(draw,x[i],y[0],x[i],y[n-1],DRAW_BLACK);
      }
      for ( i=0; i<n; i++ ) {
        DrawLine(draw,x[0],y[i],x[m-1],y[i],DRAW_BLACK);
      }
    }
    ierr = DrawSynchronizedFlush(draw); CHKERRQ(ierr);

    if (pause != -1) break;
    if (!rank) {
      DrawButton  button;
      double      xc,yc;
    
      ierr = DrawGetCoordinates(draw,&xl,&yl,&xr,&yr); CHKERRQ(ierr);
      ierr = DrawGetMouseButton(draw,&button,&xc,&yc,0,0); CHKERRQ(ierr);
      if  (button == BUTTON_RIGHT)      done = 1;
      if (button == BUTTON_LEFT)        scale = .5;
      else if (button == BUTTON_CENTER) scale = 2.;
      xl = scale*(xl  - xc) + xc;
      xr = scale*(xr  - xc) + xc;
      yl = scale*(yl  - yc) + yc;
      yr = scale*(yr  - yc) + yc;
      ierr = MPI_Bcast(&done,1,MPI_INT,0,comm);CHKERRQ(ierr);
      if (done) break;
      coor[0] = xl; coor[1] = xr; coor[2] = yl; coor[3] = yr;
      ierr = MPI_Bcast(coor,4,MPI_DOUBLE,0,comm);CHKERRQ(ierr);
      ierr = DrawSetCoordinates(draw,xl,yl,xr,yr); CHKERRQ(ierr);
    } else {
      ierr = MPI_Bcast(&done,1,MPI_INT,0,comm);CHKERRQ(ierr);
      if (done) break;
      ierr = MPI_Bcast(coor,4,MPI_DOUBLE,0,comm);CHKERRQ(ierr);
      xl = coor[0]; xr = coor[1]; yl = coor[2]; yr = coor[3];
      ierr = DrawSetCoordinates(draw,xl,yl,xr,yr); CHKERRQ(ierr);
    }
     
    ierr = DrawCheckResizedWindow(draw); CHKERRQ(ierr);
    if (!rank) {ierr = DrawClear(draw); CHKERRQ(ierr);}
    ierr = DrawSynchronizedFlush(draw); CHKERRQ(ierr);
  }
  ierr = VecRestoreArray(lvec,&va); CHKERRQ(ierr);
  ierr = GVecRestoreLocalWorkGVec(gvec,&lvec); CHKERRQ(ierr);
    
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecView_Rectangular_2D_Linear"
static int GVecView_Rectangular_2D_Linear(GVec gvec,Viewer viewer)
{
  Grid        grid;
  ViewerType  vtype;
  int         rank,ierr;

  PetscFunctionBegin;
  ierr = GVecGetGrid(gvec,&grid); CHKERRQ(ierr);
  MPI_Comm_rank(grid->comm,&rank); 
  ierr = ViewerGetType(viewer,&vtype);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER){
    if (rank == 0) {
      ierr = GridView(grid,viewer); CHKERRQ(ierr);
      ierr = ViewerFlush(viewer); CHKERRQ(ierr);
    }
    PetscBarrier(grid);
    ierr = VecView(gvec,viewer); CHKERRQ(ierr);
  } else if (vtype == DRAW_VIEWER) {
    ierr = GVecView_Rectangular_2D_Linear_Draw(gvec,viewer);CHKERRQ(ierr);
  } else {
    SETERRQ(1,1,"Viewer type not supported by PETSc object");
  }
  PetscFunctionReturn(0);
}

/* -------------------------------------------------------------------*/

#undef __FUNC__  
#define __FUNC__ "GridRefine_Rectangular_2D_Linear"
static int GridRefine_Rectangular_2D_Linear(Grid grid,int l,Grid *newgrid)
{
  GridData                   newgriddata;
  int                        ierr;
  Grid_Rectangular_2D_Linear *gridr,*newgridr;
  
  PetscFunctionBegin;
  ierr     = GridDataRefine(grid->griddata,l,&newgriddata); CHKERRQ(ierr);
  ierr     = GridDataCreateGrid(newgriddata,DISCRETIZATION_BILINEAR,newgrid);CHKERRQ(ierr);
  /* remove extra reference to gridd */ 
  ierr     = GridDataDestroy(newgriddata); CHKERRQ(ierr);
  gridr    = (Grid_Rectangular_2D_Linear *) grid->data;
  newgridr = (Grid_Rectangular_2D_Linear *) (*newgrid)->data;

  PetscMemcpy(newgridr,gridr,sizeof(Grid_Rectangular_2D_Linear));
  (*newgrid)->gridparent = grid;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateGVec_Rectangular_2D_Linear"
static int GridCreateGVec_Rectangular_2D_Linear(Grid grid,GVec *gvec)
{
  GridData_Rectangular  *gridr = (GridData_Rectangular*)grid->griddata->data;
  Vec                   v;
  int                   ierr;

  PetscFunctionBegin;
  ierr = DACreateGlobalVector(gridr->da,&v); CHKERRQ(ierr);
  PetscObjectCompose((PetscObject) v, "grid",(PetscObject) grid);CHKERRQ(ierr);

  *gvec = (GVec) v;

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGetLocalGVec_Rectangular_2D_Linear"
static int GVecGetLocalGVec_Rectangular_2D_Linear(GVec g,GVec *gvec)
{
  Grid                  grid;
  GridData              gridd;
  GridData_Rectangular  *gridr;
  Vec                   w,v;
  int                   ierr;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(g,&grid); CHKERRQ(ierr);
  gridd = grid->griddata;
  gridr = (GridData_Rectangular *)gridd->data;

  ierr = DACreateLocalVector(gridr->da,&w); CHKERRQ(ierr);
  ierr = VecDuplicate(w,&v); CHKERRQ(ierr);
  PetscObjectCompose((PetscObject) v, "grid",(PetscObject) grid);CHKERRQ(ierr);

  *gvec = (GVec) v;

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecRestoreLocalGVec_Rectangular_2D_Linear"
static int GVecRestoreLocalGVec_Rectangular_2D_Linear(GVec g,GVec *gvec)
{
  int ierr;

  PetscFunctionBegin;
  ierr = VecDestroy(*gvec);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGlobalToLocal_Rectangular_2D_Linear"
static int GVecGlobalToLocal_Rectangular_2D_Linear(GVec g,InsertMode mode, GVec l)
{
  Grid                 grid;
  GridData             gridd;
  GridData_Rectangular *gridr;
  int                  ierr;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(g,&grid); CHKERRQ(ierr);
  gridd = grid->griddata;
  gridr = (GridData_Rectangular *)gridd->data;

  ierr = DAGlobalToLocalBegin(gridr->da,g,mode,l); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(gridr->da,g,mode,l); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecLocalToGlobal_Rectangular_2D_Linear"
static int GVecLocalToGlobal_Rectangular_2D_Linear(GVec l,InsertMode mode, GVec g)
{
  Grid                 grid;
  GridData             gridd;
  GridData_Rectangular *gridr;
  int                  ierr;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(g,&grid); CHKERRQ(ierr);
  gridd = grid->griddata;
  gridr = (GridData_Rectangular *)gridd->data;

  ierr = DALocalToGlobal(gridr->da,l,mode,g); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridSetUp_Rectangular_2D_Linear"
int GridSetUp_Rectangular_2D_Linear(Grid grid)
{
  int                        ierr,flag;
  Grid_Rectangular_2D_Linear *s = (Grid_Rectangular_2D_Linear *)grid->data;
  GridData_Rectangular       *gridr = (GridData_Rectangular *)grid->griddata->data;

  PetscFunctionBegin;
  ierr              = DACreate2d(grid->comm,s->per,DA_STENCIL_BOX,gridr->m,gridr->n,PETSC_DECIDE,
                                 PETSC_DECIDE,s->dof,1,0,0,&gridr->da);CHKERRQ(ierr);
  grid->setupcalled = 1;

  ierr = OptionsHasName(PETSC_NULL,"-grid_view",&flag); CHKERRQ(ierr);
  if (flag) {
    ierr = GridView(grid,0);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecEvaluateFunction_Rectangular_2D_Linear"
static int GVecEvaluateFunction_Rectangular_2D_Linear(GVec v,PointFunction f,void* ctx)
{
  Grid      grid;
  GridData  griddata;
  DA        da;
  int       ierr,xs,ys,nx,ny,gxs,gys,gnx,gny,i,count,m,n,j;
  Scalar    *array;
  double    *x,*y,*xx,*yy;

  PetscFunctionBegin;
  /*
     Access the data layout and grid coordinate information --------------
  */
  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(griddata,&x,&y,0); CHKERRQ(ierr); 
  ierr = DAGetInfo(da,0,&m,&n,0,0,0,0,0,0,0); CHKERRQ(ierr);
  ierr = DAGetCorners(da,&xs,&ys,0,&nx,&ny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&gxs,&gys,0,&gnx,&gny,0); CHKERRQ(ierr);
  /* -------------------------------------------------------------------- */
  ierr = VecGetArray(v,&array); CHKERRQ(ierr);

  /* 
     generate  xx and yy arrays containing coordinates of all 
     nodes on this processor and evaluate function at those nodes
  */
  xx    = (double *) PetscMalloc( 2*nx*ny*sizeof(double) ); CHKPTRQ(xx);
  yy    = xx + nx*ny;
  count = 0;
  for ( j=0; j<ny; j++ ) {
    for ( i=0; i<nx; i++ ) {
      xx[count]   = x[xs+i]; 
      yy[count++] = y[ys+j];
    }
  }
  ierr = (*f)(nx*ny,xx,yy,0,array,ctx); CHKERRQ(ierr);
  PetscFree(xx);
  ierr = VecRestoreArray(v,&array); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecEvaluateFunctionGalerkin_Rectangular_2D_Linear"
static int GVecEvaluateFunctionGalerkin_Rectangular_2D_Linear(GVec v,PointFunction f,void* ctx)
{
  Grid      grid;
  GridData  griddata;
  DA        da;
  int       ierr,xs,ys,nx,ny,gxs,gys,gnx,gny,i,m,n,j,nx1,ny1,xs1,ys1;
  Scalar    *array;
  double    *x,*y,ht,hb,hr,hl,hx,hy;

  PetscFunctionBegin;
  /*
     Evaluate the function on the grid 
  */
  ierr = GVecEvaluateFunction(v,f,ctx); CHKERRQ(ierr);

  /*
     Access the data layout and grid coordinate information --------------
  */
  ierr = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(griddata,&x,&y,0); CHKERRQ(ierr); 
  ierr = DAGetInfo(da,0,&m,&n,0,0,0,0,0,0,0); CHKERRQ(ierr);
  ierr = DAGetCorners(da,&xs,&ys,0,&nx,&ny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&gxs,&gys,0,&gnx,&gny,0); CHKERRQ(ierr);
  /* -------------------------------------------------------------------- */

  ierr = VecGetArray(v,&array); CHKERRQ(ierr);

  /*
      Scale the entries of array by the average cell sizes
         .25*(hr + hl)*(ht + hb)
  */
  nx1 = nx; ny1 = ny;
  if (xs == 0) {xs1 = 1; nx1--;} else xs1 = xs;
  if (ys == 0) {ys1 = 1; ny1--;} else ys1 = ys;
  if (xs + nx == m) nx1--;
  if (ys + ny == n) ny1--;

  if (xs == 0 && ys == 0) {
    /*  Special case, lower left corner of grid */
    hr       =  x[1] - x[0];
    ht       =  y[1] - y[0];
    array[0] *= hr*ht;
  }
  if (xs + nx == m && ys == 0) {
    /* Special case, lower right corner of grid */
    hl        =  x[xs+nx-1] - x[xs+nx-2];
    ht        =  y[1] - y[0];
    array[nx-1] *= hl*ht;
  }
  if (xs == 0 && ys + ny == n) {
    /* Special case, upper left corner of grid */
    hr              =  x[1] - x[0];
    hb              =  y[ys+ny-1] - y[ys+ny-2];
    array[nx*ny-nx] *= hr*hb;
  }
  if (xs + nx == m && ys + ny == n) {
    /* Special case, upper right corner of grid */
    hl              =  x[xs+nx-1] - x[xs+nx-2];
    hb              =  y[ys+ny-1] - y[ys+ny-2];
    array[nx*ny-1] *= hl*hb;
  }
  if (ys == 0) {
    /* special case boundary along bottom of grid */
    ht =  y[1] - y[0];
    for ( i=xs1; i<xs1+nx1; i++ ) {
      hx          = x[i+1] - x[i-1];
      array[i-xs] *= .5*hx*ht;
    }
  }
  if (ys + ny == n) {
    /* special case boundary along top of grid */
    hb =  y[ys+ny-1] - y[ys+ny-2];
    for ( i=xs1; i<xs1+nx1; i++ ) {
      hx                   = x[i+1] - x[i-1];
      array[nx*ny-nx+i-xs] *= .5*hx*hb;
    }
  }
  if (xs == 0) {
    /* special case boundary along left edge */
    hr = x[1] - x[0];
    for ( i=ys1; i<ys1+ny1; i++ ) {
      hy                = y[i+1] - y[i-1];
      array[(i-ys)*nx] *= .5*hy*hr;
    }
  }
  if (xs + nx == m) {
    /* special case boundary along right edge */
    hl = x[xs+nx-1] - x[xs+nx-2];
    for ( i=ys1; i<ys1+ny1; i++ ) {
      hy                = y[i+1] - y[i-1];
      array[(i-ys+1)*nx-1] *= .5*hy*hl;
    }
  }
  /* do the interior portion of the grid */
  for ( j=ys1; j<ys1+ny1; j++ ) {
    hy = y[j+1] - y[j-1];
    for ( i=xs1; i<xs1+nx1; i++ ) {
      hx = x[i+1] - x[i-1];
      array[(j-ys)*nx+i-xs] *= .25*hy*hx;
    }
  }
  ierr = VecRestoreArray(v,&array); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateRestriction_Rectangular_2D_Linear"
int GridCreateRestriction_Rectangular_2D_Linear(Grid fgrid,Grid cgrid,GMat *gmat)
{
  GridData fgriddata,cgriddata;
  DA       fda,cda;
  double   *fx,*fy,*cx,*cy;
  int      fm,fn,fxs,fys,fnx,fny,fgys,fgxs,fgnx,fgny,cm,cn,cxs,cys,cnx,cny;
  int      cgys,cgxs,cgnx,cgny,ierr;

  PetscFunctionBegin;
  /*
     Access the data layout and grid coordinate information --------------
  */
  ierr = GridGetGridData(fgrid,&fgriddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(fgriddata,&fda); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(fgriddata,&fx,&fy,0); CHKERRQ(ierr); 
  ierr = DAGetInfo(fda,0,&fm,&fn,0,0,0,0,0,0,0); CHKERRQ(ierr);
  ierr = DAGetCorners(fda,&fxs,&fys,0,&fnx,&fny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(fda,&fgxs,&fgys,0,&fgnx,&fgny,0); CHKERRQ(ierr);

  ierr = GridGetGridData(cgrid,&cgriddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(cgriddata,&cda); CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(cgriddata,&cx,&cy,0); CHKERRQ(ierr); 
  ierr = DAGetInfo(cda,0,&cm,&cn,0,0,0,0,0,0,0); CHKERRQ(ierr);
  ierr = DAGetCorners(cda,&cxs,&cys,0,&cnx,&cny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(cda,&cgxs,&cgys,0,&cgnx,&cgny,0); CHKERRQ(ierr);
  /* -------------------------------------------------------------------- */

  /* also check dof == 1 */
  if (fgrid->gridparent != cgrid) {
    SETERRQ(PETSC_ERR_SUP,0,"Only supported for nested grids");
  }
  if (fm != 2*cm - 1) {
    SETERRQ(PETSC_ERR_SUP,0,"Only supported for factor 2 refinement");
  }
  if (fn != 2*cn - 1) {
    SETERRQ(PETSC_ERR_SUP,0,"Only supported for factor 2 refinement");
  }

  PetscFunctionReturn(0);
}

extern int GMatFDColoringCreate_Rectangular_2D_Linear(GMat);

static struct _GridOps GOps = { GridSetUp_Rectangular_2D_Linear,
                              0,
                              GridRefine_Rectangular_2D_Linear,
                              0, 
                              GridCreateGVec_Rectangular_2D_Linear,
                              GridCreateGMat_Rectangular_2D_Linear,
                              GVecGetLocalGVec_Rectangular_2D_Linear,
                              GVecRestoreLocalGVec_Rectangular_2D_Linear,
                              0,
                              0,
                              GVecGlobalToLocal_Rectangular_2D_Linear,
                              GVecLocalToGlobal_Rectangular_2D_Linear,
                              0,
                              0,
                              0,
                              GVecView_Rectangular_2D_Linear,
                              GridCreateRestriction_Rectangular_2D_Linear,
                              GVecEvaluateFunction_Rectangular_2D_Linear,
                              GVecEvaluateFunctionGalerkin_Rectangular_2D_Linear,
                              GMatFDColoringCreate_Rectangular_2D_Linear};


#undef __FUNC__  
#define __FUNC__ "DCCreateRectangular2DLinear_Private"
/*
     DCCreateRectangular2DLinear_Private - Creates a discretization 
           appropriate for problems on 1 dimensional rectangular
           grids using linear interpolation.

  Input Parameter:
.   grid - the grid that defines the region
.   dof  - number of degrees of freedom per node
.   per - either DA_NONPERIODIC or DA_XPERIODIC

  Output Parameter:
.   newdc - the discretization context

.keywords: Discrezation

.seealso: DCDestroy()
*/
int GridCreateRectangular2DLinear_Private(GridData gridd,Grid *newgrid)
{
  Grid_Rectangular_2D_Linear *s;
  Grid                       grid;

  PetscFunctionBegin;
  PetscHeaderCreate(grid,_p_Grid,struct _GridOps,GRID_COOKIE,DISCRETIZATION_LINEAR,gridd->comm,GridDestroy,GridView);
  PLogObjectCreate(grid);
  PLogObjectMemory(grid,sizeof(struct _p_Grid)+sizeof(Grid_Rectangular_2D_Linear));
  s              = PetscNew(Grid_Rectangular_2D_Linear); CHKPTRQ(s);
  PetscMemzero(s,sizeof(Grid_Rectangular_2D_Linear));
  PetscMemcpy(grid->ops,&GOps,sizeof(struct _GridOps));
  grid->data     = (void *) s;
  grid->ops->destroy  = GridDestroy_Rectangular_2D_Linear;
  grid->ops->view     = GridView_Rectangular_2D_Linear;
  grid->griddata = gridd; 
  PetscObjectReference((PetscObject)gridd);

  grid->iscoloring = 0;
  grid->fdcoloring = 0;

  s->dof         = 1;
  s->per         = DA_NONPERIODIC;
  *newgrid       = grid;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GMatFDColoringCreate_Rectangular_2D_Linear"
int GMatFDColoringCreate_Rectangular_2D_Linear(GMat J) 
{
  int                        ierr, xs,nx,ys,ny,*colors,i,ii,gxs,xs1,j;           
  int                        m,n,w,s,indices[5],*gindices,nx1,gnx,ny1,gny,gys,ys1,row;
  MPI_Comm                   comm;
  Scalar                     values[5];
  Grid                       grid;
  GridData                   griddata;
  DA                         da;

  PetscFunctionBegin;
  /*
                                  m 
          -----------|----|------------------|----|--------------
                  gxs    xs <----- nx ------>                           
                      <--------- gnx ------------>
  */
  /*
     Access the data layout and grid coordinate information --------------
  */
  ierr = GMatGetGrid(J,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);
  /* ierr = GridDataRectangularGetXYZ(griddata,&x,&y,0); CHKERRQ(ierr); */
  ierr = DAGetInfo(da,0,&m,&n,0,0,0,0,&w,&s,0); CHKERRQ(ierr);
  ierr = DAGetCorners(da,&xs,&ys,0,&nx,&ny,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&gxs,&gys,0,&gnx,&gny,0); CHKERRQ(ierr);
  /* -------------------------------------------------------------------- */

  if (w != 1)   SETERRQ(PETSC_ERR_SUP,0,"Scalar problems only");
  if (s != 1)   SETERRQ(PETSC_ERR_SUP,0,"Stencil width 1 only");
  /* also no support for periodic boundary conditions yet */

  ierr = PetscObjectGetComm((PetscObject)da,&comm); CHKERRQ(ierr);
  ierr = DAGetGlobalIndices(da,PETSC_NULL,&gindices);

  /*
      create the coloring
  */
  colors = (int *) PetscMalloc( nx*ny*sizeof(int) ); CHKPTRQ(colors);
  ii = 0;
  for ( j=ys; j<ys+ny; j++ ) {
    for ( i=xs; i<xs+nx; i++ ) {
      colors[ii++] = (i % 3) + 3*(j % 3);
    }
  }
  ierr = ISColoringCreate(comm,nx*ny,colors,&grid->iscoloring); CHKERRQ(ierr);
  PetscFree(colors);

  values[0] = values[1] = values[2] = values[3] = values[4] = 0.0;

  /*
      Generate the exact nonzero structure of the matrix
  */
  nx1 = nx; ny1 = ny;
  if (xs == 0) {xs1 = 1; nx1--;} else xs1 = xs;
  if (ys == 0) {ys1 = 1; ny1--;} else ys1 = ys;
  if (xs + nx == m) nx1--;
  if (ys + ny == n) ny1--;


  if (xs == 0 && ys == 0) {
    /*  Special case, lower left corner of grid */
    row        = 0;
    indices[0] = row; 
    indices[1] = row+1; 
    indices[2] = row+gnx; 
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (xs + nx == m && ys == 0) {
    /* Special case, lower right corner of grid */
    row        = gnx-1;
    indices[0] = row-1;
    indices[1] = row;
    indices[2] = row+gnx; 
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (xs == 0 && ys + ny == n) {
    /* Special case, upper left corner of grid */
    row        = gny*gnx-gnx;
    indices[0] = row-gnx;
    indices[1] = row;
    indices[2] = row+1;
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (xs + nx == m && ys + ny == n) {
    /* Special case, upper right corner of grid */
    row        = gny*gnx-1;
    indices[0] = row-gnx;
    indices[1] = row-1;
    indices[2] = row;
    ierr = MatSetValuesLocal(J,1,&row,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  if (ys == 0) {
    /* special case boundary along bottom of grid */
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row        = i-gxs;
      indices[0] = row-1;
      indices[1] = row;
      indices[2] = row+1; 
      indices[3] = row+gnx; 
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  if (ys + ny == n) {
    /* special case boundary along top of grid */
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row        = i-gxs+gny*gnx-gnx;
      indices[0] = row-gnx;
      indices[1] = row-1;
      indices[2] = row;
      indices[3] = row+1;
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  if (xs == 0) {
    /* special case boundary along left edge */
    for ( i=ys1; i<ys1+ny1; i++ ) {
      row        = (i-gys)*gnx;
      indices[0] = row-gnx;
      indices[1] = row;
      indices[2] = row+1;
      indices[3] = row+gnx;
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  if (xs + nx == m) {
    /* special case boundary along right edge */
    for ( i=ys1; i<ys1+ny1; i++ ) {
      row        = (i-gys+1)*gnx-1;
      indices[0] = row-gnx;
      indices[1] = row-1;
      indices[2] = row;
      indices[3] = row+gnx;
      ierr = MatSetValuesLocal(J,1,&row,4,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  /* do the interior portion of the grid */
  for ( j=ys1; j<ys1+ny1; j++ ) {
    for ( i=xs1; i<xs1+nx1; i++ ) {
      row = (j-gys)*gnx+i-gxs;
      indices[0] = row-gnx;
      indices[1] = row-1;
      indices[2] = row;
      indices[3] = row+1;
      indices[4] = row+gnx;
      ierr = MatSetValuesLocal(J,1,&row,5,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    }
  }
  ierr = MatAssemblyBegin(J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);  
  ierr = MatAssemblyEnd(J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);  

  ierr = MatFDColoringCreate(J,grid->iscoloring,&grid->fdcoloring); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


