

/* $Id: common1and2.c,v 1.9 1997/07/10 03:47:06 bsmith Exp $ */

/*
     Demonstrates how one may construct a simple finite difference operator for
   one dimensional Laplacian using PETSc GVecs.
*/

int UserComputeMatrix(GMat gmat)
{
  Grid                 grid;
  GridData             griddata;
  DA                   da;
  int                  ierr,xs,m,indices[3],M,i;
  Scalar               values[3],hl,hr;
  double               *x;

  /*
     Get the grid data structure
  */
  ierr   = GMatGetGrid(gmat,&grid); CHKERRQ(ierr);
  ierr   = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr   = GridDataRectangularGetXYZ(griddata,&x,PETSC_NULL,PETSC_NULL); CHKERRQ(ierr);
  ierr   = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);

  /*
     Loop over local rows generating the stencil 
  */
  ierr = DAGetCorners(da,&xs,0,0,&m,0,0); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);

  /*
      Note that values are inserted into the matrix using the 
    global PETSc ordering.
  */
  if (xs == 0) {
    /* special case, left end point of grid */
    i          = 0;
    indices[0] = 0; 
    indices[1] = 1; 
    hr         = x[1] - x[0];
    values[0]  = -1.0/hr;
    values[1]  = 1.0/hr;
    ierr = MatSetValues(gmat,1,&i,2,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    xs++; m--;
  }
  if (xs + m == M) {
    /* special case, right end point of grid */
    i          = M-1;
    indices[0] = M-2; 
    indices[1] = M-1; 
    hl         = x[M-1] - x[M-2];
    values[0]  = 1.0/hl;
    values[1]  = -1.0/hl;
    ierr = MatSetValues(gmat,1,&i,2,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    m--;
  }
  for ( i=xs; i<xs+m; i++ ) {
    indices[0] = i-1; 
    indices[1] = i; 
    indices[2] = i+1;
    hl         = x[i] - x[i-1];
    hr         = x[i+1] - x[i];
    values[0]  = 1.0/hl;
    values[1]  = -(hr+hl)/(hr*hl);
    values[2]  = 1.0/hr;
    ierr = MatSetValues(gmat,1,&i,3,indices,values,INSERT_VALUES);CHKERRQ(ierr); 
  }
  ierr = MatAssemblyBegin(gmat,MAT_FINAL_ASSEMBLY);
  ierr = MatAssemblyEnd(gmat,MAT_FINAL_ASSEMBLY);
  return 0;
}

/* ---------------------------------------------------------------------------------*/
/*
     Applies Dirichlet boundary conditions to boundary points of the domain 
*/
int UserComputeDirichlet(GVec x,GVec b,GMat A,
                         int (*f)(int,double*,double*,double*,Scalar*,void*),void* fctx)
{
  Grid                       grid;
  GridData                   griddata;
  int                        ierr,istart,iend,rows[2],rowcount = 0,vsize;
  Scalar                     value,diagv = 0.0;
  IS                         is;
  double                     *xx;

  ierr  = GMatGetGrid(A,&grid); CHKERRQ(ierr);
  ierr  = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr  = GridDataRectangularGetXYZ(griddata,&xx,PETSC_NULL,PETSC_NULL);  CHKERRQ(ierr);
 
  ierr = MatGetSize(A,&vsize,&vsize);  CHKERRQ(ierr);
  ierr = MatGetOwnershipRange(A,&istart,&iend); CHKERRQ(ierr);

  /*  check left end point */
  if (istart == 0) {
    ierr = MatGetValues(A,1,&istart,1,&istart,&diagv);CHKERRQ(ierr); diagv *= 2.0; 
    if (x && b) {
      ierr    = (*f)(1,xx,0,0,&value,fctx); CHKERRQ(ierr);
      ierr    = VecSetValues(x,1,&istart,&value,INSERT_VALUES); CHKERRQ(ierr);
      value   = value*diagv; 
      ierr    = VecSetValues(b,1,&istart,&value,INSERT_VALUES); CHKERRQ(ierr);
    }
    rows[rowcount++] = 0;
  }
  if (iend == vsize) {
    iend--;
    if (diagv == 0.0) {
      ierr = MatGetValues(A,1,&iend,1,&iend,&diagv);CHKERRQ(ierr); diagv *= 2.0;
    }
    if (x && b) {
      ierr    = (*f)(1,xx+iend,0,0,&value,fctx); CHKERRQ(ierr);
      ierr    = VecSetValues(x,1,&iend,&value,INSERT_VALUES); CHKERRQ(ierr);
      value   = value*diagv; 
      ierr    = VecSetValues(b,1,&iend,&value,INSERT_VALUES); CHKERRQ(ierr);
    }
    rows[rowcount++] = iend;
  }
  ierr = ISCreateGeneral(PETSC_COMM_SELF,rowcount,rows,&is); CHKERRQ(ierr);
  ierr = MatZeroRows(A,is,&diagv); CHKERRQ(ierr);
  ierr = ISDestroy(is); CHKERRQ(ierr);
  return 0;
}







