#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex9.c,v 1.2 1998/03/20 22:52:53 bsmith Exp $";
#endif

static char help[] = "Solves a nonlinear system in parallel with SNES.\n\
  \n\
The 2D driven cavity problem is solved in a velocity-vorticity formulation.\n\
The flow can be driven with the lid or with bouyancy or both:\n\
  -lidvelocity <lid>, where <lid> = dimensionless velocity of lid\n\
  -grashof <gr>, where <gr> = dimensionless temperature gradient\n\
  -prandtl <pr>, where <pr> = dimensionless thermal/momentum diffusity ratio\n\
Mesh sequencing is available, starting coarse and recursively doubling:\n\
  -mx <xg>, where <xg> = initial number of grid points in the x-direction\n\
  -my <yg>, where <yg> = initial number of grid points in the y-direction\n\
  -ncycles <ncycles>, where <ncycles> = number of refinement cycles\n\
  -printg : print grid information\n\
Graphics of the contours of (U,V,Omega,T) are available on each grid:\n\
  -contours : draw contour plots of solution\n\
Parallelism can be invoked based on the DA construct:\n\
  -Nx <npx>, where <npx> = number of processors in the x-direction\n\
  -Ny <npy>, where <npy> = number of processors in the y-direction\n\n";

/*T
   Concepts: SNES^Solving a system of nonlinear equations (parallel multicomponent example);
   Concepts: DA^Using distributed arrays;
   Routines: SNESCreate(); SNESSetFunction(); SNESSetJacobian();
   Routines: SNESSolve(); SNESSetFromOptions(); DAView();
   Routines: DACreate2d(); DADestroy(); DACreateGlobalVector(); DACreateLocalVector();
   Routines: DAGetCorners(); DAGetGhostCorners(); DALocalToGlobal();
   Routines: DAGlobalToLocalBegin(); DAGlobalToLocalEnd(); DAGetGlobalIndices();
   Processors: n
T*/

/* ------------------------------------------------------------------------

    We thank David E. Keyes for contributing the driven cavity discretization
    within this example code.

    This example incorporates mesh sequencing; ex8.c solves the problem for
    a single grid; additional files needed:
        common8and9.c - initial guess and nonlinear function evaluation 
                        routines (used by both ex8.c and ex9.c)
        ex8and9.h     - include file used by ex8.c and ex9.c

    This problem is modeled by the partial differential equation system
  
	- Lap( U ) - Grad_y( Omega ) = 0
	- Lap( V ) + Grad_x( Omega ) = 0
	- Lap( Omega ) + Div( [U*Omega,V*Omega] ) - GR*Grad_x( T ) = 0
	- Lap( T ) + PR*Div( [U*T,V*T] ) = 0

    in the unit square, which is uniformly discretized in each of x and
    y in this simple encoding.

    No-slip, rigid-wall Dirichlet conditions are used for [U,V].
    Dirichlet conditions are used for Omega, based on the definition of
    vorticity: Omega = - Grad_y( U ) + Grad_x( V ), where along each
    constant coordinate boundary, the tangential derivative is zero.
    Dirichlet conditions are used for T on the left and right walls,
    and insulation homogeneous Neumann conditions are used for T on
    the top and bottom walls. 

    A finite difference approximation with the usual 5-point stencil 
    is used to discretize the boundary value problem to obtain a 
    nonlinear system of equations.  Upwinding is used for the divergence
    (convective) terms and central for the gradient (source) terms.
    
    The Jacobian can be either
      * formed via finite differencing using coloring (the default), or
      * applied matrix-free via the option -snes_mf 
        (for larger grid problems this variant may not converge 
        without a preconditioner due to ill-conditioning).

    Mesh sequencing can be used to move from a given initial discretization
    to successively finer ones, doubling the number of intervals in
    the x- and y-directions at each cycle.  The converged solution on
    one mesh is interpolated to form the starting guess for the next grid.
    The option -ncycles indicates the number of cycles (default is 1).

    Note:  The DFVec tools used in this code will soon be replaced by a 
    new generation of grid vector utilities.

  ------------------------------------------------------------------------- */

/* 
   Include "da.h" so that we can use distributed arrays (DAs).
   Include "snes.h" so that we can use SNES solvers.  Note that this
   file automatically includes:
     petsc.h  - base PETSc routines   vec.h - vectors
     sys.h    - system routines       mat.h - matrices
     is.h     - index sets            ksp.h - Krylov subspace methods
     viewer.h - viewers               pc.h  - preconditioners
     sles.h   - linear solvers        dfvec.h - for visualization of solution
                                                and refinement
*/
#include "snes.h"
#include "da.h"
#include "dfvec.h"
#include <math.h>
#include <stdio.h>

/* 
   User-defined application context - contains data needed by the 
   application-provided call-back routine, FormFunction(). 
*/
typedef struct {
   double   lidvelocity, prandtl, grashof; 
   int      mx, my;            /* discretization in x, y directions */
   int      mc;                /* components in unknown vector */
   Vec      localX, localF;    /* ghosted local vector */
   DA       da;                /* distributed array data structure (unknowns) */
   int      rank;              /* processor rank */
   int      size;              /* number of processors */
   MPI_Comm comm;              /* MPI communicator */
   int      icycle, ncycles;   /* current/total cycles through problem */
   Vec      x_old;             /* old solution vector */
   char     **label;           /* labels for components */
   int      print_grid;        /* flag - 1 indicates printing grid info */
   int      print_vecs;        /* flag - 1 indicates printing vectors */
   int      draw_contours;     /* flag - 1 indicates drawing contours */
} AppCtx;

/* 
   User-defined routines
*/
extern int FormInitialGuess(AppCtx*,Vec);
extern int FormFunction(SNES,Vec,Vec,void*);
extern int DAGetColoring2d_1(DA,ISColoring *,Mat *);
extern int UserCycleEnd(AppCtx*,DFVec);
extern int InitializeProblem(int,AppCtx*,DFVec*);

int main( int argc, char **argv )
{
  SNES          snes;                /* nonlinear solver */
  Vec           x, r;                /* solution, residual vectors */
  Mat           J;                   /* Jacobian matrix */
  AppCtx        user;                /* user-defined work context */
  int           its;                 /* iterations for convergence */
  MatFDColoring fdcoloring;          /* matrix coloring context */
  int           dim, k, flg, ierr;

  PetscInitialize( &argc, &argv,(char *)0,help );
  MPI_Comm_rank(PETSC_COMM_WORLD,&user.rank);
  MPI_Comm_size(PETSC_COMM_WORLD,&user.size);
  user.comm = PETSC_COMM_WORLD;

  /* Repeatedly solve nonlinear problem, refining discretization each time */
  user.ncycles = 1;
  ierr = OptionsGetInt(PETSC_NULL,"-ncycles",&user.ncycles,&flg); CHKERRA(ierr);
  for (k=0; k<user.ncycles; k++) {

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Create user context, set problem data, create vector data structures.
      Also, compute the initial guess.
      - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    ierr = InitializeProblem(k,&user,&x); CHKERRA(ierr);
    ierr = VecDuplicate(x,&r); CHKERRA(ierr);
    ierr = VecDuplicate(user.localX,&user.localF); CHKERRA(ierr);

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       Create nonlinear solver context
       - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    ierr = SNESCreate(PETSC_COMM_WORLD,SNES_NONLINEAR_EQUATIONS,&snes); CHKERRA(ierr);
    ierr = VecGetSize(x,&dim); CHKERRA(ierr);
    PetscPrintf(user.comm,"cycle = %d, global size = %d, lid velocity = %g, prandtl # = %g, grashof # = %g\n",
       k+1,dim,user.lidvelocity,user.prandtl,user.grashof);


    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       Set function evaluation routine and vector
       - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    ierr = SNESSetFunction(snes,r,FormFunction,&user); CHKERRA(ierr);

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       Create matrix data structure; set Jacobian evaluation routine
       - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
    /* 
         Set up coloring information needed for sparse finite difference
         approximation of the Jacobian
     */
    {
    ISColoring iscoloring;
    ierr = DAGetColoring2d_1(user.da,&iscoloring,&J); CHKERRQ(ierr);
    ierr = MatFDColoringCreate(J,iscoloring,&fdcoloring); CHKERRQ(ierr); 
    ierr = MatFDColoringSetFunction(fdcoloring,
           (int (*)(void))FormFunction,&user); CHKERRQ(ierr);
    ierr = MatFDColoringSetFromOptions(fdcoloring); CHKERRQ(ierr); 
    ierr = ISColoringDestroy(iscoloring); CHKERRQ(ierr);
    }

    /* 
       Set Jacobian matrix data structure and default Jacobian evaluation
       routine. User can override with:
  
       -snes_mf : matrix-free Newton-Krylov method with no preconditioning
       -snes_mf_operator : form preconditioning matrix as set by the user,
                           but use matrix-free approx for Jacobian-vector
                           products within Newton-Krylov method
  
    */
    ierr = SNESSetJacobian(snes,J,J,SNESDefaultComputeJacobianWithColoring,fdcoloring); CHKERRA(ierr);

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       Customize nonlinear solver; set runtime options
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    /*
       Set runtime options (e.g., -snes_monitor -snes_rtol <rtol> -ksp_type <type>)
    */
    ierr = SNESSetFromOptions(snes); CHKERRA(ierr);

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       Solve the nonlinear system
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
    /*
       Note: The user should initialize the vector, x, with the initial guess
       for the nonlinear solver prior to calling SNESSolve().  In particular,
       to employ an initial guess of zero, the user should explicitly set
       this vector to zero by calling VecSet(). [Here we set the initial 
       guess in the routine InitializeProblem().]
    */
    ierr = SNESSolve(snes,x,&its); CHKERRA(ierr); 

    PetscPrintf(PETSC_COMM_WORLD,"Number of Newton iterations = %d\n", its );

    /*
       Visualize solution
    */

    if (user.draw_contours) {
      ierr = DFVecDrawTensorContoursX(x,250,250); CHKERRA(ierr);
    }

    /* 
       User-defined actions at conclusion of each nonlinear solve
     */
    ierr = UserCycleEnd(&user,x); CHKERRA(ierr);

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       Free work space.  All PETSc objects should be destroyed when they
       are no longer needed.
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    ierr = MatDestroy(J); CHKERRA(ierr);
    ierr = MatFDColoringDestroy(fdcoloring); CHKERRA(ierr);  
    ierr = VecDestroy(user.localX); CHKERRA(ierr); ierr = VecDestroy(x); CHKERRA(ierr);
    ierr = VecDestroy(user.localF); CHKERRA(ierr); ierr = VecDestroy(r); CHKERRA(ierr);      
    ierr = SNESDestroy(snes); CHKERRA(ierr);  
  }

  PetscFree(user.label);
  ierr = DADestroy(user.da); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}

/* ------------------------------------------------------------------- */
/* 
   InitializeProblem - Initializes the problem for a given grid size.
   This routine forms the DA and vector data structures, and also
   computes the starting solution guess for the nonlinear solver.

   Input Parameters:
   icycle - cycle number
   user - user-defined application context

   Output Parameter:
   xvec - solution vector

   Notes:
   Mesh sequencing can be used to move from a given initial discretization
   to successively finer ones, where we double the number of intervals in
   the x- and y-directions at each cycle.  
     - first cycle: Create the initial DA, extract vectors, and compute an 
                           initial guess via the routine FormInitialGuess().
     - successive cycles:  Refine the DA and vectors, and interpolate the
                           converged solution to form the starting guess
                           then next grid.
 */
int InitializeProblem(int icycle,AppCtx *user,Vec *xvec)
{
  int    Nx, Ny;              /* number of processors in x- and y- directions */
  int    xs, xm, ys, ym, i, nsub, flg, N, Nlocal, debug, ierr;
  DA     da_ref;
  Vec    xv, xvref, *vsub, *vsub_old;

  user->icycle = icycle;

  /* First time through only ... */
  if (icycle == 0) {

    /*
       Initialize problem parameters
    */
    user->mx = 4; user->my = 4; 
    ierr = OptionsGetInt(PETSC_NULL,"-mx",&user->mx,&flg); CHKERRQ(ierr);
    ierr = OptionsGetInt(PETSC_NULL,"-my",&user->my,&flg); CHKERRQ(ierr);
    /*
       No. of components in the unknown vector and auxiliary vector
    */
    user->mc = 4;
    /* 
       Problem parameters (velocity of lid, prandtl, and grashof numbers)
    */
    user->lidvelocity = 1.0/(user->mx*user->my);
    ierr = OptionsGetDouble(PETSC_NULL,"-lidvelocity",&user->lidvelocity,&flg); CHKERRQ(ierr);
    user->prandtl = 1.0;
    ierr = OptionsGetDouble(PETSC_NULL,"-prandtl",&user->prandtl,&flg); CHKERRQ(ierr);
    user->grashof = 1.0;
    ierr = OptionsGetDouble(PETSC_NULL,"-grashof",&user->grashof,&flg); CHKERRQ(ierr);
    N = user->mx*user->my;
    ierr = OptionsHasName(PETSC_NULL,"-printv",&user->print_vecs); CHKERRQ(ierr);
    ierr = OptionsHasName(PETSC_NULL,"-printg",&user->print_grid); CHKERRQ(ierr);
    ierr = OptionsHasName(PETSC_NULL,"-contours",&user->draw_contours); CHKERRQ(ierr);

    /*
       Create distributed array (DA) to manage parallel grid and vectors
       for principal unknowns (x) and governing residuals (f)
    */
    Nx = PETSC_DECIDE; Ny = PETSC_DECIDE;
    ierr = OptionsGetInt(PETSC_NULL,"-Nx",&Nx,&flg); CHKERRQ(ierr);
    ierr = OptionsGetInt(PETSC_NULL,"-Ny",&Ny,&flg); CHKERRQ(ierr);
    ierr = DACreate2d(PETSC_COMM_WORLD,DA_NONPERIODIC,DA_STENCIL_STAR,user->mx,
                    user->my,Nx,Ny,user->mc,1,PETSC_NULL,PETSC_NULL,&user->da); CHKERRQ(ierr);
    /*
       Extract global and local vectors from DA
    */
    ierr = DACreateGlobalVector(user->da,&xv); CHKERRQ(ierr);
    ierr = DACreateLocalVector(user->da,&user->localX); CHKERRQ(ierr);

    /* 
       Label PDE components
     */
    user->label = (char **) PetscMalloc(user->mc*sizeof(char*)); CHKPTRQ(user->label);
    user->label[0] = "Velocity (U)";
    user->label[1] = "Velocity (V)";
    user->label[2] = "Omega";
    user->label[3] = "Temperature";

    user->x_old = 0;
  }

  /* Subsequent times through ... */
  else {

    /* Print current vector */
    if (user->print_vecs) {
      ierr = DFVecGetComponentVectors(user->x_old,&nsub,&vsub_old); CHKERRQ(ierr);
      for (i=0; i<user->mc; i++) {
        PetscPrintf(PETSC_COMM_WORLD,"\ncomponent %d\n",i);
        ierr = VecView(vsub_old[i],VIEWER_STDOUT_WORLD); CHKERRQ(ierr);
      }
      ierr = VecDestroyVecs(vsub_old,nsub); CHKERRQ(ierr);
    }
    ierr = OptionsHasName(PETSC_NULL,"-debug",&debug); CHKERRQ(ierr);
    if (debug) {
      ierr = DFVecDrawTensorContoursX(user->x_old,250,250); CHKERRQ(ierr);
    }

    /* Refine grid and initial guess */
    ierr = DFVecRefineVector(user->x_old,&xvref); CHKERRQ(ierr);

    /* Refine distributed array.  Note this assumes that the discrete function
       refinement is the same as the DA refinement.  Currently it is, but possibly
       in the future we should handle the coordination of these better. */
    ierr = DARefine(user->da,&da_ref); CHKERRQ(ierr);
    ierr = DADestroy(user->da); CHKERRQ(ierr);
    user->da = da_ref;

    ierr = DACreateGlobalVector(user->da,&xv); CHKERRQ(ierr);
    ierr = DAGetInfo(user->da,PETSC_NULL,&user->mx,&user->my,PETSC_NULL,PETSC_NULL,
           PETSC_NULL,PETSC_NULL,PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ(ierr);
  }

  /* Get local vector */
  ierr = DACreateLocalVector(user->da,&user->localX); CHKERRQ(ierr);

  /* Print grid info */
  if (user->print_grid) {
    ierr = DAView(user->da,VIEWER_STDOUT_SELF); CHKERRQ(ierr);
    ierr = DAGetCorners(user->da,&xs,&ys,0,&xm,&ym,0); CHKERRQ(ierr);
    PetscPrintf(PETSC_COMM_WORLD,"global grid: %d X %d with %d components per node ==> global vector dimension %d\n",
      user->mx,user->my,user->mc,user->mc*user->mx*user->my); fflush(stdout);
    ierr = VecGetLocalSize(xv,&Nlocal); CHKERRQ(ierr);
    PetscSequentialPhaseBegin(user->comm,1);
    printf("[%d] local grid %d X %d with %d components per node ==> local vector dimension %d\n",
      user->rank,xm,ym,user->mc,Nlocal);
    fflush(stdout);
    PetscSequentialPhaseEnd(user->comm,1);
  }  

  /* Compute initial guess first time only; otherwise, use refined vector */
  if (user->icycle == 0)  {
    FormInitialGuess(user,xv); CHKERRQ(ierr);
  } else {
    /* Note:  The refined vector must be copied into a distributed vector
       corresponding to the refined distributed array so that the correct
       local/global vector scatters will be used. */
    /* Note:  Currently the vector returned in DFVecRefineVector()
              when using DA's is not a complete DFVec, so we also
              refine the DA, then copy the refined vector into the
              new DA-associated vector.  This is clearly not a good
              way to handle the refinement; we'll upgrade this in
              the near future. 
    */
    ierr = VecCopy(xvref,xv); CHKERRQ(ierr);
    ierr = VecDestroy(xvref); CHKERRQ(ierr);

    ierr = DAGlobalToLocalBegin(user->da,xv,INSERT_VALUES,user->localX); CHKERRQ(ierr);
    ierr = DAGlobalToLocalEnd(user->da,xv,INSERT_VALUES,user->localX); CHKERRQ(ierr);

    /* Print refined vector */
    if (user->print_vecs) {
      ierr = DFVecGetComponentVectors(xv,&nsub,&vsub); CHKERRQ(ierr);
      for (i=0; i<user->mc; i++) {
        PetscPrintf(PETSC_COMM_WORLD,"\nrefined component %d\n",i);
        ierr = VecView(vsub[i],VIEWER_STDOUT_WORLD); CHKERRQ(ierr);
      }
      ierr = VecDestroyVecs(vsub,nsub); CHKERRQ(ierr);
    }
    ierr = OptionsHasName(PETSC_NULL,"-debug",&debug); CHKERRQ(ierr);
    if (debug) {
      ierr = DFVecDrawTensorContoursX(xv,250,250); CHKERRQ(ierr);
    }
  }
  
  *xvec = xv;
  return 0;
}

/* ------------------------------------------------------------------- */
/* 
   UserCycleEnd - User-defined actions at conclusion of each nonlinear solve.

   Input Parameters:
   user - user-defined application context
   x - converged solution vector

   Notes:
   Here we save the solution vector for the next pass through the
   cycle.  Alternatively, we could write the vector in binary to a 
   file, stop the nonlinear solver, and then restart using this as 
   a new initial guess to be refined for solving a larger problem.
 */
int UserCycleEnd(AppCtx *user,Vec x)
{
  int ierr;

  /* Destroy the existing x_old vector if necessary */
  if (user->icycle) {ierr = VecDestroy(user->x_old); CHKERRQ(ierr);}

  if (user->icycle+1 != user->ncycles) {
    ierr = VecDuplicate(x,&user->x_old); CHKERRQ(ierr);
    ierr = VecCopy(x,user->x_old); CHKERRQ(ierr);
  }
  return 0;
}
