#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: griddata.c,v 1.20 1998/04/27 14:22:23 curfman Exp $";
#endif

/*
     Defines the interface to the grid functions
*/

#include "src/gvec/gvecimpl.h"    /*I "gvec.h" I*/

#undef __FUNC__  
#define __FUNC__ "GridDataView" 
/*@
   GridDataView - Allows visualizing a griddata object.

   Collective on GridData unless Viewer is sequential

   Input Parameters:
+  griddata - the griddata
-  viewer - the viewer with which to view the griddata

.keywords: grid, visualize

.seealso: GridDataDestroy(), ViewerDrawOpenX(), VIEWER_STDOUT_WORLD,
          ViewerFileOpenASCII()
@*/
int GridDataView(GridData griddata,Viewer viewer)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(griddata,GRIDDATA_COOKIE);
  if (!viewer) {viewer = VIEWER_STDOUT_SELF; }
  else PetscValidHeader(viewer);

  ierr = (*griddata->ops->view)(griddata,viewer); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataRefine"
/*@
   GridDataRefine - Refine a grid uniformly by bisection.

   Collective on GridData

   Input Parameters:
+  griddata - the griddata 
-  l - number of levels of refinement to make

   Output Parameter:
.  newgrid - the refined grid

.keywords: grid, refinement

.seealso: GridDataDestroy()
@*/
int GridDataRefine(GridData griddata,int l,GridData *newgrid)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(griddata,GRIDDATA_COOKIE);
  PetscValidPointer(newgrid);
  if (!griddata->ops->refine) SETERRQ(PETSC_ERR_SUP,0,"");

  ierr = (*griddata->ops->refine)(griddata,l,newgrid); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataDestroy" 
/*@
   GridDataDestroy - Destroys a griddata object.

   Collective on GridData

   Input Parameters:
.  grid - the grid 

.keywords: grid, destroy
@*/
int GridDataDestroy(GridData griddata)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(griddata,GRIDDATA_COOKIE);
  if (--griddata->refct > 0) PetscFunctionReturn(0);
  ierr = (*griddata->ops->destroy)(griddata); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataCopy"
/*@
   GridDataCopy - Copies a grid.

   Collective on GridData

   Input Parameter:
.  g1 - the input grid

   Output Parameter:
.  g2 - the copy

.keywords: discrete function, vector, copy

.seealso: GridDataDuplicate()
@*/
int GridDataCopy(GridData x,GridData y)
{
  int ierr;

  PetscFunctionBegin;
  if (x->ops->copy) {ierr = (*x->ops->copy)(x,y); CHKERRQ(ierr);}
  else SETERRQ(PETSC_ERR_SUP,0,"");
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataDuplicate"
/*@
   GridDataDuplicate - Duplicates a grid data datastructure.

   Collective on GridData

   Input Parameter:
.  g1 - the input grid

   Output Parameter:
.  g2 - the duplicate copy

.keywords: discrete function, vector, copy

.seealso: GridDataCopy()
@*/
int GridDataDuplicate(GridData x,GridData *y)
{
  int ierr;

  PetscFunctionBegin;
  if (x->ops->duplicate) {ierr = (*x->ops->duplicate)(x,y); CHKERRQ(ierr);}
  else SETERRQ(PETSC_ERR_SUP,0,"");
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataSetUp"
/*@
   GridDataSetUp - Set up any required internal data structures for a grid.
   Should be called before GridGetDC().

   Collective on GridData

   Input Parameter:
.  g1 - the input grid

.keywords: grid

.seealso: GridDataDestroy()
@*/
int GridDataSetUp(GridData x)
{
  int ierr;

  PetscFunctionBegin;
  if (x->setupcalled) PetscFunctionReturn(0);
  if (x->ops->setup) {ierr = (*x->ops->setup)(x); CHKERRQ(ierr);}

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataCreateGrid"
/*@
   GridDataCreateGrid - Creates a grid context from the underlying grid information.

   Collective on GridData

   Input Parameters:
+  griddata - the griddata 
-  dtype - discretization type, for example DISCRETIZATION_LINEAR

   Output Parameter:
.  newgrid - the  grid

.keywords: grid, refinement

.seealso: GridDataDestroy()
@*/
int GridDataCreateGrid(GridData griddata,DiscretizationType dtype,Grid *newgrid)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(griddata,GRIDDATA_COOKIE);
  PetscValidPointer(newgrid);
  if (!griddata->ops->creategrid) SETERRQ(PETSC_ERR_SUP,0,"");

  ierr = (*griddata->ops->creategrid)(griddata,dtype,newgrid); CHKERRQ(ierr);
  PLogObjectParent(*newgrid,griddata);
  PetscFunctionReturn(0);
}
