#include "da.h"
#include "dfvec.h"
#include <math.h>

static char help[] = "Demonstrates opening and drawing a surface with VRML\n";

int main( int argc, char **argv )
{
  DA     da;
  DF     shell;
  double *x, *y, *z, r, rl;
  int    ict, xs, xm, ys, ym, zs, zm, i, j, k, l, ierr, flg;
  int    M = 11, N = 13, P = 7, nc = 1;
  Vec    global, local;
  Scalar *ga;

  PetscInitialize(&argc,&argv,(char*)0,help);

  ierr = OptionsGetInt(PETSC_NULL,"-M",&M,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-N",&N,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-P",&P,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-nc",&nc,&flg); CHKERRA(ierr);

  ierr = DACreate3d(PETSC_COMM_WORLD,DA_NONPERIODIC,DA_STENCIL_STAR,M,N,P,
                    PETSC_DECIDE,PETSC_DECIDE,PETSC_DECIDE,nc,1,PETSC_NULL,PETSC_NULL,
                    PETSC_NULL,&da); CHKERRA(ierr);
  ierr = DACreateGlobalVector(da,&global); CHKERRA(ierr);
  ierr = VecGetArray(global,&ga); CHKERRA(ierr);

  /* Set up the data for the surface plot */
  ierr = DAGetCorners(da,&xs,&ys,&zs,&xm,&ym,&zm); CHKERRA(ierr);
  x = (double *) PetscMalloc( M*sizeof(double) ); CHKPTRA(x);
  y = (double *) PetscMalloc( N*sizeof(double) ); CHKPTRA(y);
  z = (double *) PetscMalloc( P*sizeof(double) ); CHKPTRA(z);

  for (i=0; i<M; i++) x[i] = i*(1.0/(M-1));
  for (j=0; j<N; j++) y[j] = j*(1.0/(N-1));
  for (k=0; k<P; k++) z[k] = 0;
  ierr = DFVecGetDFShell(global,&shell); CHKERRA(ierr);
  ierr = DFShellSetCoordinates(shell,M,N,P,x,y,z); CHKERRA(ierr);

  /* Create a slightly different surface for each k */
  for (k=zs; k<zs+zm; k++) {
    for (i=xs; i<xs+xm; i++) {
      for(j=ys; j<ys+ym; j++) {
        ict = (i-xs + (j-ys)*xm + (k-zs)*xm*ym)*nc;
        r = sqrt( (x[i]-0.5) * (x[i]-0.5) + (y[j]-0.5) * (y[j]-0.5));
        for (l=0; l<nc; l++) {
          rl = r*(l+1.0);
	  ga[ict+l] = (rl == 0) ? 1.0 + k * 0.2 : sin(8.0*(k*0.2+1)*rl) / (8.0 * rl);
        }
      }
    }
  }
  ierr = VecRestoreArray(global,&ga); CHKERRA(ierr);
  ierr = DFVecDrawTensorSurfaceContoursVRML(global); CHKERRA(ierr);

  ierr = VecDestroy(global); CHKERRA(ierr);
  ierr = DACreateLocalVector(da,&local); CHKERRA(ierr);
  ierr = VecDestroy(local); CHKERRA(ierr);
  ierr = DADestroy(da); CHKERRA(ierr);
  PetscFree(x); PetscFree(y); PetscFree(z);
  PetscFinalize();
  return 0;
}



