/******************************************************************************
 *
 *    pfsd_lseek.c,v : move file pointer
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder
 *
 *    PFSLib (Parallel I/O on workstations)
 *
 *    PFSLib offers parallel access to files for a parallel application
 *    running on a cluster of workstations.
 *    It is intended but not restricted to be used in message passing
 *    applications based on PVM, NXLib, MPI, and other.
 *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.
 *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or
 *    modify it under the terms of the GNU Library General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS
 *    and/or modify them under the terms of the GNU General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *
 *    PFSLib is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Library General Public License and GNU General Public License 
 *    for more details.
 *
 *    You should have received a copy of the GNU Library General Public
 *    License and the GNU General Public License along with this
 *    library; if not, write to the Free 
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *    Contact to the authors:
 *
 *    electronic mail: pfslib@informatik.tu-muenchen.de
 *
 *    paper mail:      Prof. Dr. A. Bode
 *                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
 *                     Institut f"ur Informatik
 *                     Technische Universit"at M"unchen
 *                     80290 M"unchen
 *                     Germany
 *
 *    This project was partially funded by a research grant form Intel
 *    Corporation. 
 *
 *****************************************************************************/

/******************************************************************************
 *
 *  RCS Filename : pfsd_lseek.c,v
 *  RCS Date     : 1995/11/23 17:43:11
 *  RCS Revision : 1.6
 *  RCS Author   : lamberts
 *
 *  Authors: Stefan Lamberts, Christian R"oder
 *
 *****************************************************************************/
#ifndef lint
static char rcs_id[] = "pfsd_lseek.c,v 1.6 1995/11/23 17:43:11 lamberts Rel";
#endif


#include <rpc/rpc.h>
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#include "pfsd.h"
#include "pfsd_defines.h"
#include "pfsd_macros.h"
#include "pfslib_errno.h"

extern filetabel *_cftbl;

extern int _checkfh
#ifdef ANSI_C
(pfslib_fhdl fh, struct svc_req *rqstp);
#else
();
#endif /* ANSI_C */

extern void _init_syncop
#ifdef ANSI_C
(int _ftix);
#else
();
#endif /* ANSI_C */


/* ************************************************************************ */
/* pfsd_lseek_1(): the rpc called for moving filepointers                   */
/* ************************************************************************ */

lseekres *pfsd_lseek_1
#ifdef ANSI_C
(fpmoveargs *argp, struct svc_req *rqstp)
#else
(argp, rqstp)
fpmoveargs *argp;
struct svc_req *rqstp;
#endif /* ANSI_C */
{

  static lseekres result;

  int _ftix = argp->fh.vfd;     /* File table index */
  int _ctix = argp->fh.which;   /* Client table index */
  int _clc;                     /* Current client index */
  
  
  off_t oldendpos;
  off_t oldpos;
  off_t newpos;

  bool_t all_done;
  int    syncerr;
  bool_t syncok;
  
  
  /*
  xdr_free(xdr_lseekres, (char *)&result);
  */

  /*
   * check pfslib_fhdl     
   */

  if (_checkfh(argp->fh,rqstp) < 0 )
  {
    ERR_MACRO(lseekres_u,errno);
    return (&result);
  }

  /* LAMBO wait for asynchronous io operation to finish is not
   * necessary sinc the file position is updated in getioid() */

  switch (_MODE)
  {
  case M_UNIX:
    oldendpos = _CFTE.endoffs;  /* shared eof */
    oldpos = _CLNT.foffs;       /* own offsets */
    break;
  case M_RECORD:
    oldendpos = _CLNT.endoffs;  /* own eof */
    oldpos    = _CLNT.foffs;    /* own offsets */
    break;
  case M_LOG:
  case M_SYNC:
  case M_GLOBAL:
    oldendpos = _CFTE.endoffs;  /* shared eof */
    oldpos = _CFTE.shoffs;      /* shared offsets */
    break;
  default:
    /* LAMBO Fehler */
    break;
  }
  
  switch (argp->whence)
  {
  case SEEK_SET:
    newpos = argp->offset;
    break;
  case SEEK_CUR:
    newpos = oldpos + argp->offset;
    break;
  case SEEK_END:
    newpos = oldendpos + argp->offset;
    break;
  default:
    ERR_MACRO(lseekres_u,EPFSLINVAL);
    return (&result);
  }


  switch (_MODE)
  {
  case M_UNIX:                  /* non sync own offset */
    result.stat = RPC_OK;
    _CLNT.foffs = newpos;
    if (newpos > _CFTE.endoffs)
      _CFTE.endoffs = newpos;
    result.lseekres_u.pos = (u_int)newpos;
    break;

  case M_LOG:                   /* not sync shared offset */
    result.stat = RPC_OK;
    _CFTE.shoffs = newpos;
    if (newpos > _CFTE.endoffs)
      _CFTE.endoffs = newpos;
    result.lseekres_u.pos = (u_int)newpos;
    break;
    
  case M_SYNC:                  /* sync shared offset */
  case M_GLOBAL:                /* sync shared offset */
    switch (_CFTE.sync)
    {
    case PFSD_SYNC_NONE:         /* first of global operation */
      _init_syncop(_ftix);

      _CFTE.sync     = PFSD_SYNC_LSEEK;
      
    case PFSD_SYNC_LSEEK:

      _CLNT.syncdone = TRUE;
      _CLNT.transp   = rqstp->rq_xprt;
      _CLNT.foffs    = newpos;  /* save new position for later test */
      
      all_done = TRUE;
      syncok = TRUE;
      for (_clc=0; _clc < _CFTE.how_many; _clc++)
      {
        if (!_CNCL.syncdone)
        {
          all_done = FALSE;
          break;
        }
        else if (_CNCL.foffs != newpos)
        {
          syncerr = EPFSLMIXIO;
          syncok  = FALSE;
        }
      }
      if (all_done)
      {
        if (syncok)
        {
          result.stat = RPC_OK;
          _CFTE.shoffs = newpos; /* Set new position */
          if (newpos > _CFTE.endoffs)
            _CFTE.endoffs = newpos;
          result.lseekres_u.pos = (u_int)newpos;
        }
        else
        {
          ERR_MACRO(lseekres_u,syncerr);
        }          

        /* Send reply */
        for (_clc=0; _clc < _CFTE.how_many; _clc++)
        {
          if (!svc_sendreply(_CNCL.transp, xdr_lseekres, (char *)&result))
          {
            svcerr_systemerr(_CNCL.transp);
          }
        }
        _CFTE.sync = PFSD_SYNC_NONE;
      }
      return ((lseekres *)0); /* Delay reply or already sent */

    default:
      fprintf(stderr,"MISMATCHED SYNCHRONISING OPERATIONS");
      ERR_MACRO(lseekres_u,EPFSLMIXIO);
    }
    break;

  case M_RECORD:                /* sync own offset */
    switch (_CFTE.sync)
    {
    case PFSD_SYNC_NONE:         /* first of global operation */
      _init_syncop(_ftix);

      _CFTE.sync     = PFSD_SYNC_LSEEK;

      _CFTE.shoffs   = newpos;  /* save new position for later test */
      
    case PFSD_SYNC_LSEEK:
      _CLNT.syncdone = TRUE;
      _CLNT.transp   = rqstp->rq_xprt;

      /* Test new position  */
      if (newpos != _CFTE.shoffs)
      {
        _CLNT.syncok  = FALSE;
        _CLNT.syncerr = EPFSLMIXIO;
      }

      all_done = TRUE;
      syncok = TRUE;
      for (_clc = 0; _clc < _CFTE.how_many; _clc++)
      {
        if (!_CNCL.syncdone)
        {
          all_done = FALSE;
          break;
        }
        else if (!_CNCL.syncok)
        {                   /* There was an error */
          syncok  = FALSE;
          syncerr = _CNCL.syncerr;
        }
      }
      if (all_done)
      {
        if (syncok)
        {
          result.stat = RPC_OK;
          result.lseekres_u.pos = (u_int)newpos;
        }
        else
        {
          ERR_MACRO(lseekres_u,syncerr);
        }          

        /* Send reply and set new file position */
        for (_clc=0; _clc < _CFTE.how_many; _clc++)
        {
          if (syncok)
          {
            _CNCL.foffs = newpos; /* Set new position */
            if (newpos > _CFTE.endoffs)
              _CNCL.endoffs = newpos;
            if (newpos > _CFTE.endoffs)
              _CFTE.endoffs = newpos;
          }
          
          if (!svc_sendreply(_CNCL.transp, xdr_lseekres, (char *)&result))
          {
            svcerr_systemerr(_CNCL.transp);
          }
        }
        _CFTE.sync = PFSD_SYNC_NONE;
      }
      return ((lseekres *)0); /* Delay reply or already sent */
    default:
      fprintf(stderr,"MISMATCHED SYNCHRONISING OPERATIONS");
      ERR_MACRO(lseekres_u,EPFSLMIXIO);
    }
    break;
  default:
    /* LAMBO Fehler */
    break;
  }
  
  return(&result);
}
