/* *********************************************************************** *
 * 
 * SPMD example using PVM 3.3.x
 *
 * This PVM program perfoms a distributed integer matrix vector
 * multiplication. The data is read and written from ot to  a file using
 * different I/O modes. 
 *
 * *********************************************************************** */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>

#include <pvm3.h>
#include <pfslib.h>

int main(argc, argv)
int argc;
char **argv;
{
  int nproc;                    /* number of processes */
  int mynum;                    /* my number within the processes */
  int mytid;                    /* my task id */
  int *tids;                    /* array of task ids */
  int fd;                       /* file descriptor */
  int xdim, ydim;               /* X and Y dimension */
  int mylines;                  /* my number of lines of the matrix*/
  int *vector;                  /* the vector */
  int *matrix;                  /* my part of the matrix */
  int *result;                  /* my part of the result vector */
  int i,j;                      /* counter */

  if (argc != 4)
  {
    fprintf(stderr,"usage: %s pfsd_host number_of_processes filename\n",
            argv[0]);
    exit(1);
  }
  
  if ((nproc = atoi(argv[2])) <= 0)
  {
    fprintf(stderr,"usage: %s pfsd_host number_of_processes filename\n",
            argv[0]);
    exit(1);
  }
    
  /* allocate memory for the tids */
  if ((tids = (int *)malloc(sizeof(int)*nproc)) == NULL)
  {
    perror("malloc()");
    exit(1);
  }
  
  if ((mytid = pvm_mytid()) < 0) /* enroll in pvm */
    exit(1);
  
  tids[0] = pvm_parent();       /* find out if I am parent or child */

  if(tids[0] < 0)               /* I am the parent */
  {
    tids[0] = mytid;

    mynum = 0;                  /* my number is 0 */

    /* start up nproc-1 copies of myself */
    pvm_spawn(argv[0],
              &argv[1],
              PvmTaskDefault,  "",
              nproc-1,
              &tids[1]);

    /* multicast tids array to children */
    pvm_initsend(PvmDataDefault);
    pvm_pkint(tids, nproc, 1);
    pvm_mcast(&tids[1], nproc-1, 0);
  }
  else    /* I am a child */
  {
    pvm_recv(tids[0], 0);    /* receive tids array */
    pvm_upkint(tids, nproc, 1);

    /* Use the index in the tid array as rank */
    for(i=1; i<nproc ; i++)
      if( mytid == tids[i] )
      { mynum = i; break; }
  }

  printf("I am process %d of %d\n",mynum,nproc);
  fflush(stdout);

	pfslib_init(argv[1], nproc, mynum); /* Initialize PFSLib */

  printf("Initialized PFSLib\n");
  fflush(stdout);

  printf("Opening file %s\n",argv[3]);
  fflush(stdout);

  /* open input file with I/O mode M_GLOBAL */
	fd = gopen(argv[3], O_RDWR, M_GLOBAL,0644);

  printf("Opened file\n");

  /* read X and Y dimension */
  cread(fd,(char *)&xdim,sizeof(int));
  printf("Read X dimension: %d\n",xdim);
  fflush(stdout);
  
  cread(fd,(char *)&ydim,sizeof(int));
  printf("Read Y dimension: %d\n",ydim);
  fflush(stdout);
  
  /* Calculate number of matrix lines for this process */
  mylines = ydim/nproc;
  if ((ydim%nproc) > mynum) mylines++;

  printf("I have to process %d lines of the matrix\n",mylines);
  fflush(stdout);
  
  /* allocate memory for my part of the matrix */
  if ((matrix = (int *)malloc(sizeof(int)*mylines*xdim)) == NULL)
  { perror("malloc()"); exit(1); }

  setiomode(fd,M_SYNC);         /* set I/O mode to M_SYNC */
  
  /* Read my part of the matrix */
  cread(fd,(char *)matrix,sizeof(int)*mylines*xdim);
  
  printf("Read my part of the matrix\n");
  fflush(stdout);

  /* allocate memory for vector */
  if ((vector = (int *)malloc(sizeof(int)*xdim)) == NULL)
  { perror("malloc()"); exit(1); }
  
  setiomode(fd,M_GLOBAL);       /* set I/O mode to M_GLOBAL */

  /* read vector */
  cread(fd,(char *)vector,sizeof(int)*xdim);
  printf("Read vector\n");
  fflush(stdout);

  /* allocate memory for my part of the result vector */
  if ((result = (int *)malloc(sizeof(int)*mylines)) == NULL)
  { perror("malloc()"); exit(1); }
  
  /* Calculate my part of the result vector */
  for (i=0; i<mylines; i++)
  {
    result[i] = 0;
    for (j=0; j<xdim; j++)
      result[i] += (matrix[(i*xdim)+j] * vector[j]);
  }
  
  printf("Calculated my part of the result vector\n");
  fflush(stdout);

  setiomode(fd,M_SYNC);         /* set I/O mode to M_SYNC */

  /* Write my part of the result */
  cwrite(fd,(char *)result,sizeof(int)*mylines);

  printf("Written my part of the result vector\n");
  fflush(stdout);
  
	close(fd);                    /* close file */

  pvm_exit();                   /* program finished exit pvm */
  exit(0);
}


