/******************************************************************************
 *
 *    clnt_ioop.c,v : Input/Output operation
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder
 *
 *    PFSLib (Parallel I/O on workstations)
 *
 *    PFSLib offers parallel access to files for a parallel application
 *    running on a cluster of workstations.
 *    It is intended but not restricted to be used in message passing
 *    applications based on PVM, NXLib, MPI, and other.
 *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.
 *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or
 *    modify it under the terms of the GNU Library General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS
 *    and/or modify them under the terms of the GNU General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *
 *    PFSLib is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Library General Public License and GNU General Public License 
 *    for more details.
 *
 *    You should have received a copy of the GNU Library General Public
 *    License and the GNU General Public License along with this
 *    library; if not, write to the Free 
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *    Contact to the authors:
 *
 *    electronic mail: pfslib@informatik.tu-muenchen.de
 *
 *    paper mail:      Prof. Dr. A. Bode
 *                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
 *                     Institut f"ur Informatik
 *                     Technische Universit"at M"unchen
 *                     80290 M"unchen
 *                     Germany
 *
 *    This project was partially funded by a research grant form Intel
 *    Corporation. 
 *
 *****************************************************************************/

/******************************************************************************
 *
 *  RCS Filename : clnt_ioop.c,v
 *  RCS Date     : 1995/11/29 14:18:38
 *  RCS Revision : 1.18
 *  RCS Author   : lamberts
 *
 *  Authors: Stefan Lamberts, Christian R"oder
 *
 *****************************************************************************/
#ifndef lint
static char rcs_id[] = "clnt_ioop.c,v 1.18 1995/11/29 14:18:38 lamberts Rel";
#endif

#include <rpc/rpc.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/uio.h>
#include <errno.h>
#include <stdio.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <signal.h>
#ifdef SUN4
#include <memory.h>
#else
#include <string.h>
#endif /* SUN4 */
#include <stdlib.h>
#include <unistd.h>

#include "pfsd.h"
#include "clnt_defines.h"
#include "clnt_macros.h"
#include "pfslib_errno.h"

extern ioIDtabel _pfslib_ioidtbl[];
extern int _pfslib_maxfiles;
extern pfslib_fhdl * _pfslib_ftbl;
extern int _pfslib_clt_thr;
extern CLIENT *_pfslib_clnt;
extern char _pfslib_host[];
extern pfslib_ipaddr _pfslib_host_ipaddr;

extern void _vectordistribute
#ifdef ANSI_C
(char *buffer, struct iovec *iov, int iovcnt);
#else
();
#endif /* ANSI_C */

extern long _svc_ioop
#ifdef ANSI_C
(long ioID, char *buffer, CLIENT *clnt);
#else
();
#endif /* ANSI_C */

extern long _iod_ioop
#ifdef ANSI_C
(long ioID, char *buffer);
#else
();
#endif /* ANSI_C */

extern long _vectorlength
#ifdef ANSI_C
(struct iovec *iov, int iovcnt);
#else
();
#endif /* ANSI_C */

extern int _getioID
#ifdef ANSI_C
(int fd, int op, unsigned int iolen);
#else
();
#endif /* ANSI_C */

void _copy_child_shm
#ifdef ANSI_C
(long ioID)
#else
(ioID)
long ioID;
#endif /* ANSI_C */
{
  struct shmid_ds shmds;
  char *shmbuf;
  
  switch (_IDTE.op)
  {
  case READ_OP:
  case READV_OP:
    if ((shmbuf = (char *)shmat(_IDTE.shmid,(char *)0,0)) == (char *)-1)
    {
      perror("_copy_child_shm(): shmat()");
      exit (1);
    }
    if (_IDTE.op == READ_OP)
      memcpy(_IDTE.buffer,shmbuf,_IDTE.iolen);
    else
      _vectordistribute(shmbuf,_IDTE.iov,_IDTE.iovcnt);
    
    if (shmdt(shmbuf) < 0)
    {
      perror("iohandle(): shmdt()");
    }
    
    if (shmctl(_IDTE.shmid,IPC_RMID,&shmds) < 0)
    {
      perror("_copy_child_shm(): shmctl(..,IPC_RMID,..)");
    }    
    break;
  }
  return;
}

  
#ifndef NO_CHILDSIGNAL
/* ************************************************************************* */
/* In case of asynchronous IO with a forked child, the signal of the child   */
/* causes the client to take appropriate actions with the shared memory segm */
/* ************************************************************************* */
void _chld_handler
#ifdef ANSI_C
(int sig)
#else
(sig)
int sig;
#endif /* ANSI_C */
{
  
	int status;      /* status of the finished child */
	int pid;       /* pid of the finished child */

  int ioID;

  int tmperrno = errno;
  
	while ((pid = waitpid(-1,&status,WNOHANG)) != 0)
	{
		if ((pid < 0) && (errno == ECHILD))
			break;

    if (pid < 0)
    {
      perror("waitpid()");
      exit (1);
    }
    
    /* Find the ioID */
    for (ioID=0;ioID<MAXIOID;ioID++)
    {
      if (_IDTE.childpid == pid)
        break;
    }
      
    if (ioID >= MAXIOID)
    {
#ifdef DEBUG
      fprintf(stderr,
              "PFSLib: Couldn't find matching ioid for childprocess %d\n",pid);
#endif
      continue;
    }

    if (WIFSIGNALED(status) || WIFEXITED(status))
    {
      if ((!WIFSIGNALED(status)) && (WEXITSTATUS(status) == 0))
        _copy_child_shm(ioID);
      
      _IDTE.childpid  = -1;
      _IDTE.state     = IDT_CLTDONE;
      _IDTE.childexit = status;
      _IDTE.shmid     = -1;
    }
  }
  
  errno = tmperrno;

	return;
}
#endif /* ! NO_CHILDSIGNAL */


static int pfslib_ioop
#ifdef ANSI_C
(long ioID)
#else
(ioID)
long ioID;
#endif /* ANSI_C */
{
  
#ifndef NO_CHILDSIGNAL
#ifdef BSD_SIG
  int oldmask;
#else
  sigset_t set;
  sigset_t oset;
#endif /* BSD_SIG */
#endif /* ! NO_CHILDSIGNAL */

  if (_IDTE.csync)
  {
    /* Client works synchronously */

    if (_IDTE.ssync)
    {                           /* Server synchron */
      if ((_IDTE.reallen = _svc_ioop(ioID,_IDTE.buffer,_pfslib_clnt)) < 0)
        return (-1);
    }
    else if ((_IDTE.reallen = _iod_ioop(ioID,_IDTE.buffer)) < 0)
      return (-1);
    
    _IDTE.state = IDT_CLTDONE;

    return (0);
  }

  /* Client asynchron */
  switch (_IDTE.op)
  {
  case READV_OP:
  case READ_OP:
    /* get an shared memory segment for read operations */
    if ((_IDTE.shmid = shmget(IPC_PRIVATE,_IDTE.iolen,
                              0600|IPC_CREAT|IPC_EXCL)) < 0)
      return (-1);
    break;
  }
  
#ifndef NO_CHILDSIGNAL
#ifdef BSD_SIG
  /* Block SIGCHLD to set _IDTE */
  oldmask = sigblock(sigmask(SIGCHLD));
#else
  sigemptyset(&set);
  sigaddset(&set,SIGCHLD);
  
  sigprocmask(SIG_BLOCK,&set,&oset);

  sigemptyset(&set);
#endif /* BSD_SIG */
#endif /* ! NO_CHILDSIGNAL */

  if ((_IDTE.childpid = fork()) < 0 )
  {
    /* LAMBO should free shm segment */

#ifndef NO_CHILDSIGNAL
#ifdef BSD_SIG
    sigsetmask(oldmask);
#else
    sigprocmask(SIG_SETMASK,&oset,(sigset_t *)0);
#endif /* BSD_SIG */
#endif /* ! NO_CHILDSIGNAL */

    return (-1);
  }
  else if (_IDTE.childpid == 0)
  {
    /* Child process */
    char *buf;
    int ex = 0;
    CLIENT *cld_clnt;

    struct hostent *hp;
    struct sockaddr_in addr;
    int sockp;

    int sooptarg;

    int retry;
  
#ifdef CHILD_HALT
    int dbg = 1;
    while (dbg);
#endif

/*    fprintf(stderr,"Child process BEGIN\n"); */

    switch (_IDTE.op)
    {
    case READV_OP:
    case READ_OP:
      if ((buf = (char *)shmat(_IDTE.shmid,(char *)0,0))
          == (char *)-1)
      {
        perror("child process: _pfslib_ioop(): shmat()");
        exit (1);
      }
      break;
    }
    
    if (_IDTE.ssync)
    {                           /* Server synchron */

      memset(&addr, 0, sizeof(struct sockaddr_in));
      addr.sin_family = AF_INET;
      memcpy(&addr.sin_addr, &_pfslib_host_ipaddr,IPADDRLEN);
      addr.sin_port = htons(0);

      retry = -1;
      do
      {
        retry++;
        sockp = RPC_ANYSOCK;
        cld_clnt = clnttcp_create(&addr, PFSD, PFSD_VERS,
                                  &sockp, RPC_IOD_BUFSZ, RPC_IOD_BUFSZ);
      }
      while ((retry < CONNECT_RETRIES) &&
             (cld_clnt == NULL) && (errno == ETIMEDOUT));
      
      if (cld_clnt == NULL)
      {
        clnt_pcreateerror("clnttcp_create()");
        exit(1);
      }

      /* Set REUSEADDR */
      sooptarg = 1;
      setsockopt(sockp,SOL_SOCKET,SO_REUSEADDR,(char *)&sooptarg,
                 sizeof(sooptarg));
      /* TCP_NODELAY */
      sooptarg = 1;
      setsockopt(sockp,IPPROTO_TCP,TCP_NODELAY,(char *)&sooptarg,
                 sizeof(sooptarg));

      cld_clnt->cl_auth = _pfslib_clnt->cl_auth;
      
      if (_svc_ioop(ioID,buf,cld_clnt) != _IDTE.iolen) ex = 1;

      clnt_destroy(cld_clnt);
    }
    else if (_iod_ioop(ioID,buf) != _IDTE.iolen) ex = 1;

    switch (_IDTE.op)
    {
    case READV_OP:
    case READ_OP:
      if (shmdt(buf) < 0)
        perror("child process: pfslib_ioop: shmdt()");
      break;
    }
    
/*    fprintf(stderr,"Child process EXITS\n"); */

    exit (ex);
    /* End of child process */
    /* Now the interrupt handler chld_handler will be executed.
     * It takes the appropriate actions concerning the shm buffer and the
     * state of the ioid */
  }

#ifndef NO_CHILDSIGNAL
#ifdef BSD_SIG
  sigsetmask(oldmask);
#else
  sigprocmask(SIG_SETMASK,&oset,(sigset_t *)0);
#endif /* BSD_SIG */
#endif /* ! NO_CHILDSIGNAL */

  return (0);
}



long _ioopv
#ifdef ANSI_C
(int op, int fd, struct iovec *iov, int iovcnt, bool_t ccall)
#else
(op, fd, iov, iovcnt, ccall)
int op;
int fd;
struct iovec *iov;
int iovcnt;
bool_t ccall;
#endif /* ANSI_C */
{
	long	 ioID;
  unsigned int nbytes;
  
  fd -= VFDOFFSET;

  /* Check argument */
  if ((fd < 0) || (fd >= _pfslib_maxfiles) ||
      (_pfslib_ftbl[fd].vfd == -1))
  {
    errno = EPFSLBADF;
    return (-1);
  }

  nbytes = _vectorlength(iov,iovcnt);

  if ((ioID = _getioID(fd,op,nbytes)) < 0)
    return(-1);
  
  /* set values in iodtable */
  _IDTE.state  = IDT_PENDING;
  _IDTE.iov    = iov;
  _IDTE.iovcnt = iovcnt;
  
  if ((_IDTE.iolen < _pfslib_clt_thr) || ccall)
    _IDTE.csync = TRUE;
  else
    _IDTE.csync = FALSE;
  
  if (pfslib_ioop(ioID) < 0 )
    return(-1);

  return(ioID);
}




long _ioop
#ifdef ANSI_C
(int op, int fd, char *buffer, unsigned int nbytes, bool_t ccall)
#else
(op, fd, buffer, nbytes, ccall)
int op;
int fd;
char *buffer;
unsigned int nbytes;
bool_t ccall;
#endif /* ANSI_C */
{
	long	ioID;

  fd -= VFDOFFSET;

  /* Check argument */
  if ((fd < 0) || (fd >= _pfslib_maxfiles) ||
      (_pfslib_ftbl[fd].vfd == -1))
  {
    errno = EPFSLBADF;
    return (-1);
  }

  if ((ioID = _getioID(fd,op,nbytes)) < 0)
    return(-1);
  
  /* set values in iodtable */
  _IDTE.state = IDT_PENDING;
  _IDTE.buffer = buffer;
  
  if ((_IDTE.iolen < _pfslib_clt_thr) || ccall)
    _IDTE.csync = TRUE;
  else
    _IDTE.csync = FALSE;
  
  if (pfslib_ioop(ioID) < 0 )
    return(-1);

  return(ioID);
}
