/* ************************************************************************* *
 *                                                                           *
 *    pfsd_open.c,v
 *    pfsd procedures to open a PFSLib file
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_open.c,v
 *  RCS Date     : 1996/12/27 15:47:16
 *  RCS Revision : 1.6
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_open.c,v 1.6 1996/12/27 15:47:16 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "pfsd.h"
#include "pfsd_defines.h"
#include "pfsd_macros.h"
#include "pfslib_errno.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern filetabel *_cftbl;

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern int _iod_start _PH((char *ipaddr));
extern int _dist_del   _PH((int _ftix));
extern int _dist_init  _PH((int _ftix));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Open a file                                                               *
 * ************************************************************************* */
openres *pfsd_open_1
#ifdef ANSI_C
(openargs *argp, struct svc_req *rqstp)
#else  /* ANSI_C */
(argp, rqstp)
openargs *argp;
struct svc_req *rqstp;
#endif /* ANSI_C */
{

  static openres  result;

  int _ftix;                    /* File table index */
  int _ctix = argp->which;      /* Client table index */

  bool_t all_done;
  bool_t syncok;
  
  int          _clc;            /* counter for loop over clienttable */
  int          _cid;            /* counter for loop over IO-idtable */


  struct authunix_parms *unix_cred;

  struct sockaddr_in *sockadr;

  /*
   * xdr_free(xdr_openres, (char *)&result);
   */

  /* Test input parameter */

  if ((argp->which < 0) || (argp->which >= argp->how_many) ||
      (argp->how_many < 1))
  {
    ERR_MACRO(openres_u,EPFSLINVAL);
    return (&result);
  }
  

  if ((argp->oflags & O_NDELAY) || (argp->oflags & O_NONBLOCK))
  {
    /* We don't know yet how to support O_APPEND in a meaningfull way */
    ERR_MACRO(openres_u,EPFSLNDELAY);
    return(&result);
  }
  
  switch (argp->mode)
  {
  case M_UNIX:
  case M_LOG:
  case M_RECORD:
  case M_SYNC:
  case M_GLOBAL:
    break;
  default:
    ERR_MACRO(openres_u,EPFSLINVAL);
    return (&result);
  }
  

  switch (rqstp->rq_cred.oa_flavor)
  {
  case AUTH_UNIX:
    unix_cred = (struct authunix_parms *) rqstp->rq_clntcred;
    break;
  default:
    break;
  }
  sockadr = svc_getcaller(rqstp->rq_xprt);

  /*
   * has the file already been opened ?
   */

  for (_ftix=0; _ftix < MAXNUMFILES; _ftix++)
  {
    /* Look for a matching file in the filetable */
    if ((_CFTE.actual > 0) &&    /* element in use */
        (_CFTE.cltab != (CltTabEl *)0) && /* cttab allocated */
        (_CFTE.actual < _CFTE.how_many) && /* Missing processes */
        (_CFTE.uid == unix_cred->aup_uid) &&   /* unix user id */
        (_CFTE.gid == unix_cred->aup_gid) &&   /* unix group id */
        (strcmp(argp->filename,_CFTE.filename) == 0) && /* correct filename */
        (argp->oflags == _CFTE.oflags) && /* correct openflags */
        (argp->permission == _CFTE.permission) && /* correct permission */
        (argp->how_many == _CFTE.how_many) && /* correct client count */
        (_ctix < _CFTE.how_many) && /* correct which count */
        (!_CLNT.access) && /* not accessing already */
        /* If it's a gopen() it's a gopen by all others too*/
        (((argp->global) && (_CFTE.sync == PFSD_SYNC_OPEN) && 
          (!_CLNT.syncdone) &&  /* this process missing .. */
          (_CFTE.mode == argp->mode)) || /* .. and wants the same mode */
         ((!argp->global) && (_CFTE.sync != PFSD_SYNC_OPEN))))
    {
      /* OK this file is allready open just set the access and actual
         counters */

      _CFTE.actual++;
      _CLNT.access = TRUE;
      /* LAMBO Set client authetification */
      _CLNT.ident.pid = argp->pid;
      _CLNT.ident.uid = unix_cred->aup_uid;
      _CLNT.ident.gid = unix_cred->aup_gid;
      _CLNT.ident.aup_time = unix_cred->aup_time;
      strcpy(_CLNT.ident.hname,unix_cred->aup_machname);
      strcpy(_CLNT.ident.ipaddr,inet_ntoa(sockadr->sin_addr));

      /* Check whether ther is a IOD running on the clients machine 
       * and start one if not */
      if (_iod_start(_CLNT.ident.ipaddr) < 0)
      {
        _CLNT.syncok  = FALSE;
        _CLNT.syncerr = errno;
      }

      if (_CFTE.sync == PFSD_SYNC_OPEN)
      {
        _CLNT.syncdone = TRUE;
        _CLNT.transp = rqstp->rq_xprt;

        /* Check wheter all did the gopen() */
        all_done = TRUE;
        syncok = TRUE;
        for (_clc = 0; _clc < _CFTE.how_many; _clc++)
        {
          if (!_CNCL.syncdone)
          {
            all_done = FALSE;
            break;
          }
          else if (!_CNCL.syncok)
          {                   /* There was an error */
            syncok  = FALSE;
          }
        }
        if (all_done)
        {
          if (syncok)
          {
            result.stat = RPC_OK;
            result.openres_u.fh.vfd     =  _ftix;
            result.openres_u.fh.how_many = _CFTE.how_many;
            result.openres_u.fh.mode    =  _CFTE.mode;
          }
          else
          {
            ERR_MACRO(openres_u, _CLNT.syncerr);
          }

          /* Send reply */
          for (_clc=0; _clc < _CFTE.how_many; _clc++)
          {
            if (syncok) 
            {
              result.openres_u.fh.pid     = _CNCL.ident.pid;
              result.openres_u.fh.which   = _clc;
            }

            if (!svc_sendreply(_CNCL.transp, xdr_openres, (char *)&result))
            {
              svcerr_systemerr(_CNCL.transp);
            }
          }

          if (!syncok)
          {
            (void)_dist_del(_ftix);
            /* Reset file table */
            free(_CFTE.cltab);
            _CFTE.cltab = (CltTabEl *)0;
            _CFTE.actual = 0;
          }
          else
            _CFTE.sync = PFSD_SYNC_NONE;
        }
        return ((openres *)0); /* Reply was already sent or must be delayed */
      }        

      /* Not synchronizing */
      result.stat = RPC_OK;
      result.openres_u.fh.vfd      = _ftix;
      result.openres_u.fh.how_many = _CFTE.how_many;
      result.openres_u.fh.which    = _ctix;
      result.openres_u.fh.pid      = argp->pid;
      result.openres_u.fh.mode     = _CFTE.mode;
      
      return (&result);
    }
  }

  /* HA, there is no matching open file */

  /* Let'd find a free tbl element */

  for (_ftix = 0; _ftix < MAXNUMFILES; _ftix++)
    if (_CFTE.actual == 0)
      break;
  
  if (_ftix >= MAXNUMFILES)
  {
    ERR_MACRO(openres_u, EPFSLMFILE);
    return (&result);
  }
  
  /* Allocate the clttab */
  if ((_CFTE.cltab = (CltTabEl *)malloc(sizeof(CltTabEl)*argp->how_many))
      == (CltTabEl *)0)
  {
    perror("pfsd: pfsd_open_1(): malloc()");
    ERR_MACRO(openres_u, errno);
    return (&result);
  }

  /* Initialize the table entry */
  _CFTE.oflags     = argp->oflags;
  _CFTE.permission = argp->permission;
  _CFTE.uid = unix_cred->aup_uid;
  _CFTE.gid = unix_cred->aup_gid;
  strcpy(_CFTE.filename,argp->filename);
  if (argp->global)
    _CFTE.mode     = argp->mode;
  else
    _CFTE.mode     = M_UNIX;
  _CFTE.shoffs     = 0;
  _CFTE.how_many   = argp->how_many;
  _CFTE.actual     = 1;

  for (_clc=0; _clc < _CFTE.how_many; _clc++)
  {
    /* 
     * reset everything in the memory which might be there from previous calls
     */
    
    _CNCL.access   = FALSE;
    _CNCL.foffs    = 0;
    _CNCL.syncdone = FALSE;
    _CNCL.syncok   = TRUE;
    _CNCL.syncerr  = 0;
    _CNCL.transp   = (SVCXPRT *)0;

    _CNCL.ioseq    = 0;

    for (_cid=0; _cid<MAXIOID; _cid++)
    {
      _CNID.state  = ID_UNUSED;
      _CNID.ioseq  = 0;
      _CNID.oldmode = FALSE;
    }      
  }

  _CLNT.ident.pid = argp->pid;
  _CLNT.ident.uid = unix_cred->aup_uid;
  _CLNT.ident.gid = unix_cred->aup_gid;
  _CLNT.ident.aup_time = unix_cred->aup_time;
  strcpy(_CLNT.ident.hname, unix_cred->aup_machname);
  strcpy(_CLNT.ident.ipaddr,inet_ntoa(sockadr->sin_addr));


  _CLNT.access  = TRUE;

  if ((argp->global) && (argp->how_many > 1))
  {
    _CFTE.sync = PFSD_SYNC_OPEN;
    _CLNT.syncdone = TRUE;
    _CLNT.transp = rqstp->rq_xprt;
  }
  else
    _CFTE.sync = PFSD_SYNC_NONE;
  
  /* Check whether ther is a IOD running on the clients machine 
   * and start one if not */
  if (_iod_start(_CLNT.ident.ipaddr) < 0)
  {
    _CLNT.syncok = FALSE;
    _CLNT.syncerr = errno;
  }
  /* Read the distribution */
  else if (_dist_init(_ftix) < 0)
  {
    _CLNT.syncok  = FALSE;
    _CLNT.syncerr = errno;
  }
  else
  {  
    /* Set end of file offset */  
    for (_clc = 0; _clc < _CFTE.how_many; _clc++)
      _CNCL.endoffs = _CFTE.endoffs;
  }

  if (_CFTE.sync == PFSD_SYNC_OPEN) 
    return ((openres *)0);      /* Delay reply */
  
  if (!_CLNT.syncok)
  {
    ERR_MACRO(openres_u, _CLNT.syncerr);
    
    (void)_dist_del(_ftix);
    free(_CFTE.cltab);
    _CFTE.cltab = (CltTabEl *)0;
    _CFTE.actual = 0;
  
    return (&result);
  }
  
  result.stat = RPC_OK;
  result.openres_u.fh.vfd      = _ftix;
  result.openres_u.fh.how_many = _CFTE.how_many;
  result.openres_u.fh.which    = _ctix;
  result.openres_u.fh.pid      = argp->pid;
  result.openres_u.fh.mode     = _CFTE.mode;
  
  return(&result);
}

