/* ************************************************************************* *
 *                                                                           *
 *    pfsd_offs.c,v
 *    pfsd procedures to set the file offset for an IO operation
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_offs.c,v
 *  RCS Date     : 1996/12/27 15:47:15
 *  RCS Revision : 1.3
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_offs.c,v 1.3 1996/12/27 15:47:15 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>

#include "pfsd.h"
#include "pfsd_defines.h"
#include "pfsd_macros.h"
#include "pfslib_errno.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern filetabel *_cftbl;

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern DIST_LIST _dist_get_list _PH((int _ftix, int _ctix, int _ioID));
extern void      _dist_del_list _PH((DIST_LIST list));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Set offset in IO mode M_UNIX                                              *
 * ************************************************************************* */
void set_offset_unix
#ifdef ANSI_C
(int _ftix, int _ctix, int _ioID)
#else  /* ANSI_C */
(_ftix, _ctix, _ioID)
int _ftix;
int _ctix;
int _ioID;
#endif /* ANSI_C */
{
  switch (_IOID.op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    if (_CFTE.oflags & O_APPEND)
    {
      _IOID.offset = _CFTE.endoffs;
      _CFTE.endoffs += _IOID.iolen;
      _CLNT.foffs = _CFTE.endoffs;
    }
    else
    {
      _IOID.offset = _CLNT.foffs;
      _CLNT.foffs += _IOID.iolen;
      if (_CLNT.foffs > _CFTE.endoffs)
        _CFTE.endoffs = _CLNT.foffs;
    }
    break;
  case READ_OP:
  case READV_OP:
    _IOID.offset = _CLNT.foffs;
    if ((_CLNT.foffs + _IOID.iolen) > _CFTE.endoffs)
      /* Would read past EOF */
      _CLNT.foffs = _CFTE.endoffs;
    else
      _CLNT.foffs += _IOID.iolen;
    break;
  }
  _IOID.state = ID_OFSSET;
  return;
}

/* ************************************************************************* *
 * Set offset in IO mode M_LOG                                               *
 * ************************************************************************* */
void set_offset_log
#ifdef ANSI_C
(int _ftix, int _ctix, int _ioID)
#else  /* ANSI_C */
(_ftix, _ctix, _ioID)
int _ftix;
int _ctix;
int _ioID;
#endif /* ANSI_C */
{
  switch (_IOID.op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    if (_CFTE.oflags & O_APPEND)
    {
      _IOID.offset   = _CFTE.endoffs;
      _CFTE.endoffs += _IOID.iolen;
      _CFTE.shoffs   = _CFTE.endoffs;
    }
    else 
    {
      _IOID.offset = _CFTE.shoffs;
      _CFTE.shoffs += _IOID.iolen;
      if (_CFTE.shoffs > _CFTE.endoffs)
        _CFTE.endoffs = _CFTE.shoffs;
    }
    break;
  case READ_OP:
  case READV_OP:
    _IOID.offset = _CFTE.shoffs;
    if ((_CFTE.shoffs + _IOID.iolen) > _CFTE.endoffs)
      /* Would read past EOF */
      _CFTE.shoffs = _CFTE.endoffs;
    else
      _CFTE.shoffs += _IOID.iolen;
    break;
  }
  _IOID.state = ID_OFSSET;
  return;
}

/* ************************************************************************* *
 * Set offset in IO mode M_RECORD                                            *
 * ************************************************************************* */
void set_offset_record
#ifdef ANSI_C
(int _ftix, int _ctix, int _ioID)
#else  /* ANSI_C */
(_ftix, _ctix, _ioID)
int _ftix;
int _ctix;
int _ioID;
#endif /* ANSI_C */
{
  switch (_IOID.op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    if (_CFTE.oflags & O_APPEND)
    {
      _IOID.offset   = _CLNT.endoffs+(_ctix * _IOID.iolen);
      _CLNT.endoffs += (_CFTE.how_many * _IOID.iolen);
      _CLNT.foffs    = _CLNT.endoffs;
    }
    else 
    {
      _IOID.offset = _CLNT.foffs+(_ctix * _IOID.iolen);
      _CLNT.foffs += (_CFTE.how_many * _IOID.iolen);
      if (_CLNT.foffs > _CLNT.endoffs)
        _CLNT.endoffs = _CLNT.foffs;
    }
    if (_CLNT.foffs > _CFTE.endoffs)
      _CFTE.endoffs = _CLNT.foffs;
    break;
  case READ_OP:
  case READV_OP:
    _IOID.offset = _CLNT.foffs+(_ctix * _IOID.iolen);
    if ((_CLNT.foffs + (_CFTE.how_many * _IOID.iolen)) > _CLNT.endoffs)
      /* Would read past EOF */
      _CLNT.foffs = _CLNT.endoffs;
    else
      _CLNT.foffs += (_CFTE.how_many * _IOID.iolen);
    break;
  }
  _IOID.state = ID_OFSSET;
  return;
}

/* ************************************************************************* *
 * Set offset in IO mode M_GLOBAL                                            *
 * ************************************************************************* */
void set_offset_global
#ifdef ANSI_C
(int _ftix, int _ctix, int _ioID)
#else  /* ANSI_C */
(_ftix, _ctix, _ioID)
int _ftix;
int _ctix;
int _ioID;
#endif /* ANSI_C */
{
  int _clc;
  int _cid;
  int mixio;
  static getdistres result;
  
  /* Check whether all processes did the same call */

  mixio = FALSE;
  for(_clc = 0; _clc < _CFTE.how_many; _clc++)
  {
    /* Find matching ioID of other processes and check their arguments */
    for (_cid = 0; _cid < MAXIOID; _cid++)
    {
      if (((_CNID.state == ID_ALLOC) || (_CNID.state == ID_DISTPEND)) &&
          (!_CNID.oldmode) &&
          (_CNID.ioseq == _IOID.ioseq)) /* This is the matching id */
      {
        if ((_CNID.op != _IOID.op) || (_CNID.iolen != _IOID.iolen))
          mixio = TRUE;
        break;
      }
    }
      
    if (_cid >= MAXIOID)      /* Client clc didn't perform
                                 the call yet */
      return;
  }    
  /* all processes performed the call;
   * set the offset and the errorflag for EMIXIO if necessary */
  result.getdistres_u.res.oflags = ((_CFTE.oflags &
                                     (~(O_APPEND|O_EXCL|O_TRUNC))) |
                                    O_CREAT);
  result.getdistres_u.res.perm = _CFTE.permission;
  for (_clc = 0; _clc < _CFTE.how_many; _clc++)
  {
    for (_cid = 0; _cid < MAXIOID; _cid++)
    {
      if (((_CNID.state == ID_ALLOC) || (_CNID.state == ID_DISTPEND))&&
          (!_CNID.oldmode) &&
          (_CNID.ioseq == _IOID.ioseq)) /* This is the matching call */
      {
        if (mixio)
          _CNID.mixio = TRUE;
        else 
        {
          result.stat = RPC_OK;
          switch (_CNID.op)
          {
          case WRITE_OP:
          case WRITEV_OP:
            if (_CFTE.oflags & O_APPEND)
              _CNID.offset = _CFTE.endoffs;
            else
              _CNID.offset = _CFTE.shoffs;
            break;
            
          case READ_OP:
          case READV_OP:
            _CNID.offset = _CFTE.shoffs;
            break;
          }
        }
        if (_CNID.state == ID_ALLOC)
          _CNID.state = ID_OFSSET;
        else
        {
          if (mixio)
          {
            ERR_MACRO(getdistres_u,  EPFSLMIXIO);
          }
          else
          {
            result.stat = RPC_OK;
            switch (_CNID.op)
            {
            case WRITE_OP:
            case WRITEV_OP:
              if (_clc == 0)
                result.getdistres_u.res.list = _dist_get_list(_ftix,_clc,_cid);
              else
              {
                result.getdistres_u.res.list.DIST_LIST_len = 0;
                result.getdistres_u.res.list.DIST_LIST_val = NULL;
              }
              break;
            case READ_OP:
            case READV_OP:
              result.getdistres_u.res.list = _dist_get_list(_ftix,_clc,_cid);
              break;
            }
          }
          if (!svc_sendreply(_CNID.transp, xdr_getdistres, (char *)&result))
          {
            svcerr_systemerr(_CNID.transp);
          }
          if (result.stat == RPC_OK)
            _dist_del_list(result.getdistres_u.res.list);
          _CNID.state = ID_DISTDONE;
        }
        break;
      }
    }
  }
  
  switch (_IOID.op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    if (_CFTE.oflags & O_APPEND)
    {
      _CFTE.endoffs += _IOID.iolen;
      _CFTE.shoffs   = _CFTE.endoffs;
    }
    else
    {
      _CFTE.shoffs += _IOID.iolen;
      if (_CFTE.shoffs > _CFTE.endoffs)
        _CFTE.endoffs = _CFTE.shoffs;
    }
    break;
  case READ_OP:
  case READV_OP:
    if ((_CFTE.shoffs + _IOID.iolen) > _CFTE.endoffs)
      /* Would read past EOF */
      _CFTE.shoffs = _CFTE.endoffs;
    else
      _CFTE.shoffs += _IOID.iolen;
    break;
  }
  return;
}

/* ************************************************************************* *
 * Set offset in IO mode M_SYNC                                              *
 * ************************************************************************* */
void set_offset_sync
#ifdef ANSI_C
(int _ftix, int _ctix, int _ioID)
#else  /* ANSI_C */
(_ftix, _ctix, _ioID)
int _ftix;
int _ctix;
int _ioID;
#endif /* ANSI_C */
{    
  static getdistres result;
  
  int mixio = FALSE;
  int _clc;
  int _cid;
  off_t offs;

  for(_clc = 0; _clc < _CFTE.how_many; _clc++)
  {
    /* Find matching ioID of other processes and check their arguments */
    for (_cid = 0; _cid < MAXIOID; _cid++)
    {
      if (((_CNID.state == ID_ALLOC) || (_CNID.state == ID_DISTPEND)) &&
          (!_CNID.oldmode) &&
          (_CNID.ioseq == _IOID.ioseq)) /* This is the matching call */
      {
        if (_CNID.op != _IOID.op)
          mixio = TRUE;
        break;
      }
    }
      
    if (_cid >= MAXIOID)      /* Current client didn't perform
                                 the call yet */
      return;
  }
    
  /* Set offsets if all processes did the call */
  /* All previous iread or iwrite calls must have done this already;
     otherwise the incomming call would match to a different set.
     Hence, it's ok to update the file pointer without explicit waiting
     for other pending iread/iwrite calls */
  result.getdistres_u.res.oflags = ((_CFTE.oflags &
                                     (~(O_APPEND|O_EXCL|O_TRUNC))) |
                                    O_CREAT);
  result.getdistres_u.res.perm = _CFTE.permission;

  if (((_IOID.op == WRITE_OP) || (_IOID.op == WRITEV_OP)) &&
      (_CFTE.oflags & O_APPEND))
    offs = _CFTE.endoffs;
  else
    offs = _CFTE.shoffs;

  for (_clc = 0; _clc < _CFTE.how_many; _clc++)
  {
    for (_cid = 0; _cid < MAXIOID; _cid++)
    {
      if (((_CNID.state == ID_ALLOC) || (_CNID.state == ID_DISTPEND)) &&
          (!_CNID.oldmode) &&
          (_CNID.ioseq == _IOID.ioseq)) /* This is the matching call */
      {
        if (mixio)
          _CNID.mixio = TRUE;
        else
        {
          _CNID.offset = offs;
          offs += _CNID.iolen;
        }
        if (_CNID.state == ID_ALLOC)
          _CNID.state = ID_OFSSET;
        else 
        {
          if (mixio)
          {
            ERR_MACRO(getdistres_u, EPFSLMIXIO);
          }
          else
          {
            result.stat = RPC_OK;
            result.getdistres_u.res.list = _dist_get_list(_ftix,_clc,_cid);
          }
          
          if (!svc_sendreply(_CNID.transp, xdr_getdistres, (char *)&result))
          {
            svcerr_systemerr(_CNID.transp);
          }
          if (result.stat == RPC_OK)
            _dist_del_list(result.getdistres_u.res.list);
          _CNID.state = ID_DISTDONE;
        }
        break;
      }
    }
  }      
  switch (_IOID.op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    if (_CFTE.oflags & O_APPEND)
    {
      _CFTE.endoffs = offs;
      _CFTE.shoffs  = offs;
    }
    else
    {
      _CFTE.shoffs = offs;
      if (_CFTE.shoffs > _CFTE.endoffs)
        _CFTE.endoffs = _CFTE.shoffs;
    }
    break;
  case READ_OP:
  case READV_OP:
    if (offs > _CFTE.endoffs)
      _CFTE.shoffs = _CFTE.endoffs;
    else
      _CFTE.shoffs = offs;
    break;
  }
  return;
}
