/* ************************************************************************* *
 *                                                                           *
 *    pfsd_iod.c,v
 *    pfsd procedures for pfsd -- iod interaction
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_iod.c,v
 *  RCS Date     : 1996/12/27 15:47:11
 *  RCS Revision : 1.9
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_iod.c,v 1.9 1996/12/27 15:47:11 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#ifdef SUN4
#include <memory.h>
#endif /* SUN4 */

#include "iod.h"
#include "global.h"
#include "pfslib_errno.h"
#include "rpc_defaults.h"

/* ************************************************************************* *
 * Type definitions                                                          *
 * ************************************************************************* */

struct IODLIST
{
  pid_t           pid;
  char            host[MAXNLEN];
  char            ipaddr[IPADDRLEN];
  struct IODLIST *next;
};

typedef struct IODLIST IODLIST;
static IODLIST *iodlist = NULL;
static AUTH    *cl_auth = NULL;

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Delete an iod form the list of known iod processes                        *
 * ************************************************************************* */
static int iod_delete
#ifdef ANSI_C
(IODLIST *dle)
#else  /* ANSI_C */
(dle)
IODLIST *dle;
#endif /* ANSI_C */
{
  IODLIST *le;
  IODLIST *ple = NULL;
  
  for (le = iodlist; le != NULL; ple = le, le = le->next)
  {
    if (le == dle)
    {

#ifdef DEBUG
      fprintf(stdout,"pfsd: iod_delete(): Deleting iod on host %s\n",
              le->host);
#endif /* DEBUG */

      if (ple == NULL)
        iodlist = le->next;
      else
        ple->next = le->next;
      free(le);
      return (0);
    }
  }
  return (-1);
}

/* ************************************************************************* *
 * Handler for SIGCHLD signals                                               *
 * ************************************************************************* */
void _iod_sigchld_handler
#ifdef ANSI_C
(int signo)
#else  /* ANSI_C */
(signo)
int signo;
#endif /* ANSI_C */
{
  int state;
  int pid;
  IODLIST *le;

  while ((pid = waitpid(-1,&state,WNOHANG)) != 0)
  {
    if ((pid < 0) && (errno == ECHILD))
      break;

    if (pid < 0)
      perror("pfsd: __iod_sigchld_handler()");
    else
    {
      /* Get ascociated entry */
      for (le = iodlist; (le != NULL) && (le->pid != pid); le = le->next);

      if (WIFSIGNALED(state))
      {
        fprintf(stderr,"pfsd: rsh %d terminated by signal %d\n",
                pid, WTERMSIG(state));
        /* Check whether iod is still running */
        if (le != NULL)
        {
          if (callrpc(le->host, IOD, IOD_VERS, NULLPROC,
                      xdr_void, NULL, xdr_void, NULL) == RPC_SUCCESS)
          {
            /* Only set pid to 0 */
            le->pid = 0;
          }
          else
          {                       /* !RPC_SUCCESS */
            iod_delete(le);
          }
        }
      } /* if (WIFSIGNALED) */
      else if (WIFEXITED(state))
      {
        if (WEXITSTATUS(state) != 0)
          fprintf(stderr,"pfsd: rsh %d terminated unsuccessfully with status %d\n",
                  pid, WEXITSTATUS(state));
#ifdef DEBUG
        else
        {
          fprintf(stdout,"pfsd: rsh %d terminated successfully\n",
                  pid);
        }
#endif /* DEBUG */
        /* Check whether iod is still running */
        if (le != NULL)
        {
          if (callrpc(le->host, IOD, IOD_VERS, NULLPROC,
                      xdr_void, NULL, xdr_void, NULL) == RPC_SUCCESS)
          {
            /* Only set pid to 0 */
            le->pid = 0;
          }
          else
          {                     /* !RPC_SUCCESS */
            iod_delete(le);
          }
        }
      } /* if (WIFEXITED) */
      else if (WIFSTOPPED(state))
      {
        fprintf(stderr,"pfsd: rsh %d stopped by signal %d\n",
                pid,WSTOPSIG(state));
      }
      else 
      {
        fprintf(stderr,"pfsd: waitpid() returned unknown state for rsh %d\n",
                pid);
      }
    }
  }

  return;
}

/* ************************************************************************* *
 * Send exit requests to all iod processes                                   *
 * ************************************************************************* */
int _iod_exit
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  IODLIST *le;
  int res = 0;
  
  CLIENT *clnt;
  u_long ina;
  struct sockaddr_in addr;
  struct timeval ti;
  int sock;

  char arg;
  
#ifdef BSD_SIG
  int omask;
#else  /* BSD_SIG */
  sigset_t set;
  sigset_t oset;
#endif /* BSD_SIG */

  /* Block SIGCHLD to prevent iodlist to be altered while running through it */
#ifdef BSD_SIG
  omask = sigblock(sigmask(SIGCHLD));
#else  /* BSD_SIG */
  sigemptyset(&set);
  sigaddset(&set,SIGCHLD);
  sigprocmask(SIG_BLOCK,&set,&oset);
#endif /* BSD_SIG */

  for (le = iodlist; le != NULL; le = le->next)
  {
    ina = inet_addr(le->ipaddr);
  
    memset(&addr,0,sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = ina;
    addr.sin_port = htons(0);
    ti.tv_sec  = DFLTTIMEOUT;
    ti.tv_usec = 0;
    sock = RPC_ANYSOCK;

    if ((clnt = clntudp_create(&addr, IOD, IOD_VERS, ti, &sock)) == NULL)
    {
      clnt_pcreateerror("pfsd: _iod_exit(): clntudp_create()");
      res = -1;
    }
    else 
    {
      if (cl_auth == NULL)
        cl_auth = authunix_create_default();

      clnt->cl_auth = cl_auth;

#ifdef DEBUG
      fprintf(stdout,"pfsd: _iod_exit(): Sending exit request to iod on %s\n",
              le->host);
#endif /* DEBUG */

      if (iod_exit_1(&arg,clnt) == NULL)
      {
        clnt_perror(clnt,"pfsd: _iod_exit(): iod_exit_1()");
        res = -1;
      }
    
      clnt_destroy(clnt);
      close(sock);
    }
  }
  
#ifdef BSD_SIG
  sigsetmask(omask);
#else  /* BSD_SIG */
  sigprocmask(SIG_SETMASK,&oset,NULL);
#endif /* BSD_SIG */

  return (res);
}

/* ************************************************************************* *
 * Send a signal to all iod processes                                        *
 * ************************************************************************* */
int _iod_signal
#ifdef ANSI_C
(int sig)
#else  /* ANSI_C */
(sig)
int sig;
#endif /* ANSI_C */
{
  IODLIST *le;
  int res = 0;
  
#ifdef BSD_SIG
  int omask;
#else  /* BSD_SIG */
  sigset_t set;
  sigset_t oset;
#endif /* BSD_SIG */

  /* Block SIGCHLD to prevent iodlist to be altered while running through it */
#ifdef BSD_SIG
  omask = sigblock(sigmask(SIGCHLD));
#else  /* BSD_SIG */
  sigemptyset(&set);
  sigaddset(&set,SIGCHLD);
  sigprocmask(SIG_BLOCK,&set,&oset);
#endif /* BSD_SIG */

  for (le=iodlist;le != NULL; le= le->next)
  {
    if (le->pid != 0)
    {
      if (kill(le->pid,sig) < 0)
      {
        perror("pfsd: _iod_signal(): kill()");
        res = -1;
      }
    }
  }
  
#ifdef BSD_SIG
  sigsetmask(omask);
#else  /* BSD_SIG */
  sigprocmask(SIG_SETMASK,&oset,NULL);
#endif /* BSD_SIG */

  return (res);
}

/* ************************************************************************* *
 * Start an iod on a machine if there is none                                *
 * ************************************************************************* */
int _iod_start
#ifdef ANSI_C
(char *ipaddr)
#else  /* ANSI_C */
(ipaddr)
char *ipaddr;
#endif /* ANSI_C */
{
  struct hostent *hp;
  IODLIST *le;
  u_long ina;
  int strc = 1;
  int res = 0;
#ifndef NO_IOD_START_CHK
  int retry;
  enum clnt_stat rpcres;
#endif /* NO_IOD_STARTX_CHK */

#ifdef IODBG
  int state;
#endif /* IODBG */
  
#ifdef BSD_SIG
  int omask;
#else  /* BSD_SIG */
  sigset_t set;
  sigset_t oset;
#endif /* BSD_SIG */

  /* Block SIGCHLD to prevent iodlist to be altered while running through it */
#ifdef BSD_SIG
  omask = sigblock(sigmask(SIGCHLD));
#else  /* BSD_SIG */
  sigemptyset(&set);
  sigaddset(&set,SIGCHLD);
  sigprocmask(SIG_BLOCK,&set,&oset);
#endif /* BSD_SIG */

  /* Check whether there exists a iod on the remote machine */
  for (le = iodlist;
       (le != NULL) && ((strc = strcmp(ipaddr,le->ipaddr)) != 0);
       le = le->next);

  if (strc != 0)
  {
    /* There is no entry; create a new one */
    ina = inet_addr(ipaddr);
  
    if ((hp = gethostbyaddr(&ina,sizeof(ina),AF_INET)) == NULL)
    {
      perror("pfsd: _iod_start(): gethostbyaddr");
      res = -1;
    }
    else if ((le = (IODLIST *)malloc(sizeof(IODLIST))) == NULL)
    {
      perror("pfsd: _iod_start(): malloc()");
      res = -1;
    }
    else
    {
      strcpy(le->host,hp->h_name);
      strcpy(le->ipaddr,ipaddr);
      le->pid = 0;

      if (iodlist == NULL)
        le->next = NULL;
      else
        le->next = iodlist;
      iodlist = le;
    }
  }
#ifdef NO_IOD_START_CHK
  else
  {                             /* There is an entry for this machine */
#ifdef BSD_SIG
    sigsetmask(omask);
#else  /* BSD_SIG */
    sigprocmask(SIG_SETMASK,&oset,NULL);
#endif /* BSD_SIG */

    return (res);
  }
#endif /* NO_IOD_START_CHK */
  
  if (res == 0)
  {
#ifndef NO_IOD_START_CHK

    /* Check if the iod is running */
    if (callrpc(le->host, IOD, IOD_VERS, NULLPROC,
                xdr_void, NULL, xdr_void, NULL) != RPC_SUCCESS)
    {                           /* rpc unsuccessful: iod dos not exists */
#endif /* !NO_IOD_START_CHK */
      
      if (le->pid != 0)
        kill(le->pid,SIGKILL);  /* Kill asociated rsh process */

      /* There is no iod start a new one */
      if ((le->pid = fork()) < 0)
      {
        perror("pfsd: _iod_start(): fork()");
        iod_delete(le);
        res = -1;
      }
      else if (le->pid == 0)
      {                     /* Child process */
#ifdef DEBUG
        fprintf(stdout,"pfsd: _iod_start(): Starting iod on host %s\n",
                le->host);
#endif /* DEBUG */

        /* Start iod with rsh */
        execl(RSH_PATH,
              RSH_PATH,
#ifdef SUN4
              le->host,
#else  /* SUN4 */
              le->ipaddr,
#endif /* SUN4 */
              RSH_OPTIONS,
              IOD_EXEC_PATH,
              (char *)0);
        /* NEVER RETURNED */
        perror("pfsd: _iod_start(): execl()");
        exit(1);
      } /* Child process */
      else
      {                     /* Parent */

#ifdef IODBG
        /* Wait for the rsh process to terminate.
           I.e. iod went into background */
        if (waitpid(le->pid,&state,0) < 0)
        {
          perror("pfsd: _iod_start(): waitpid()");
          kill(le->pid,SIGKILL);  /* Kill rsh process */
          iod_delete(le);
          res = -1;
        }
        else 
        {                   /* waitpid ok */
          if (WIFSIGNALED(state))
          {
            fprintf(stderr,"pfsd: rsh %d terminated by signal %d\n",
                    le->pid, WTERMSIG(state));
            /* nevertheless iod prcocess may be running */
            le->pid = 0;
          }
          else if (WIFEXITED(state))
          {
            if (WEXITSTATUS(state) != 0)
            {
              fprintf(stderr,"pfsd: rsh %d terminated unsuccessfully with status %d\n",
                      le->pid, WEXITSTATUS(state));
              /* nevertheless iod prcocess may be running */
            }
#ifdef DEBUG
            else
            {               /* WEXITSTATUS == 0 */
              fprintf(stdout,"pfsd: rsh %d terminated successfully\n",
                      le->pid);
            }
#endif /* DEBUG */
            le->pid = 0;
          } /* WIFEXITED */
          else if (WIFSTOPPED(state))
          {
            fprintf(stderr,"pfsd: rsh %d stopped by signal %d\n",
                    le->pid,WSTOPSIG(state));
          }
          else
          {                   /* !WIFSIGNALED && !WIFSTOPPED */
            fprintf(stderr,"pfsd: waitpid() returned unknown state for rsh %d\n",
                    le->pid);
          }
        } /* waitpid ok */
#endif /* IODBG */

#ifndef NO_IOD_START_CHK
        if (res == 0)       /* might be -1 due to waitpid */
        {

          /* wait for iod to start */
          for (retry=0; retry < IODSTART_RPCRETRY; retry++)
          {
            if ((rpcres = callrpc(hp->h_name, IOD, IOD_VERS, NULLPROC,
                                  xdr_void, NULL, xdr_void, NULL))
                == RPC_SUCCESS)
              break;
            else switch (rpcres)
            {
            case RPC_SUCCESS:
              break;
            case RPC_TIMEDOUT:
            case RPC_FAILED:
#ifndef HPPA
            case RPC_INTR:
#endif /* !HPPA */
              break;
            case RPC_PROGUNAVAIL:
            case RPC_PMAPFAILURE:
            case RPC_PROGNOTREGISTERED:
              sleep(1);
              break;
            case RPC_CANTENCODEARGS:
            case RPC_CANTDECODERES:
            case RPC_CANTSEND:
            case RPC_CANTRECV:
            case RPC_VERSMISMATCH:
            case RPC_AUTHERROR:
            case RPC_PROGVERSMISMATCH:
            case RPC_PROCUNAVAIL:
            case RPC_CANTDECODEARGS:
            case RPC_SYSTEMERROR:
            case RPC_UNKNOWNHOST:
            case RPC_UNKNOWNPROTO:
              /* Don't try again */
              retry = IODSTART_RPCRETRY;
              break;
            }
          } /* for(retry=0; retry< IODSTART_RPCRETRY; retry++) */
                 
          if (retry >= IODSTART_RPCRETRY)
          {                 /* iod didn't start */
            if (le->pid != 0)
              kill(le->pid,SIGKILL);  /* Kill rsh process */
            iod_delete(le);
            errno = EPFSLNOTIOD;
            res = -1;
          }
        } /* if (res == 0) */
#endif /* !NO_IOD_START_CHK */
      } /* Parent */
#ifndef NO_IOD_START_CHK
    } /* rpc unsuccessful */
#endif /* !NO_IOD_START_CHK */
  } /* if (res == 0) */
  
#ifdef BSD_SIG
  sigsetmask(omask);
#else  /* BSD_SIG */
  sigprocmask(SIG_SETMASK,&oset,NULL);
#endif /* BSD_SIG */

  return (res);
}


/* ************************************************************************* *
 * Send an unlink request to an iod                                          *
 * ************************************************************************* */
int _iod_unlink
#ifdef ANSI_C
(char *ipaddr, char *filename)
#else  /* ANSI_C */
(ipaddr, filename)
char *ipaddr;
char *filename;
#endif /* ANSI_C */
{
  CLIENT *clnt;
  u_long ina;
  struct sockaddr_in addr;
  struct timeval ti;
  int sock = RPC_ANYSOCK;
  iod_unlinkargs arg;
  iod_errorres  *result;
  
  arg.fn     = filename;
  
  ina = inet_addr(ipaddr);
  
  memset(&addr,0,sizeof(struct sockaddr_in));
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = ina;
  addr.sin_port = htons(0);
  ti.tv_sec  = DFLTTIMEOUT;
  ti.tv_usec = 0;
 
  if ((clnt = clntudp_create(&addr, IOD, IOD_VERS, ti, &sock)) == NULL)
  {
    clnt_pcreateerror("pfsd: _iod_unlink(): clntudp_create()");
    return (-1);
  }

  if (cl_auth == NULL)
    cl_auth = authunix_create_default();
  
  clnt->cl_auth = cl_auth;

  if ((result = iod_unlink_1(&arg,clnt)) == NULL)
  {
    clnt_perror(clnt,"pfsd: _iod_unlink(): iod_unlink_1()");
    clnt_destroy(clnt);
    close(sock);
    return (-1);
  }
  
  clnt_destroy(clnt);
  close(sock);

  if (result->stat == RPC_ERR)
  {
    errno = EPFSLREMOTE;
    fprintf(stderr,"pfsd: _iod_unlink():%s",result->iod_errorres_u.err.errstr);
    xdr_free(xdr_iod_errorres,(char *)result);
    return (-1);
  }
  
  xdr_free(xdr_iod_errorres,(char *)result);
  return (0);
}

/* ************************************************************************* *
 * Send a truncate request to an iod                                         *
 * ************************************************************************* */
int _iod_trunc
#ifdef ANSI_C
(char *ipaddr, char *filename, long offset)
#else  /* ANSI_C */
(ipaddr, filename, offset)
char *ipaddr;
char *filename;
long offset;
#endif /* ANSI_C */
{
  CLIENT *clnt;
  u_long ina;
  struct sockaddr_in addr;
  struct timeval ti;
  int sock = RPC_ANYSOCK;
  iod_truncargs arg;
  iod_errorres  *result;
  
#ifdef DEBUG
  fprintf(stdout,"_iod_trunc(%s,%s,%ld)\n",ipaddr,filename,offset);
#endif /* DEBUG */

  arg.fn     = filename;
  arg.offset = offset;
  
  ina = inet_addr(ipaddr);
  
  memset(&addr,0,sizeof(struct sockaddr_in));
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = ina;
  addr.sin_port = htons(0);
  ti.tv_sec  = DFLTTIMEOUT;
  ti.tv_usec = 0;
 
  if ((clnt = clntudp_create(&addr, IOD, IOD_VERS, ti, &sock)) == NULL)
  {
    clnt_pcreateerror("pfsd: _iod_trunc(): clntudp_create()");
    return (-1);
  }

  if (cl_auth == NULL)
    cl_auth = authunix_create_default();

  clnt->cl_auth = cl_auth;

  if ((result = iod_trunc_1(&arg,clnt)) == NULL)
  {
    clnt_perror(clnt,"pfsd: _iod_trunc(): iod_trunc_1()");
    clnt_destroy(clnt);
    close(sock);
    return (-1);
  }
  
  clnt_destroy(clnt);
  close(sock);

  if (result->stat == RPC_ERR)
  {
    errno = EPFSLREMOTE;
    fprintf(stderr,"pfsd: _iod_trunc():%s",result->iod_errorres_u.err.errstr);
    xdr_free(xdr_iod_errorres,(char *)result);
    return (-1);
  }
  
  xdr_free(xdr_iod_errorres,(char *)result);
  return (0);
}

/* ************************************************************************* *
 * Print information on iod processes                                        *
 * ************************************************************************* */
void _iod_prtlist 
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  IODLIST *le;

#ifdef BSD_SIG
  int omask;
#else  /* BSD_SIG */
  sigset_t set;
  sigset_t oset;
#endif /* BSD_SIG */

  /* Block SIGCHLD to prevent iodlist to be altered while running through it */
#ifdef BSD_SIG
  omask = sigblock(sigmask(SIGCHLD));
#else  /* BSD_SIG */
  sigemptyset(&set);
  sigaddset(&set,SIGCHLD);
  sigprocmask(SIG_BLOCK,&set,&oset);
#endif /* BSD_SIG */

  printf("IODLIST\n");
  
  for (le = iodlist; le != NULL; le = le->next)
  {
    printf("IODTABLE ELEMENT\n");
    printf("\thost\t: %s\n",le->host);
    printf("\tipaddr\t: %s\n",le->ipaddr);
    printf("\tpid\t: %d\n",(int)le->pid);
  }
  printf("\n");
  return;
}

