/* ************************************************************************* *
 *                                                                           *
 *    pfsd_init.c,v
 *    pfsd procedures to initialize a pfsd process
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_init.c,v
 *  RCS Date     : 1996/12/27 15:47:09
 *  RCS Revision : 1.9
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_init.c,v 1.9 1996/12/27 15:47:09 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#ifndef RS6K
#include <sys/time.h>
#include <sys/resource.h>
#endif /* ! RS6K */
#include <signal.h>
#include <sys/wait.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>

#include "pfsd.h"
#include "pfsd_defines.h"
#include "pfsd_macros.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern int  _iod_exit             _PH((void));
extern int  _iod_signal           _PH((int sig));
extern void _iod_sigchld_handler  _PH((int signo));
extern int  _dist_del             _PH((int _ftix));
extern int  _iod_start            _PH((char *ipaddr));

#undef _PH

/* ************************************************************************* *
 * Global variables
 * ************************************************************************* */

filetabel *_cftbl;

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Send an exits request to all iod processes                                *
 * ************************************************************************* */
static void exit_iods
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  _iod_exit();
}

/* ************************************************************************* *
 * Send a SIGTERM to all iod processes.                                      *
 * ************************************************************************* */
static void kill_iods
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  _iod_signal(SIGTERM);
}

/* ************************************************************************* *
 * Unregister at the portmapper                                              *
 * ************************************************************************* */
void unregister
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{

#ifdef DEBUG
  fprintf(stdout,"pfsd: unregister(): unregistering at portmapper\n");
#endif /* DEBUG */

  svc_unregister(PFSD,PFSD_VERS);
}  

/* ************************************************************************* *
 * Signal handler for all signals causing a process termination              *
 * ************************************************************************* */
void _pfslib_sighandler
#ifdef ANSI_C
(int signo)
#else  /* ANSI_C */
(signo)
int signo;
#endif /* ANSI_C */
{
#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction  sigact;
#endif

  int _ftix;

  fprintf(stderr,"pfsd: _pfslib_sighandler(): Received signal: exiting\n");

  /* Reset the signal handler */
#ifdef BSD_SIG
  vec.sv_handler = void(*)()0;
  vec.sv_mask = 0;
  vec.sv_flags = SV_RESETHAND;
  if (sigvec(signo, &vec, (struct sigvec *)0) != 0)
  {
    perror("pfsd: _pfslib_sighandler(): sigvec()");
    exit(1);
  }
#else  /* ! BSD_SIG */
  sigact.sa_handler = SIG_DFL;
  sigact.sa_flags = 0;
  if (sigaction(signo, &sigact, (struct sigaction *)0) != 0)
  {
    perror("pfsd: _pfslib_sighandler(): sigaction()");
    exit(1);
  }
#endif /* BSD_SIG */

  /* Try to save open files */
  for (_ftix=0; _ftix<MAXNUMFILES; _ftix++)
  {
    if (_CFTE.actual > 0)
    {
      (void) _dist_del(_ftix);
      free(_CFTE.cltab);
      _CFTE.cltab = (CltTabEl *)0;
      _CFTE.actual = 0;
    }
  }
  
  exit_iods();                  /* Send exit requests to iods */
  kill_iods();                  /* Kill iods */
  unregister();                 /* Unregister at portmapper */
  
  kill(getpid(),signo);         /* Send this signal again */
}

/* ************************************************************************* *
 * Activate the signal handler for all fatal signals                         *
 * ************************************************************************* */
static void chatch_sigs
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  
#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction  sigact;
#endif
  int i;
  int sig[] =
  {
    SIGHUP,
    SIGINT,
    SIGQUIT,
    SIGILL,
    SIGTRAP,
    SIGABRT,
#ifndef LINUX
    SIGEMT,
#else
    SIGUNUSED,
#endif
    SIGFPE,
#ifndef LINUX
    SIGBUS,
#endif
    SIGSEGV,
#ifndef LINUX
    SIGSYS,
#endif
    SIGPIPE,
    SIGALRM,
    SIGTERM,
#ifdef LINUX
    SIGSTKFLT,
#endif
#ifndef HPPA
    SIGXCPU,
    SIGXFSZ,
#endif
    SIGVTALRM,
    SIGPROF,
    SIGUSR1,
    SIGUSR2,
#ifdef RS6K
    SIGMIGRATE,
    SIGPRE,
#endif
    -1
  };
  
  for (i=0; sig[i] > 0; i++)
  {
#ifdef BSD_SIG
    vec.sv_handler = _pfslib_sighandler;
    vec.sv_mask = sigmask(sig[i]);
    vec.sv_flags = 0;
    if (sigvec(sig[i], &vec, (struct sigvec *)0) != 0)
    {
      perror("pfsd: chatch_sigs(): sigvec()");
      exit(1);
    }
#else  /* ! BSD_SIG */
    sigact.sa_handler = _pfslib_sighandler;
    sigact.sa_flags = 0;
    if ((sigemptyset(&sigact.sa_mask) != 0) ||
        (sigaddset(&sigact.sa_mask,sig[i]) != 0) ||
        (sigaction(sig[i], &sigact, (struct sigaction *)0) != 0))
    {
      perror("pfsd: chatch_sigs():sigaction()");
      exit(1);
    }
#endif /* BSD_SIG */
  }
}

/* ************************************************************************* *
 * Initialize pfsd                                                           *
 * ************************************************************************* */
void _pfsd_init
#ifdef ANSI_C
(int argc, char **argv)
#else  /* ANSI_C */
(argc, argv)
int argc;
char **argv;
#endif /* ANSI_C */
{
#ifndef RS6K
  struct rlimit      pfsd_fd_limit, fd_curlim;
#endif /* ! RS6K */

  int _ftix;
  int i;

#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction  sigact;
#endif

  struct hostent *hp;
  char ipaddr[IPADDRLEN];
  
  /* Setup SIGCHLD handler */
#ifdef BSD_SIG
  vec.sv_handler = _iod_sigchld_handler;
  vec.sv_mask = sigmask(SIGCHLD);
  vec.sv_flags = 0;
  if (sigvec(SIGCHLD, &vec, (struct sigvec *)0) != 0)
  {
    perror("pfsd: _pfsd_init(): sigvec()");
    exit(1);
  }
#else  /* ! BSD_SIG */
  sigact.sa_handler = _iod_sigchld_handler;
  sigact.sa_flags = 0;
  if ((sigemptyset(&sigact.sa_mask) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGCHLD) != 0) ||
      (sigaction(SIGCHLD, &sigact, (struct sigaction *)0) != 0))
  {
    perror("pfsd: _pfsd_init():sigaction()");
    exit(1);
  }
#endif /* BSD_SIG */

  /* Set file descriptor limit to maximum */
#ifndef RS6K
  if (getrlimit(RLIMIT_NOFILE, &fd_curlim) < 0 )
  {
    perror("pfsd: _pfsd_init(): getrlimit()");
    exit(1);
  }

  pfsd_fd_limit.rlim_cur = fd_curlim.rlim_max;
  pfsd_fd_limit.rlim_max = fd_curlim.rlim_max;

  if (setrlimit(RLIMIT_NOFILE,&pfsd_fd_limit) < 0 )
  {
    perror("pfsd: _pfsd_init(): setrlimit()");
    exit(1);
  }
#endif /* ! RS6K */

  /* allocate memory for _cftbl (static during runtime) */
  if ((_cftbl = (filetabel *)malloc(sizeof(filetabel)*MAXNUMFILES ))
      == NULL )
  {
    perror("pfsd: _pfsd_init(): _malloc()");
    exit(1);
  }

  for (_ftix=0; _ftix<MAXNUMFILES; _ftix++)
  {
    _CFTE.actual = 0;
    _CFTE.cltab  = (CltTabEl *)0;
    _CFTE.distd  = NULL;
  }

#ifdef SUN4
  on_exit(unregister);
  on_exit(kill_iods);
  on_exit(exit_iods);
#else  /* SUN4 */
  atexit(unregister);
  atexit(kill_iods);
  atexit(exit_iods);
#endif /* SUN4 */

  /* Start iod processes */
  for (i=1; i< argc; i++)
  {
    if ((hp = gethostbyname(argv[i])) == NULL)
      perror("pfsd: _pfsd_init(): gethostbyname()");
    else
    {
      strcpy(ipaddr,inet_ntoa(*(struct in_addr *)hp->h_addr));
      if (_iod_start(ipaddr) < 0)
      {
        perror("pfsd: _pfsd_init(): _iod_start()");
      }
#ifdef DEBUG
      else
      {
        fprintf(stdout,"pfsd: _pfsd_init(): started iod on %s\n",argv[i]);
      }
#endif /* DEBUG */
    }
  }
  
  /* Setup signal handler */
  chatch_sigs();
  
#ifdef DEBUG
  fprintf(stdout,"pfsd: _pfsd_init(): Done\n");
#endif /* DEBUG */

  return;
}






