/* ************************************************************************* *
 *                                                                           *
 *    pfsd_frl.c,v
 *    pfsd procedures to manage Free Region Lists for distributed files
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_frl.c,v
 *  RCS Date     : 1996/05/02 15:25:59
 *  RCS Revision : 1.4
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_frl.c,v 1.4 1996/05/02 15:25:59 lamberts V2_0_B";
#endif


#ifdef TIGHT_FILES

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "global.h"
#include "pfsd_frl.h"
#include "pfslib_errno.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern int _iod_trunc _PH((char *ipaddr, char *filename, long offset));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Free region list                                                          *
 * ************************************************************************* */
void _frl_del_rl
#ifdef ANSI_C
(FRL_REG *rl)
#else  /* ANSI_C */
(rl)
FRL_REG *rl;
#endif /* ANSI_C */
{
  if (rl == NULL) return;
  
  _frl_del_rl(rl->next);

  free(rl);
}

/* ************************************************************************* *
 * Free file region list                                                     *
 * ************************************************************************* */
void _frl_del_list
#ifdef ANSI_C
(FRL_FILE *fr)
#else  /* ANSI_C */
(fr)
FRL_FILE *fr;
#endif /* ANSI_C */
{
  if (fr == NULL) return;
  
  _frl_del_list(fr->next);
  _frl_del_rl(fr->regs);

  free(fr);
  return;
}

/* ************************************************************************* *
 * Create a new region list element                                          *
 * ************************************************************************* */
static FRL_REG *frl_new_rl
#ifdef ANSI_C
(int ofs, int eop)
#else  /* ANSI_C */
(ofs, eop)
int ofs;
int eop;
#endif /* ANSI_C */
{
  FRL_REG *rl;

  if ((rl = (FRL_REG *)malloc(sizeof(FRL_REG))) == NULL)
  {
    perror("pfsd: frl_new_rl(): malloc()");
    return (NULL);
  }
    
  rl->ofs  = ofs;
  rl->eop  = eop;
  rl->next = NULL;
    
  return (rl);
}
  
/* ************************************************************************* *
 * Mark a region in a file as unoccupied                                     *
 * ************************************************************************* */
static int frl_unoccreg
#ifdef ANSI_C
(FRL_FILE *fr, int ofs, int eop)
#else  /* ANSI_C */
(fr, ofs, eop)
FRL_FILE *fr;
int ofs;
int eop;
#endif /* ANSI_C */
{
  FRL_REG *n;
  FRL_REG *c;
  FRL_REG *p;
  FRL_REG *pp;

  if ((ofs < 0) || (eop <= ofs) || (eop > fr->eof))
  {
    errno = EINVAL;
    return (-1);
  }

  if (fr->regs == NULL)
  {                             /* xxx*--n--(xxx|eof */
    if (eop == fr->eof)
    {
      /* xxx*--n--(|eof
       *    |eof        */
      fr->eof = ofs;
      return (0);
    }
    /* xxx*--n--(XXX|eof */
    if ((fr->regs = frl_new_rl(ofs,eop)) == NULL)
      return (-1);
    return (0);
  }
  
  for (c=fr->regs, p=NULL, pp=NULL; c != NULL; pp=p, p=c, c= c->next)
  {
    if (ofs < c->ofs)
    {
      /* This is the place to put the element */
      /* xxxxxx*--n--(xxx*--c--(XXX|eof */
      
      if (p == NULL)            /* First element in the list */
      {                         /* xxx*--n--(xxx*--c--(XXX|eof */
        if (eop == c->ofs)
        {                       /* xxx*--n--(*--c--(XXX|eof */
          c->ofs = ofs;
          return (0);
        }
        if (eop < c->ofs)
        {                       /* xxx*--n--(XXX*--c--(XXX|eof */
          if ((n = frl_new_rl(ofs,eop)) == NULL)
            return (-1);
          n->next = c;
          fr->regs = n;
          return (0);
        }
        /* xxx*--n-*-(-c--(XXX|eof */
        errno = EPFSLFRLOVL;     /* Overlapping is an error */
        return (-1);
      }
      
      /* There is a previous element */
      /* xxx*--p--(xxx*--n--(xxx*--c--(XXX|eof */
      if (p->eop == ofs)
      {                         /* xxx*--p--(*--n--(xxx*--c--(XXX|eof */
        if (eop == c->ofs)
        {                       /* xxx*--p--(*--n--(*--c--(xxx|eof */
          p->eop  = c->eop;
          p->next = c->next;
          free(c);
          return (0);
        }
        if (eop < c->ofs)
        {                       /* xxx*--p--(*--n--(XXX*--c--(XXX|eof */
          p->eop = eop;
          return (0);
        }
        /* xxx*--p--(*--n-*-(-c--(XXX|eof */
        errno = EPFSLFRLOVL;     /* Overlapping is an error */
        return (-1);
      }
      
      if (p->eop < ofs)
      {                         /* xxx*--p--(XXX*--n--(xxx*--c--(XXX|eof */
        if (eop == c->ofs)
        {                       /* xxx*--p--(XXX*--n--(*--c--(XXX|eof */
          c->ofs = ofs;
          return (0);
        }
        if (eop < c->ofs)
        {                       /* xxx*--p--(XXX*--n--(XXX*--c--(XXX|eof */
          if ((n = frl_new_rl(ofs,eop)) == NULL)
            return (-1);
          p->next = n;
          n->next = c;
          return(0);
        }
        /* xxx*--p--(XXX*--n-*-(-c--(XXX|eof */
        errno = EPFSLFRLOVL;     /* Overlapping is an error */
        return (-1);
      }
      /* xxx*--p-*-(-n--(xxx*--c--(XXX|eof */
      errno = EPFSLFRLOVL;     /* Overlapping is an error */
      return (-1);
    }
  }
  
  /* We are at the end of the list */
  /* xxxxxx*--p--(xxx*--n--(xxx|eof */
  if (p->eop == ofs)
  {                             /* xxx*--p--(*--n--(xxx|eof */
    if (eop == fr->eof)
    {                           /* xxx*--p--(*--n--(| eof
                                 *    | eof               */
      fr->eof = p->ofs;
      free(p);
      if (pp == NULL)
      {                         /* xxx*--p--(*--n--(|eof
                                 * xxx|eof               */
        fr->regs = NULL;
        return(0);
      }
      
      /* xxx*--pp--(XXX*--p--(*--n--(| eof
       *               | eof               */
      pp->next = NULL;
      return (0);
    }
    /* xxx*--p--(*--n--(XXX|eof */
    p->eop = eop;
    return (0);
  }
  if (p->eop < ofs)
  {                             /* xxx*--p--(XXX*--n--(xxx|eof */
    if (eop == fr->eof)
    {                           /* xxx*--p--(XXX*--n--(|eof
                                 *              |eof        */
      fr->eof = ofs;
      return (0);
    }
    /* xxx*--p--(XXX*--n--(XXX|eof */
    if ((n = frl_new_rl(ofs,eop)) == NULL)
      return (-1);
    p->next = n;
    return (0);
  }
  /* xxx*--p-*-(-n--(xxx|eof */
  errno = EPFSLFRLOVL;           /* Overlapping is an error */
  return (-1);
}


/* ************************************************************************* *
 * Mark a region in a file as unoccupied                                     *
 * ************************************************************************* */
int _frl_unocc_reg
#ifdef ANSI_C
(FRL_FILE *fr, char *ipaddr, char *filename, int ofs, int length)
#else  /* ANSI_C */
(fr, ipaddr, filename, ofs, length)
FRL_FILE *fr;
char *ipaddr;
char *filename;
int ofs;
int length;
#endif /* ANSI_C */
{
  FRL_FILE *f;

  for (f=fr; f != NULL; f=f->next)
  {
    if ((strcmp(f->ipaddr,ipaddr) == 0) &&
        (strcmp(f->filename,filename) == 0))
    {
      if (frl_unoccreg(f, ofs, ofs + length) < 0)
        return (-1);
      return (0);
    }
  }
  
  /* It is impossible to free a region in a file that doen't exist */
  errno = EPFSLFRLNOENT;
  return (-1);
}

/* ************************************************************************* *
 * Get a list of free regions in a file with a specified length              *
 * ************************************************************************* */
static FRL_REG *frl_get_reglst
#ifdef ANSI_C
(FRL_FILE *fr, int length)
#else  /* ANSI_C */
(fr, length)
FRL_FILE *fr;
int length;
#endif /* ANSI_C */
{
  FRL_REG *res    = NULL;
  FRL_REG *resend = NULL;
  FRL_REG *rescur;
  int len;
  
  for (len = length;
       (len > 0) && (fr->regs != NULL);
       len -= rescur->eop - rescur->ofs)
  {
    /* xxxx*--c--(XXX|eof */
           
    if ((fr->regs->eop - fr->regs->ofs) > len)
    {                           /* xxxx*----c---(XXX|eof
                                 *     nnnnn*-c-(XXX|eof
                                 *     *-rc-(            */
      if ((rescur = (FRL_REG *)malloc(sizeof(FRL_REG))) == NULL)
      {
        perror("pfsd: frl_get_reglst(): malloc()");
        _frl_del_rl(res);
        return (NULL);
      }
      rescur->ofs   = fr->regs->ofs;
      fr->regs->ofs = rescur->eop = rescur->ofs + len;
    }
    else 
    {                           /* xxxx*---c---(XXX|eof
                                 *     NNNNNNNNnnnn
                                 *     *--rc---(        */
      rescur   = fr->regs;
      fr->regs = fr->regs->next;       /* Remove form list */
    }

    /* Append at end of result list */
    if (res == NULL)
      resend = res = rescur;
    else
      resend = resend->next = rescur;
    resend->next = NULL;
  }

  if (len > 0)
  {
    /* There are no more free regions
     * Append at end of file */
    /* xxxxxx|eof
     * xxxxxxNNNNN|eof */
    if ((rescur = (FRL_REG *)malloc(sizeof(FRL_REG))) == NULL)
    {
      perror("pfsd: frl_get_reglst(): malloc()");
      _frl_del_rl(res);
      return (NULL);
    }
    
    rescur->ofs = fr->eof;
    rescur->eop = fr->eof += len;

    /* Append at end of result list */
    if (res == NULL)
      resend = res = rescur;
    else
      resend = resend->next = rescur;
    resend->next = NULL;
  }
  
  return (res);
}


/* ************************************************************************* *
 * Get a list of free regions in a file with a specified length              *
 * ************************************************************************* */
FRL_REG *_frl_get_rl
#ifdef ANSI_C
(FRL_FILE **fr, char *ipaddr, char *filename, int length)
#else  /* ANSI_C */
(fr, ipaddr, filename, length)
FRL_FILE **fr;
char *ipaddr;
char *filename;
int length;
#endif /* ANSI_C */
{
  FRL_FILE *f;

  for (f=*fr; f != NULL; f=f->next)
  {
    if ((strcmp(f->ipaddr,ipaddr) == 0) &&
        (strcmp(f->filename,filename) == 0))
      return (frl_get_reglst(f,length));
  }
  
  /* This is a new file */
  if ((f = (FRL_FILE *)malloc(sizeof(FRL_FILE))) == NULL)
  {
    perror("pfsd: _frl_get_rl(): malloc()");
    return (NULL);
  }
  
  strcpy(f->ipaddr,ipaddr);
  strcpy(f->filename,filename);
  f->eof  = 0;
  f->regs = NULL;

  /* Prepend at file list */
  f->next = *fr;
  *fr     = f;

  return(frl_get_reglst(f,length));
}


/* ************************************************************************* *
 * Mark a region in a file as occupied                                       *
 * ************************************************************************* */
static int frl_occreg
#ifdef ANSI_C
(FRL_FILE *fr, int ofs, int eop)
#else  /* ANSI_C */
(fr, ofs, eop)
FRL_FILE *fr;
int ofs;
int eop;
#endif /* ANSI_C */
{
  FRL_REG *c;
  FRL_REG *n;
  FRL_REG *p;

  if (ofs > eop)
  {
    errno = EINVAL;
    return (-1);
  }

  for (c=fr->regs, p=NULL; c != NULL;)
  {
    if (ofs < c->eop)
    {
      /* This free region has to be altered */
      /* xxx*--c--(XXxxXX|eof */
      /*  nnnnnnnnnnnnnnnnnn  */
      if (ofs <= c->ofs)
      {
        /* xxx*--c--(XXxxXX|eof */
        /*  nnnnnnnnnnnnnnnnnn  */
        if (eop < c->eop)
        {
          /* xxx*---c---(XXxxXX|eof */
          /*  nnnnnnn                */
          if (eop <= c->ofs)
          {
            /* xxx*---c---(XXxxXX|eof */
            /*  nn                    */
            /* xxx*---c---(XXxxXX|eof */
            return (0);
          }
          /* xxx*---c---(XXxxXX|eof */
          /*  nnNNN                 */
          /* xxxXXX*-c--(XXxxXX|eof */
          c->ofs = eop;
          return (0);
        }
        /* xxx*--c--(XXxxXX|eof */
        /*   nNNNNNNNnnnnnnnnn  */
        /* xxxXXXXXXXXXxxXX|eof */
        if (p == NULL)
          n = fr->regs = c->next;
        else
          n = p->next = c->next;
        free(c);
        c = n; 
      }
      else
      {
        /* xxx*--c--(XXxxXX|eof */
        /*      NNnnnnnnnnnnnn  */
        if (eop < c->eop)
        {
          /* xxx*-----c-----(XXxxXX|eof */
          /*         NNN                */
          /* xxx*-c-(XXX*-n-(XXxxXX|eof */
          if ((n = frl_new_rl(eop,c->eop)) == NULL)
            return (-1);
          c->eop = ofs;
          n->next = c->next;
          c->next = n;
          return (0);
        }
        /* xxx*---c---(XXxxXX|eof */
        /*         NNNNnnnnnnnnn  */
        /* xxx*-c-(XXXXXXxxXX|eof */
        c->eop = ofs;
        p = c;
        c = c->next;
      }
    }
    else
    {
      p = c;
      c = c->next;
    }
  }

  if (ofs <= fr->eof)
  {
    /* xxxxXXXX|eof */
    /* nnnnnNNnnnn  */
    if (eop <= fr->eof)
    {
      /* xxxxXXXX|eop */
      /* nnnnnNNn     */
      /* xxxxXXXX|eop */
      return (0);
    }
    /* xxxxXXXX|eof
     * nnnnnNNNNNN
     * xxxxXXXXXXX|eof */
    fr->eof = eop;
    return (0);
  }
  /* xxxxXXXX|eof
   *             NNN
   * xxxxXXXX*--(XXX|eof*/
  if ((n = frl_new_rl(fr->eof,ofs)) == NULL)
      return (-1);
  
  if (p == NULL)
    fr->regs = n;
  else
    p->next = n;
  fr->eof = eop;
  return (0);
}


/* ************************************************************************* *
 * Mark a region in a file as occupied                                       *
 * ************************************************************************* */
FRL_FILE * _frl_occ_reg
#ifdef ANSI_C
(FRL_FILE *fr, char *ipaddr, char *filename, int offset, int length)
#else  /* ANSI_C */
(fr, ipaddr, filename, offset, length)
FRL_FILE *fr;
char *ipaddr;
char *filename;
int offset;
int length;
#endif /* ANSI_C */
{

  FRL_FILE *f;

  for (f=fr; f != NULL; f=f->next)
  {
    if ((strcmp(f->ipaddr,ipaddr) == 0) &&
        (strcmp(f->filename,filename) == 0))
    {
      if (frl_occreg(f,offset,offset+length) < 0)
        return (NULL);
      return (fr);
    }
  }
  
  /* This is a new file */
  if ((f = (FRL_FILE *)malloc(sizeof(FRL_FILE))) == NULL)
  {
    perror("pfsd: _frl_occ_reg(): malloc()");
    return (NULL);
  }
  
  strcpy(f->ipaddr,ipaddr);
  strcpy(f->filename,filename);
  f->eof  = 0;
  f->regs = NULL;

  /* Prepend at file list */
  f->next = fr;
  fr     = f;

  if (frl_occreg(f,offset,offset+length) < 0)
  {
    _frl_del_list(fr);
    return(NULL);
  }

  return (fr);
}


/* ************************************************************************* *
 * Truncate distributed files to their actual length                         *
 * ************************************************************************* */
int _frl_trunc 
#ifdef ANSI_C
(FRL_FILE *fr)
#else  /* ANSI_C */
(fr)
FRL_FILE *fr;
#endif /* ANSI_C */
{
  FRL_FILE *f;
  
  for (f=fr; f != NULL; f=f->next)
  {
    if (_iod_trunc(f->ipaddr,f->filename,f->eof) < 0)
      return (-1);
  }

  return (0);
}

/* ************************************************************************* *
 * Print free regions.                                                       *
 * ************************************************************************* */
void _frl_prt
#ifdef ANSI_C
(FRL_FILE *fr)
#else  /* ANSI_C */
(fr)
FRL_FILE *fr;
#endif /* ANSI_C */
{
  FRL_FILE *f;
  FRL_REG  *r;
  
  for (f=fr; f != NULL; f=f->next)
  {
    printf("\tFREE REGIONS IN FILE\n");
    printf("\tipaddr\t\t: %s\n",f->ipaddr);
    printf("\tfilename\t: %s\n",f->filename);
    printf("\teof\t\t: %d\n",f->eof);
    if (f->regs == NULL)
      printf("\tregs\t\t: NULL\n");
    else
    {
      printf("\t  Offset   End Offset\n");
      for (r = f->regs; r != NULL; r = r->next)
        printf("\t%10d %10d\n",r->ofs,r->eop);
    }
  }
  
  return;
}
        
#endif /* TIGHT_FILES */
