/* ************************************************************************* *
 *                                                                           *
 *    pfsd_dist.c,v
 *    pfsd procedures to obtain and store distribution information
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_dist.c,v
 *  RCS Date     : 1996/12/27 15:47:06
 *  RCS Revision : 1.9
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_dist.c,v 1.9 1996/12/27 15:47:06 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <rpc/rpc.h>
#include <fcntl.h>
#include <unistd.h>

#include "pfsd.h"
#include "pfsd_macros.h"
#include "pfsd_defines.h"
#include "pfsd_avl.h"
#include "pfslib_errno.h"
#ifdef TIGHT_FILES
#include "pfsd_frl.h"
#endif

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern filetabel *_cftbl;

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern AVLBAUM  *_avl_get_leaf      _PH((AVLBAUM *r, off_t g_ofs));
extern AVLBAUM  *_avl_insert_region _PH((AVLBAUM *r, off_t g_ofs, off_t g_len, 
                                         char *filename, char *ipaddr, 
                                         off_t p_ofs));
extern AVLBAUM  *_avl_del_region    _PH((AVLBAUM *r, off_t g_ofs,
                                         off_t g_eop));
extern void      _avl_del_tree      _PH((AVLBAUM *r));
extern int       _iod_start         _PH((char *ipaddr));
extern int       _iod_unlink        _PH((char *ipaddr, char *filename));
#ifdef TIGHT_FILES
extern int       _frl_unocc_reg     _PH((FRL_FILE *fr, char *ipaddr,
                                        char *filename, int ofs, int length));
extern FRL_REG  *_frl_get_rl        _PH((FRL_FILE **fr, char *ipaddr, 
                                        char *filename, int length));
extern void      _frl_del_rl        _PH((FRL_REG *rl));
extern void      _frl_del_list      _PH((FRL_FILE *fr));
extern int       _frl_trunc         _PH((FRL_FILE *fr));
extern FRL_FILE *_frl_occ_reg       _PH((FRL_FILE *fr, char *ipaddr, 
                                         char *filename, int offset,
                                         int length));
extern void      _frl_prt           _PH((FRL_FILE *fr));
#else  /* TIGHT_FILES */
extern int       _iod_trunc         _PH((char *ipaddr, char *filename,
                                         long offset));
#endif /* TIGHT_FILES */

#undef _PH

/* ************************************************************************* *
 * Definitions                                                               *
 * ************************************************************************* */

struct DIST_DESCR
{
  u_int        opencnt;
  bool_t       unlink;
  AVLBAUM      *avl;
#ifdef TIGHT_FILES
  FRL_FILE     *frl;
#endif /* TIGHT_FILES */
};
typedef struct DIST_DESCR DIST_DESCR;

#define _CFTE_AVL ((DIST_DESCR *)_CFTE.distd)->avl
#ifdef TIGHT_FILES
#define _CFTE_FRL ((DIST_DESCR *)_CFTE.distd)->frl
#endif /* TIGHT_FILES */
#define _CFTE_OC  ((DIST_DESCR *)_CFTE.distd)->opencnt
#define _CFTE_UL  ((DIST_DESCR *)_CFTE.distd)->unlink


/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Free disribution information                                              *
 * ************************************************************************* */
void _dist_free
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  _avl_del_tree(_CFTE_AVL);
#ifdef TIGHT_FILES
  _frl_del_list(_CFTE_FRL);
#endif /* TIGHT_FILES */
  return;
}


/* ************************************************************************* *
 * Get a distribution list from an AVL tree for an offset with a length      *
 * ************************************************************************* */
static DIST_LIST dist_getlst
#ifdef ANSI_C
(AVLBAUM *r, off_t g_ofs, off_t g_len)
#else  /* ANSI_C */
(r, g_ofs, g_len)
AVLBAUM *r;
off_t g_ofs;
off_t g_len;
#endif /* ANSI_C */
{
  AVLBAUM   *tl;
  DIST_LIST list;
  off_t      g_eop = g_ofs + g_len;

  tl = _avl_get_leaf(r,g_ofs);

  list.DIST_LIST_val = NULL;
  
  for (list.DIST_LIST_len = 0;
       (tl != NULL) && (tl->g_ofs < g_eop);
       tl = tl->nl,list.DIST_LIST_len++)
  {
    if (list.DIST_LIST_len == 0)
    {
      if ((list.DIST_LIST_val = (DIST_LIST_EL *)malloc(sizeof(DIST_LIST_EL)))
          == NULL)
      {
        perror("pfsd: dist_getlst(): malloc()");
        return (list);
      }
    }
    else
    {
      if ((list.DIST_LIST_val =
           (DIST_LIST_EL *)realloc(list.DIST_LIST_val,
                                   sizeof(DIST_LIST_EL)*
                                   (list.DIST_LIST_len+1))) == NULL)
      {
        perror("pfsd: dist_getlst(): realloc()");
        list.DIST_LIST_len = 0;
        return (list);
      }
    }
    
    strcpy(list.DIST_LIST_val[list.DIST_LIST_len].ipaddr,tl->leaf.ipaddr);
    strcpy(list.DIST_LIST_val[list.DIST_LIST_len].filename,tl->leaf.filename);
    list.DIST_LIST_val[list.DIST_LIST_len].done = FALSE;
    if (tl->g_ofs <= g_ofs)
    {
      /* r:    *------(
       * tl:  **-----(((
       */
      list.DIST_LIST_val[list.DIST_LIST_len].offset = g_ofs - tl->leaf.p_nofs;
      if (tl->g_eop >= g_eop)
        /* r:   *-----(
         * tl: **-----((
         */
        list.DIST_LIST_val[list.DIST_LIST_len].length = g_len;
      else
        /* r:   *-----(
         * tl: **----(
         */
        list.DIST_LIST_val[list.DIST_LIST_len].length = tl->g_eop - g_ofs;
    }
    else
    {
      /* r:  *-----(
       * tl:  *---(((
       */
      list.DIST_LIST_val[list.DIST_LIST_len].offset = tl->g_ofs - tl->leaf.p_nofs;
      if (tl->g_eop >= g_eop)
        /* r:  *-----(
         * tl:  *----((
         */
        list.DIST_LIST_val[list.DIST_LIST_len].length = g_eop - tl->g_ofs;
      else
        /* r:  *-----(
         * tl:  *---(
         */
        list.DIST_LIST_val[list.DIST_LIST_len].length = tl->g_eop - tl->g_ofs;
    }
  }

  return (list);
}

/* ************************************************************************* *
 * Free a distibution list.                                                  *
 * ************************************************************************* */
void _dist_del_list
#ifdef ANSI_C
(DIST_LIST list)
#else  /* ANSI_C */
(list)
DIST_LIST list;
#endif /* ANSI_C */
{
  if (list.DIST_LIST_len == 0) return;
  
  free(list.DIST_LIST_val);
  return;
}

/* ************************************************************************* *
 * Get a distribution list for an IO identifier                              *
 * ************************************************************************* */
DIST_LIST _dist_get_list
#ifdef ANSI_C
(int _ftix, int _ctix, int _ioID)
#else  /* ANSI_C */
(_ftix, _ctix, _ioID)
int _ftix;
int _ctix;
int _ioID;
#endif /* ANSI_C */
{
  DIST_LIST list;
#ifdef TIGHT_FILES
  int i;
  FRL_REG *rl;
  FRL_REG *rlc;
  off_t offset;
#endif /* TIGHT_FILES */

  S_TIME(_dist_get_list);

  switch(_IOID.op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    /* der Dateiname sollte noch Maschinenspezifisch werden */
#ifdef TIGHT_FILES
    /* Get the list of current distribution of this part of the file */
    list = dist_getlst(_CFTE_AVL,_IOID.offset,(off_t)_IOID.iolen);

    /* Free these regions in the appropriate files */
    for (i=0;i<list.DIST_LIST_len;i++)
    {
      if (strcmp(list.DIST_LIST_val[i].filename,DEV_ZERO) != 0)
      {
        if (_frl_unocc_reg(_CFTE_FRL,
                           list.DIST_LIST_val[i].ipaddr,
                           list.DIST_LIST_val[i].filename,
                           list.DIST_LIST_val[i].offset,
                           list.DIST_LIST_val[i].length) < 0)
        {
          _dist_del_list(list);
          list.DIST_LIST_len = 0;
          list.DIST_LIST_val = NULL;
          return (list);
        }
      }
    }
    _dist_del_list(list);
    
    /* Get the regions in the local file */
    if ((rl = _frl_get_rl(&(_CFTE_FRL),
                          _CLNT.ident.ipaddr,
                          _CFTE.filename,_IOID.iolen)) == NULL)
    {
      list.DIST_LIST_len = 0;
      list.DIST_LIST_val = NULL;
      return (list);
    }      

    /* Insert these regions in the AVL tree */
    for (rlc=rl, offset=_IOID.offset;
         rlc != NULL;
         offset += rlc->eop - rlc->ofs, rlc=rlc->next)
    {
      if ((_CFTE_AVL = _avl_insert_region(_CFTE_AVL,
                                          offset, rlc->eop - rlc->ofs,
                                          _CFTE.filename, _CLNT.ident.ipaddr,
                                          rlc->ofs))
          == NULL)
      {
        /* This is unrecoverable */
        _frl_del_rl(rl);
        list.DIST_LIST_len = 0;
        list.DIST_LIST_val = NULL;
        return (list);
      }
    }
    _frl_del_rl(rl);
#else  /* TIGHT_FILES */
    if ((_CFTE_AVL = _avl_insert_region(_CFTE_AVL,
                                        _IOID.offset, (off_t)_IOID.iolen,
                                        _CFTE.filename, _CLNT.ident.ipaddr,
                                        _IOID.offset)) == NULL)
    {
      /* This is unrecoverable */
      list.DIST_LIST_len = 0;
      list.DIST_LIST_val = NULL;
      return (list);
    }
#endif /* TIGHT_FILES */
  }

  /* Get the distribution list from the AVL tree */
  list = dist_getlst(_CFTE_AVL,_IOID.offset,(off_t)_IOID.iolen);

  E_TIME(_dist_get_list);

  return (list);
}

/* ************************************************************************* *
 * Set the unlink flag.                                                      *
 * ************************************************************************* */
int _dist_setunlink
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  _CFTE_UL = TRUE;
  
  return (0);
}


/* ************************************************************************* *
 * Unlink distributed files and the distribution file.                       *
 * ************************************************************************* */
static int dist_unlink
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  int res;
  DIST_LIST list;
  int i,j;
  char  distfname[MAXNLEN];

  if (_CFTE_AVL == NULL)
    return (0);
  
  list = dist_getlst(_CFTE_AVL,0,(_CFTE_AVL)->g_eop);

  if (list.DIST_LIST_len == 0)
    return (-1);
  
  for (i=0;i<list.DIST_LIST_len;i++)
  {
    if (list.DIST_LIST_val[i].done) continue;
    if (_iod_unlink(list.DIST_LIST_val[i].ipaddr,
                    list.DIST_LIST_val[i].filename) < 0)
      res = -1;
    for (j=i; j < list.DIST_LIST_len; j++)
    {
      if ((list.DIST_LIST_val[j].done) ||
          (strcmp(list.DIST_LIST_val[i].ipaddr,
                  list.DIST_LIST_val[j].ipaddr) != 0) ||
          (strcmp(list.DIST_LIST_val[i].filename,
                  list.DIST_LIST_val[j].filename) != 0))
        continue;
      list.DIST_LIST_val[j].done = TRUE;
    }
  }
  _dist_del_list(list);
    
  strcpy(distfname,_CFTE.filename);
  strcat(distfname,DISTEXT);
  
  if (unlink(distfname) < 0)
  {
    perror("dist_unlink: unlink()");
    return (-1);
  }
  
  _dist_free(_ftix);

  return (res);
}

/* ************************************************************************* *
 * Truncate a distributed file.                                              *
 * ************************************************************************* */
int _dist_trunc
#ifdef ANSI_C
(int _ftix, off_t maxlen)
#else  /* ANSI_C */
(_ftix, maxlen)
int _ftix;
off_t maxlen;
#endif /* ANSI_C */
{
  DIST_LIST list;
  int i;
#ifndef TIGHT_FILES
  int j;
#endif /* TIGHT_FILES */

  if (maxlen < 0)
  {
    errno = EINVAL;
    return (-1);
  }
  
  if (_CFTE_AVL == NULL)
  {
    if (maxlen == 0)
      return (0);
    if ((_CFTE_AVL = _avl_insert_region(_CFTE_AVL, 0,maxlen,
                                        DEV_ZERO,LOCALHOST,0))
        == NULL)
      return (-1);
    return (0);
  }
  
  if ((_CFTE_AVL)->g_eop == maxlen)
    /* Size is ok */
    return (0);

  if ((_CFTE_AVL)->g_eop > maxlen)
  {
    /* Delete part of the file(s) */
    list = dist_getlst(_CFTE_AVL, maxlen,(_CFTE_AVL)->g_eop);
    if (list.DIST_LIST_len == 0)
      return (-1);
    
#ifdef TIGHT_FILES
    for (i=0;i < list.DIST_LIST_len;i++)
    {
      if (strcmp(list.DIST_LIST_val[i].filename,DEV_ZERO) != 0)
      {
        if (_frl_unocc_reg(_CFTE_FRL,
                           list.DIST_LIST_val[i].ipaddr,
                           list.DIST_LIST_val[i].filename,
                           list.DIST_LIST_val[i].offset,
                           list.DIST_LIST_val[i].length) < 0)
        {
          _dist_del_list(list);
          return (-1);
        }
      }
    }
    
    if (_frl_trunc(_CFTE_FRL) < 0)
    {
      _dist_del_list(list);
      return (-1);
    }

#else  /* TIGHT_FILES */

    for (i=0;i < list.DIST_LIST_len;i++)
    {
      if (list.DIST_LIST_val[i].done) continue;
      if (strcmp(list.DIST_LIST_val[i].filename,DEV_ZERO) != 0)
      {
        if (_iod_trunc(list.DIST_LIST_val[i].ipaddr,
                       list.DIST_LIST_val[i].filename,
                       list.DIST_LIST_val[i].offset) < 0)
        {
          _dist_del_list(list);
          return (-1);
        }
      }
      for (j=i; j < list.DIST_LIST_len; j++)
      {
        if ((list.DIST_LIST_val[j].done) ||
            (strcmp(list.DIST_LIST_val[i].ipaddr,
                    list.DIST_LIST_val[j].ipaddr) != 0) ||
            (strcmp(list.DIST_LIST_val[i].filename,
                    list.DIST_LIST_val[j].filename) != 0))
          continue;
        list.DIST_LIST_val[j].done = TRUE;
      }
    }

#endif /* TIGHT_FILES */

    _dist_del_list(list);
    
    errno = 0;
    if (((_CFTE_AVL = _avl_del_region(_CFTE_AVL,
                                      maxlen,
                                      (_CFTE_AVL)->g_eop)) == NULL) && 
        (errno != 0))
      return (-1);
    return (0);
  }
  
  /* Enlarge file */
  if ((_CFTE_AVL = _avl_insert_region(_CFTE_AVL, (_CFTE_AVL)->g_eop,
                                  maxlen,DEV_ZERO,LOCALHOST,0))
      == NULL)
    return (-1);
  return (0);
}

/* ************************************************************************* *
 * Read the distribution information of a file and build an AVL tree.        *
 * ************************************************************************* */
static int dist_read
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  FILE    *file;
  int      res;
  off_t    g_ofs;
  off_t    g_len;
  char     ipaddr[IPADDRLEN];
  off_t    p_ofs;

  char  distfname[MAXNLEN];
  char  filename[MAXNLEN];
  
  strcpy(distfname,_CFTE.filename);
  strcat(distfname,DISTEXT);
  
  _dist_free(_ftix);

#ifdef TIGHT_FILES
  _CFTE_FRL = NULL;
#endif /* TIGHT_FILES */
  _CFTE_AVL = NULL;
  _CFTE.endoffs = 0;

  if ((file = fopen(distfname,"r")) == NULL)
  {
    if ((errno == ENOENT) && (_CFTE.oflags & O_CREAT))
      return (0);
    else
    {
      perror("pfsd: dist_read(): fopen()");
      return (-1);
    }
  }
  
  while ((res=fscanf(file,"%ld %ld %ld %s %s\n",
                     &g_ofs,&g_len,&p_ofs,
                     ipaddr,filename)) == 5)
  {
    /* Check whether ther is a IOD running on the clients machine 
     * and start one if not */
    if (_iod_start(ipaddr)< 0)
    {
      perror("pfsd: dist_read(): _iod_start()");
      _dist_free(_ftix);
      fclose(file);
      return (-1);
    }
    if ((_CFTE_AVL = _avl_insert_region(_CFTE_AVL, g_ofs, g_len, filename,
                                        ipaddr, p_ofs)) == NULL)
    {
      perror("pfsd: dist_read(): _avl_insert_region()");
      _dist_free(_ftix);
      fclose(file);
      return (-1);
    }
#ifdef TIGHT_FILES
    if (strcmp(filename,DEV_ZERO) != 0)
    {
      if ((_CFTE_FRL = _frl_occ_reg(_CFTE_FRL,ipaddr,filename, p_ofs, g_len))
          == NULL)
      {
        perror("pfsd: dist_read(): _fr_occ_reg()");
        _dist_free(_ftix);
        fclose(file);
        return(-1);
      }
    }
#endif /* TIGHT_FILES */
  }

  fclose(file);

  if (res != EOF)
  {
    fprintf(stderr,"pfsd: dist_read(): Error in Distfile %s\n",distfname);
    _dist_free(_ftix);
    errno = EPFSLDISTF;
    return(-1);
  }
  
  if (_CFTE.oflags & O_TRUNC)
  {
    if (_dist_trunc(_ftix,0) < 0)
      return (-1);
  }

  if (_CFTE_AVL != NULL)
    _CFTE.endoffs = (_CFTE_AVL)->g_eop;
  return (0);
}

/* ************************************************************************* *
 * Initialize disribution information                                        *
 * ************************************************************************* */
int _dist_init
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  int _cftix;

  /* There should be additional checks for access permissions in this
     procedure ! */

  for (_cftix=0; _cftix < MAXNUMFILES; _cftix++)
  {
    /* Look for a matching file in the filetable */
    if ((_ftix != _cftix) &&       /* Other file table entry */
        (_CCFTE.actual > 0) &&    /* element in use */
        (strcmp(_CFTE.filename,_CCFTE.filename) == 0)) /* correct filename */
    {
      _CFTE.distd = _CCFTE.distd; /* Use this ditribution information */
      _CFTE_OC++;               /* Increase open count */
      return (0);
    }
  }
    
  if ((_CFTE.distd = malloc(sizeof(DIST_DESCR))) == NULL)
  {
    perror("pfsd: _dist_init(): malloc()");
    return (-1);
  }
  
#ifdef TIGHT_FILES
  _CFTE_FRL = NULL;
#endif /*  TIGHT_FILES*/
 
  _CFTE_AVL = NULL;

  _CFTE_OC = 1;
  _CFTE_UL = FALSE;

  return (dist_read(_ftix));
}

/* ************************************************************************* *
 * Write the distribution information to a file                              *
 * ************************************************************************* */
static int dist_write
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  FILE *file;
  AVLBAUM *t;

  char distfname[MAXNLEN];
  
  strcpy(distfname,_CFTE.filename);
  strcat(distfname,DISTEXT);
  
  if ((file = fopen(distfname,"w")) == NULL)
  {
    perror("pfsd: dist_write(): fopen()");
    return(-1);
  }
  
  if (_CFTE_AVL != NULL)
  {
    for (t=(_CFTE_AVL)->lc; t!=NULL; t=t->nl)
    {
      fprintf(file,"%10ld %10ld %10ld %s %s\n",
              t->g_ofs,t->g_eop - t->g_ofs,t->g_ofs - t->leaf.p_nofs,
              t->leaf.ipaddr, t->leaf.filename);
    }
  }
  
  fclose(file);

#ifdef TIGHT_FILES
  _frl_trunc(_CFTE_FRL);
#endif /* TIGHT_FILES */
  return(0);
}

/* ************************************************************************* *
 * Delete disribution information                                            *
 * ************************************************************************* */
int _dist_del
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  int res = 0;
  int res1 = 0;
  
  _CFTE_OC--;
  if (_CFTE_OC > 0)             /* The file is still open by other 
                                   applications */
    return (0);

  res = dist_write(_ftix);

  if (_CFTE_UL)
  {
    res1 = dist_unlink(_ftix);
    if (res >= 0) res = res1;
  }

  _dist_free(_ftix);
  free(_CFTE.distd);
  _CFTE.distd = NULL;

  return (res);
}


/* ************************************************************************* *
 * Print distribution information of a file.                                *
 * ************************************************************************* */
void _dist_prt
#ifdef ANSI_C
(int _ftix)
#else  /* ANSI_C */
(_ftix)
int _ftix;
#endif /* ANSI_C */
{
  AVLBAUM *t;
  
  printf("\tFILE DISTRIBUTION\n");

  if (_CFTE_AVL != NULL)
  {
    printf("\t  Global                Offset\n");
    printf("\t  offset      Length    in file     IP address   Filename\n");
    for (t=(_CFTE_AVL)->lc; t!=NULL; t=t->nl)
    {
      printf("\t%10ld %10ld %10ld %s %s\n",
             t->g_ofs,t->g_eop - t->g_ofs, t->g_ofs - t->leaf.p_nofs,
             t->leaf.ipaddr, t->leaf.filename);

    }
  }
  
#ifdef TIGHT_FILES
  _frl_prt(_CFTE_FRL);
#endif /* TIGHT_FILES */

  return;
}
