/* ************************************************************************* *
 *                                                                           *
 *    pfsd_avl.c,v
 *    pfsd procedures for AVL trees holding distribution information
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : pfsd_avl.c,v
 *  RCS Date     : 1996/05/02 15:25:52
 *  RCS Revision : 1.5
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "pfsd_avl.c,v 1.5 1996/05/02 15:25:52 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "global.h"
#include "pfsd_avl.h"
#include "pfslib_errno.h"


/* ************************************************************************* *
 * Definitions                                                               *
 * ************************************************************************* */

/* H"ohe H(a) eines Baumes ist max(H(a->lt->h),H(a->ut->h)) + 1 */
#define height(a) (a->lt->h > a->ut->h) ? (a->lt->h+1) : (a->ut->h + 1)

/* Liefert true falls |H(a)-H(b)| <= 1 */
#define avl(b,c) ((b->h > c->h) && ((b->h - c->h) < 2)) ||\
                 ((b->h <= c->h) && ((c->h - b->h) < 2))

/* ************************************************************************* *
 * Prototype declarations                                                    *
 * ************************************************************************* */
#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

static AVLBAUM *avl_ins_region  _PH((AVLBAUM *r, off_t g_ofs, off_t g_eop, 
                                     char *filename, char *ipaddr,
                                     off_t p_nofs));

AVLBAUM        *_avl_del_region _PH((AVLBAUM *r, off_t g_ofs, off_t g_eop));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Rotate AVL tree to the left                                               *
 * ************************************************************************* */
static AVLBAUM *avl_rot_left
#ifdef ANSI_C
(AVLBAUM *r)
#else  /* ANSI_C */
(r)
AVLBAUM *r;
#endif /* ANSI_C */
{
  /*        r  <-+                ut
   *      /  \   |               /  \
   *     lt  ut -+    -->       r   utut
   *        /  \               / \
   *     utlt  utut          lt  utlt
   */

  AVLBAUM *ut;
  
  ut        = r->ut;
  
  r->ut     = ut->lt;
  r->g_eop  = r->ut->g_eop;
  r->uc     = r->ut->uc;
  r->h      = height(r);
  
  ut->lt    = r;
  ut->g_ofs = r->g_ofs;
  ut->lc    = r->lc;
  ut->h     = height(ut);
  
  return (ut);
}

/* ************************************************************************* *
 * Rotate AVL tree to the rigth.                                             *
 * ************************************************************************* */
static AVLBAUM *avl_rot_right
#ifdef ANSI_C
(AVLBAUM *r)
#else  /* ANSI_C */
(r)
AVLBAUM *r;
#endif /* ANSI_C */
{
  /*    +->   r                 lt
   *    |   /  \               /  \
   *    +- lt  ut  -->      ltlt   r
   *      /  \                    / \
   *   ltlt  ltut              ltut  ut
   */

  AVLBAUM *lt;
  
  lt = r->lt;
  
  r->lt = lt->ut;
  r->g_ofs = r->lt->g_ofs;
  r->lc = r->lt->lc;
  r->h = height(r);
  
  lt->ut = r;
  lt->g_eop = r->g_eop;
  lt->uc = r->uc;
  lt->h = height(lt);
  
  return (lt);
}

/* ************************************************************************* *
 * Rotate AVL tree to the right and then to the left.                        *
 * ************************************************************************* */
static AVLBAUM *avl_rot_rightleft
#ifdef ANSI_C
(AVLBAUM *r)
#else  /* ANSI_C */
(r)
AVLBAUM *r;
#endif /* ANSI_C */
{
  /*                r                  r                      utlt
   *               / \                / \                  /       \
   *             lt   ut             lt  utlt             r         ut
   *                 /  \     -->       /    \    -->    / \       /   \
   *             utlt    utut      utltlt    ut        lt utltlt utltut utut
   *            /    \                      /  \        
   *       utltlt    utltut            utltut  utut    
   */

  AVLBAUM *ut;
  AVLBAUM *utlt;
  
  ut   = r->ut;
  utlt = ut->lt;
  
  r->ut = utlt->lt;
  r->g_eop = r->ut->g_eop;
  r->uc = r->ut->uc;
  r->h  = height(r);
  
  ut->lt = utlt->ut;
  ut->g_ofs = ut->lt->g_ofs;
  ut->lc = ut->lt->lc;
  ut->h  = height(ut);

  utlt->lt = r;
  utlt->g_ofs = utlt->lt->g_ofs;
  utlt->lc = utlt->lt->lc;
  utlt->ut = ut;
  utlt->g_eop = utlt->ut->g_eop;
  utlt->uc = utlt->ut->uc;
  utlt->h  = height(utlt);
  
  return (utlt);
}

/* ************************************************************************* *
 * Rotate AVL tree to the left and then to the right.                        *
 * ************************************************************************* */
static AVLBAUM *avl_rot_leftright
#ifdef ANSI_C
(AVLBAUM *r)
#else  /* ANSI_C */
(r)
AVLBAUM *r;
#endif /* ANSI_C */
{
  /*                r                    r                        ltut
   *               / \                  / \                  /          \
   *             lt   ut             ltut  ut               lt           r
   *            /  \         -->     /    \    -->         / \          / \
   *        ltlt    ltut            lt    ltutut        ltlt ltutlt ltutut ut
   *               /    \          /  \
   *          ltutlt    ltutut   ltlt  ltutlt    
   */

  AVLBAUM *lt;
  AVLBAUM *ltut;
  
  lt   = r->lt;
  ltut = lt->ut;
  
  r->lt = ltut->ut;
  r->g_ofs = r->lt->g_ofs;
  r->lc = r->lt->lc;
  r->h  = height(r);
  
  lt->ut = ltut->lt;
  lt->g_eop = lt->ut->g_eop;
  lt->uc = lt->ut->uc;
  lt->h  = height(lt);

  ltut->ut = r;
  ltut->g_eop = ltut->ut->g_eop;
  ltut->uc = ltut->ut->uc;
  ltut->lt = lt;
  ltut->g_ofs = ltut->lt->g_ofs;
  ltut->lc = ltut->lt->lc;
  ltut->h  = height(ltut);
  
  return (ltut);
}

/* ************************************************************************* *
 * Create and initialize a new leaf for an AVL tree.                         *
 * ************************************************************************* */
static AVLBAUM *avl_new_leaf
#ifdef ANSI_C
(off_t g_ofs, off_t g_eop, char *ipaddr, char *filename, off_t p_nofs)
#else  /* ANSI_C */
(g_ofs, g_eop, ipaddr, filename, p_nofs)
off_t g_ofs;
off_t g_eop;
char *ipaddr;
char *filename;
off_t p_nofs;
#endif /* ANSI_C */
{
  AVLBAUM *l;
  
  if ((l = (AVLBAUM *)malloc(sizeof(AVLBAUM))) == NULL)
  {
    perror("pfsd: avl_new_leaf() malloc()");
    return (NULL);
  }
    
  l->h = 0;
  l->g_ofs = g_ofs;
  l->cut = -1;
  l->g_eop = g_eop;
  l->lc = l;
  l->uc = l;
  l->lt = NULL;
  l->ut = NULL;
  l->nl = NULL;
  strcpy(l->leaf.ipaddr,ipaddr);
  strcpy(l->leaf.filename,filename);
  l->leaf.p_nofs = p_nofs;
  
  return(l);
}

/* ************************************************************************* *
 * Delete a AVL tree recursively.                                            *
 * ************************************************************************* */
void _avl_del_tree
#ifdef ANSI_C
(AVLBAUM *r)
#else  /* ANSI_C */
(r)
AVLBAUM *r;
#endif /* ANSI_C */
{
  if (r == NULL) return;

  if (r->h > 0)
  {
    _avl_del_tree(r->lt);
    _avl_del_tree(r->ut);
  }
 free(r);
}

/* ************************************************************************* *
 * Concatenate two AVL trees maintaining the AVL criterium                   *
 * ************************************************************************* */
static AVLBAUM *avl_cat_tree
#ifdef ANSI_C
(AVLBAUM *r, AVLBAUM *lt, AVLBAUM *ut)
#else  /* ANSI_C */
(r, lt, ut)
AVLBAUM *r;
AVLBAUM *lt;
AVLBAUM *ut;
#endif /* ANSI_C */
{
  /* Konkatenieren zweier B"aume lt und ut, die beide AVL sind.
   * Es d"urfen nicht beide B"aume leer sein
   * Falls r nicht NULL kann r als freier Knoten verwendet werden
   * 
   */

  if ((lt == NULL) && (ut == NULL))
  {
    if (r != NULL) free(r);
    errno = EPFSLAVL;
    fprintf(stderr,"pfsd: avl_cat_tree(): Fatal error: both trees are empty\n");
    return(NULL);               /* Fehler */
  }

  /* Falls einer der B"aume leer ist liefere den anderen zur"uck 
   * und l"osche r falls r nicht NULL */
  if (lt == NULL)
  {
    if (r != NULL) free(r);
    return(ut);                 /* AVL bleibt erf"ullt */
  }
  if (ut == NULL)
  {
    lt->uc->nl = NULL;          /* Letzes Element */
    if (r != NULL) free(r);
    return (lt);                /* AVL bleibt erf"ullt */
  }


  if ((strcmp(lt->uc->leaf.ipaddr,ut->lc->leaf.ipaddr)==0) &&
      (strcmp(lt->uc->leaf.filename,ut->lc->leaf.filename) == 0) &&
      (lt->uc->leaf.p_nofs == ut->lc->leaf.p_nofs) &&
      (lt->uc->g_eop == ut->lc->g_ofs))
  {
    /* Kombination der Eckelemente
     * z.B.
     * lt: *----|---|-(
     * rt:            *-|--|----(
     * lt' *----|---|---(
     * rt'              *--|----(
     */

    /* die h"ochst ecke in lt und die kleinste Ecke in ut sind
     * Kombierbbar fasse dies zusammen
     * Entferne aus Element dem gr"o"seren Unterbaum
     * und f"uge den entsprechenden Teil beim kleineren hinzu
     */
    if (lt->h <= ut->h)
    {
      if ((lt = avl_ins_region(lt,ut->lc->g_ofs,ut->lc->g_eop,
                           ut->lc->leaf.filename,ut->lc->leaf.ipaddr,
                           ut->lc->leaf.p_nofs)) == NULL)
      {
        perror("pfsd: avl_cat_tree(): Fatal error: Deleting subtrees: avl_ins_region()");
        _avl_del_tree(ut);
        if (r != NULL) free(r);
        return (NULL);
      }
      errno = 0;
      if (((ut = _avl_del_region(ut,ut->lc->g_ofs,ut->lc->g_eop)) == NULL) &&
          (errno != 0))
      {
        perror("pfsd: avl_cat_tree(): Fatal error: deleting subtrees: _avl_del_region()");
        _avl_del_tree(lt);
        if (r != NULL) free(r);
        return (NULL);
      }
    }
    else
    {
      if ((ut = avl_ins_region(ut,lt->uc->g_ofs,lt->uc->g_eop,
                           lt->uc->leaf.filename,lt->uc->leaf.ipaddr,
                           lt->uc->leaf.p_nofs)) == NULL)
      {
        perror("pfsd: avl_cat_tree(): Fatal error: deleting subtrees: avl_ins_region()");
        _avl_del_tree(lt);
        if (r != NULL) free(r);
        return (NULL);
      }
      errno = 0;
      if (((lt = _avl_del_region(lt,lt->uc->g_ofs,lt->uc->g_eop)) == NULL) &&
          (errno != 0))
      {
        perror("pfsd: avl_cat_tree(): Fatal error: deleting subtrees: _avl_del_region()");
        _avl_del_tree(ut);
        if (r != NULL) free(r);
        return (NULL);
      }          
    }
    return(avl_cat_tree(r,lt,ut));
  }

  /* AVL Kriterium ist erf"ullt */
  if (avl(lt,ut))
  {
    /* Neuer Knoten mit beiden Teilb"aumen als Unterb"aumen
     *
     *     r
     *    / \
     *   lt ut
     */
    if (r == NULL)
      if ((r = (AVLBAUM *)malloc(sizeof(AVLBAUM))) == NULL)
      {
        perror("pfsd: avl_cat_tree(): Fatal error: deleting subtrees: malloc()");
        _avl_del_tree(lt);
        _avl_del_tree(ut);
        return (NULL);
      }
      
    lt->uc->nl = ut->lc;        /* N"achstes Blatt */

    r->g_ofs  = lt->g_ofs;
    r->g_eop  = ut->g_eop;
    r->lt  = lt;
    r->ut  = ut;
    r->lc  = lt->lc;
    r->uc  = ut->uc;
    r->cut = lt->g_eop;            /* Das ist sicher definiert, auch falls
                                   zwischen lt und ut ein Loch ist. */
    r->h   = height(r);
    return (r);                 /* AVL ist erf"ullt */
  }
  
  /* Rekursiv anwenden
   * 
   * Beispiel: h(ut) < h(lt)
   *
   *     lt                    lt
   *    /  \           -->    /  \
   * ltlt  ltut   ut       ltlt  ltut' = avl_cat_tree(r,ltut,ut)
   *
   * eventuell rotieren um die H"ohe auszugleichen:
   *
   * max{H(ltut),H(ut)} <= H(ltut') <= max{H(ltut),H(ut)} + 1
   * H(ltut) = H(lt)-{1,2}
   * H(ut) <= H(lt)-2 sonst w"are avl(lt,ut) erf"ullt
   *  ==> max{H(ltut),H(ut)} = H(ltut)
   *  ==> H(lt)-{1,2} <= H(ltut') <= H(lt)
   * 
   * Da die H"ohe von ltut' gr"o"ser als dir H"ohe von ltut kann das avl 
   * Kriterium nur verletzt sein, wenn 
   *     H(ltut') = H(ltlt)+2
   * ist.  Dann ist eine Rebalancierung notwendig.
   *
   * Falls        lt <-+
   *             /  \  |
   *      h-2 ltlt  ltut' h
   *               /    \
   *   h-{1,2} ltutlt' ltutut' h-1<*****
   *
   * links-rotation
   * 
   *                      ltut' h+{0,1}   
   *                     /    \
   *            h-{0,1} lt    ltutut' h-1
   *                   /  \
   *            h-2 ltlt ltutlt' h-{1,2}
   *
   * Falls                lt <-+
   *                    /    \ |
   *             h-2 ltlt  +->ltut' h
   *                       | /   \
   *                 h-1 ltutlt' ltutut' h-2<*****
   *                     /    \
   *       h-{2,3} ltutltlt' ltutltut' h-{3,2}
   *
   * rechts-links-rotation 
   *
   *                       ltutlt' h
   *             /                             \
   *       h-1 lt                              ltut' h-1
   *          /  \                            /     \
   *   h-2 ltlt ltutltlt' h-{2,3}  h-{3,2}ltutltut' ltutut' h-2
   *
   */
  if (lt->h > ut->h)
  {
    /* Baum ut Unterbaum ltut anf"ugen */
    if ((lt->ut = avl_cat_tree(r,lt->ut,ut)) == NULL)
    {                           /* Error in lower recursion */
      _avl_del_tree(lt);
      return (NULL);
    }

    lt->g_eop = lt->ut->g_eop;
    lt->uc = lt->ut->uc;
    lt->h  = height(lt);

    if (avl(lt->lt,lt->ut))
      return (lt);              /* AVL ist erf"ullt */

    /* AVL Kriterium nicht erf"ullt */
    if (lt->ut->ut->h == (lt->ut->h - 2))
      /* Linksrotation von a */
      return (avl_rot_left(lt));    /* AVL ist erf"ullt */
        
    return (avl_rot_rightleft(lt)); /* AVL ist erf"ullt */
  }

  /* lt->h < ut->h
     Baum lt links an utlt h"angen */
  if ((ut->lt = avl_cat_tree(r,lt,ut->lt)) == NULL)
  {                             /* Error in lower recursion */
    _avl_del_tree(ut);
    return (NULL);
  }
  
  ut->g_ofs = ut->lt->g_ofs;
  ut->lc    = ut->lt->lc;
  ut->h     = height(ut);

  if (avl(ut->ut,ut->lt))
    return (ut);                /* AVL ist erf"ullt */

  /* AVL Kriterium nicht erf"ullt */
  if (ut->lt->lt->h == (ut->lt->h - 2))
    return (avl_rot_right(ut));     /* AVL ist erf"ullt */
        
  return (avl_rot_leftright(ut));   /* AVL ist erf"ullt */
}
  
/* ************************************************************************* *
 * Delete a file region from an AVL tree maintaining the AVL criterium       *
 * ************************************************************************* */
AVLBAUM *_avl_del_region
#ifdef ANSI_C
(AVLBAUM *r, off_t g_ofs, off_t g_eop)
#else  /* ANSI_C */
(r, g_ofs, g_eop)
AVLBAUM *r;
off_t g_ofs;
off_t g_eop;
#endif /* ANSI_C */
{
  if (r == NULL)
    return (NULL);

  if (r->h == 0)
  {
    /* Blatt */

    if ((r->g_ofs >= g_ofs) && (r->g_eop <= g_eop))
    {
      /* 
       * r:       *-----(       *-----(
       * region: *-------(      *-----(
       *
       *  Das ganze Blatt l"oschen */
      free(r);
      return (NULL);            /* AVL bleibt erf"ullt */
    }
    
    /* 
     * r:         *-----(       *-----(
     * region:       *----(       *---(
     * r':        *--(          *-(
     */
    if ((r->g_eop <= g_eop) && (r->g_eop > g_ofs))
    {
      r->g_eop = g_ofs;
      return(r);                /* AVL blebt erf"ullt */
    }

    /* 
     * r:         *-----(       *-----(
     * region:  *----(          *--(
     * r':           *--(          *--(
     */
    if ((r->g_ofs >= g_ofs) && (r->g_ofs < g_eop))
    {
      r->g_ofs = g_ofs;
      return (r);               /* AVL bleibt erf"ullt */
    }
    
    /* r:        *------(
     * region:     *--(
     * ist unm"oglich, da nur von vorne oder hinten etwas gel"oscht
     * werden kann
     */
    if ((r->g_ofs < g_ofs) && (r->g_eop > g_eop))
    {
      errno = EPFSLAVL;
      return (NULL);
    }
    
    /* Keine "Uberscneidung */
    return (r);                 /* AVL bleibt erf"ullt */
  }
  
  /* Baum
   *
   * r: *----|----(
   */
  
  if ((r->g_ofs >= g_ofs) && (r->g_eop <= g_eop))
  {
    /* Ganzer Baum
     *
     * a:       *---|---(
     * region: *---------(
     *
     * L"osche ganzen Baum
     */
    _avl_del_tree(r);
    return (NULL);              /* AVL bleibt erf"ullt */
  }
  
  if (g_eop <= r->cut)
  {
    /* Betrifft nur linken Teilbaum
     * 
     * r:        *----|----(
     * region: *----(
     */
    
    r->lt = _avl_del_region(r->lt,g_ofs,g_eop);
  }
  else if (g_ofs >= r->cut)
  {
    /* Betrifft nur rechten Teilbaum
     * 
     * a:        *----|----(
     * region:          *----(
     */
    r->ut = _avl_del_region(r->ut,g_ofs,g_eop);
  }
  else
  {
    /* Betrifft beide Teilb"aume
     * 
     * a:        *----|----(
     * region:      *----(
     */
  
    r->lt = _avl_del_region(r->lt,g_ofs,g_eop);
    r->ut = _avl_del_region(r->ut,g_ofs,g_eop);
  }  
  
  return(avl_cat_tree(r,r->lt,r->ut));
}


/* ************************************************************************* *
 * Insert a file region in an AVL tree maintaining the AVL criterium.        *
 * ************************************************************************* */
static AVLBAUM *avl_ins_region
#ifdef ANSI_C
(AVLBAUM *r, off_t g_ofs, off_t g_eop, char *filename, char *ipaddr, 
 off_t p_nofs)
#else  /* ANSI_C */
(r, g_ofs, g_eop, filename, ipaddr, p_nofs)
AVLBAUM *r;
off_t g_ofs;
off_t g_eop;
char *filename;
char *ipaddr;
off_t p_nofs;
#endif /* ANSI_C */
{
  /* r erf"ullt AVL */

  AVLBAUM *l;
  AVLBAUM *m;
  
  if (r == NULL)
  {
    /* Neuen Baum erzeugen */
    return (avl_new_leaf(g_ofs, g_eop, ipaddr, filename, p_nofs)); /* AVL OK */
  }
  
  if (r->h == 0)
  {
    /* Blatt */
    if ((r->g_ofs >= g_ofs) && (r->g_eop <= g_eop))
    {
      /* "Uberschreibt Blattinhalt
       * r:         *----(
       * region:   *------(
       * r':       *------(
       */
      r->g_ofs = g_ofs;
      r->g_eop = g_eop;
      strcpy(r->leaf.ipaddr,ipaddr);
      strcpy(r->leaf.filename,filename);
      r->leaf.p_nofs = p_nofs;
      return (r);               /* AVL OK */
    }
    /* r:           *-----(
     * region:     **--(
     *  oder            *-((
     *  oder          *-(       */
    if ((g_ofs <= r->g_eop) &&
        (strcmp(r->leaf.ipaddr, ipaddr) == 0) &&
        (strcmp(r->leaf.filename,filename) == 0) &&
        (r->leaf.p_nofs == p_nofs))
    {
      /* "Uberschneidung in einem Blatt ==> Erweiterung */
      if (r->g_ofs > g_ofs)
        r->g_ofs = g_ofs;
      if (r->g_eop < g_eop)
        r->g_eop = g_eop;
      return (r);               /* AVL OK */
    }
    
    /* Neues Blatt ist notwendig f"ur Region */
    if ((l = avl_new_leaf(g_ofs, g_eop, ipaddr, filename, p_nofs)) == NULL)
    {
      perror("pfsd: avl_ins_region(): Fatal error: deleting subtree: avl_new_leaf()");
      _avl_del_tree(r);
      return (NULL);
    }
    
    if (l->g_ofs <= r->g_ofs)
    {
      if (l->g_eop >= r->g_ofs)
      {
        /* r:          *----(
         * region:   *---(
         * l:        *---(
         * r':           *--(
         *           *---|--(
         * niedriger ein Blatt dazu */
    
        r->g_ofs = l->g_eop;
        return (avl_cat_tree(NULL,l,r)); /* AVL OK */
      }
      /* r:                *--(
       * region:     *--(
       * l:          *--(
       * r':               *--(
       * m:/dev/zero    *--(
       *             *--|--|--(
       */
      
      /* Neues Blatt */
      if ((m = avl_new_leaf(l->g_eop, r->g_ofs, LOCALHOST, DEV_ZERO,(off_t)0))
          == NULL)
      {
        perror("pfsd: avl_ins_region(): Fatal error: deleting subtrees: avl_new_leaf()");
        _avl_del_tree(l);
        _avl_del_tree(r);
        return (NULL);
      }
    
      if ((l = avl_cat_tree(NULL,l,m)) == NULL)
      {
        perror("pfsd: avl_ins_region(): Fatal error: deleting subtree: avl_cat_tree()");
        _avl_del_tree(r);
        return (NULL);
      }
      return (avl_cat_tree(NULL,r,l)); /* AVL OK */
    }
    
    /* l->g_ofs > r->g_ofs */
    if (l->g_eop >= r->g_eop)
    {
      if (l->g_ofs <= r->g_eop)
      {
        /* r:      *----(
         * region:    *----(
         * l:         *----(
         * r':     *--(
         *         *--|----(
         * h"oher ein Blatt dazu */
        r->g_eop = l->g_ofs;
        return (avl_cat_tree(NULL,r,l)); /* AVL OK */
      }
      /* r:          *---(
       * region:            *----(
       * l:                 *----(
       * r':         *---(
       * m:/dev/zero     *-(
       *             *---|--|----(
       */

      /* Neues Blatt */
      if ((m = avl_new_leaf(r->g_eop,l->g_ofs,LOCALHOST,DEV_ZERO,(off_t)0))
          == NULL)
      {
        perror("pfsd: avl_ins_region(): Fatal error: deleting subtrees: avl_new_leaf():");
        _avl_del_tree(l);
        _avl_del_tree(r);
        return (NULL);
      }
    
      if ((m = avl_cat_tree(NULL,m,l)) == NULL)
      {
        perror("pfsd: avl_ins_region(): Fatal error: deleting subtree: avl_cat_tree()");
        _avl_del_tree(r);
        return (NULL);
      }
      return (avl_cat_tree(NULL,r,m)); /* AVL OK */
    }
    
    /* r:       *------(
     * region:    *--(
     * l:         *--(
     * r':      *-(
     * m:            *-(
     *          *-|--|-(
     * In die Mitte einf"ugen */

    /* Neues Blatt */
    if ((m = avl_new_leaf(l->g_eop,r->g_eop,
                      r->leaf.ipaddr,r->leaf.filename,r->leaf.p_nofs))
        ==NULL)
    {
      perror("pfsd: avl_ins_region(): Fatal error: deleting subtrees: avl_new_leaf()");
      _avl_del_tree(l);
      _avl_del_tree(r);
      return (NULL);
    }
    
    r->g_eop = l->g_ofs;

    if ((l = avl_cat_tree(NULL,l,m)) == NULL)
    {
      perror("pfsd: avl_ins_region: Fatal error: deleting subtree: avl_cat_tree()");
      _avl_del_tree(r);
      return (NULL);
    }
    return (avl_cat_tree(NULL,r,l)); /* AVL OK */
  }
  
  /* r ist Baum   *-----|-----(
   * r->lt        *--?--(         Baum oder Blatt
   * r->ut              *--?--(   Baum oder Blatt
   */
  
  if ((g_ofs <= r->g_ofs) && (g_eop >= r->g_eop))
  {
    /* "Uberschreibt ganzen Baum
     * 
     * r:        *----|---(
     * region:  *----------( 
     * r':      *----------(
     */
    
    _avl_del_tree(r->lt);
    _avl_del_tree(r->ut);
    r->h = 0;
    r->g_ofs = g_ofs;
    r->g_eop = g_eop;
    r->uc = r;
    r->lc = r;
    r->lt = NULL;
    r->ut = NULL;
    r->nl = NULL;
    strcpy(r->leaf.ipaddr,ipaddr);
    strcpy(r->leaf.filename,filename);
    r->leaf.p_nofs = p_nofs;
    
    return (r);
  }
  
  if (g_eop <= r->cut)
  {
    /* Betrifft nur linken Teilbaum
     * 
     * r:        *---|----(
     * region: *---(
     * r->lt': *---?-(
     * r'      *-----|----(
     *
     */
    if ((r->lt = avl_ins_region(r->lt,g_ofs,g_eop,filename,ipaddr,p_nofs))
        == NULL)
    {
      perror("pfsd: avl_ins_region(): Fatal error: deleting subtree: avl_ins_region()");
      _avl_del_tree(r);
      return (NULL);
    }
  }
  else if (g_ofs >= r->cut)
  {
    /* Betrifft nur rechten Teilbaum analog */
    if ((r->ut = avl_ins_region(r->ut,g_ofs,g_eop,filename,ipaddr,p_nofs))
        == NULL)
    {
      perror("pfsd: avl_ins_region(): Fatal error: deleting subtree: avl_ins_region()");
      _avl_del_tree(r);
      return (NULL);
    }
  }
  else 
  {
    /* Betrifft beide Teilb"aume
     *
     * r:      *----|----(          *----|----(       *----|----(
     * region:   *---------(      *---------(           *-----(
     * r->lt': *-?--(             *-?----(            *-?--(
     * r->ut':      *----?-(             *--?-(            *--?-(
     * r':     *----|------(      *------|----(       *----|----(
     */

    if ((r->lt = avl_ins_region(r->lt,g_ofs,r->cut,filename,ipaddr,p_nofs))
        == NULL)
    {
      perror("pfsd: avl_ins_region(): Fatal error: deleting subtrees: avl_ins_region()");
      _avl_del_tree(r);
      return (NULL);
    }
  
    if ((r->ut = avl_ins_region(r->ut,r->cut,g_eop,filename,ipaddr,p_nofs))
        == NULL)
    {
      perror("pfsd: avl_ins_region(): Fatal error: deleting subtrees: avl_ins_region()");
      _avl_del_tree(r);
      return (NULL);
    }
  }
  
  return (avl_cat_tree(r,r->lt,r->ut));
}

/* ************************************************************************* *
 * Insert a file region in an AVL tree and make sure there is no             *
 * undefined region at the beginning.                                        *
 * ************************************************************************* */
AVLBAUM *_avl_insert_region
#ifdef ANSI_C
(AVLBAUM *r, off_t g_ofs, off_t g_len, char *filename, char *ipaddr, 
 off_t p_ofs)
#else  /* ANSI_C */
(r, g_ofs, g_len, filename, ipaddr, p_ofs)
AVLBAUM *r;
off_t g_ofs;
off_t g_len;
char *filename;
char *ipaddr;
off_t p_ofs;
#endif /* ANSI_C */
{
  AVLBAUM *lt;
  AVLBAUM *ut;

  if ((r == NULL) && (g_ofs != 0))
  {
    if ((lt = avl_ins_region(r,(off_t)0,g_ofs,DEV_ZERO,LOCALHOST,(off_t)0))
        == NULL)
      return (NULL);
    if ((ut = avl_ins_region(r,g_ofs,g_ofs+g_len,filename,ipaddr,g_ofs-p_ofs))
        == NULL)
    {
      _avl_del_tree(lt);
      return (NULL);
    }
    return(avl_cat_tree(NULL,lt,ut));
  }
  
  return (avl_ins_region(r,g_ofs,g_ofs + g_len,filename,ipaddr,g_ofs - p_ofs));
}

/* ************************************************************************* *
 * Get a leaf of an AVL tree for a certain offset in the file.               *
 * ************************************************************************* */
AVLBAUM *_avl_get_leaf
#ifdef ANSI_C
(AVLBAUM *r, off_t g_ofs)
#else  /* ANSI_C */
(r, g_ofs)
AVLBAUM *r;
off_t g_ofs;
#endif /* ANSI_C */
{
  AVLBAUM *tr;
  
  if (r == NULL) return (NULL);
  
  if (g_ofs >= r->g_eop) return(NULL); /* Offset not in tree */

  for (tr = r; tr->h > 0; tr = (g_ofs < tr->cut) ? tr->lt : tr->ut);
  
  return (tr);
}

