/* ************************************************************************* *
 *                                                                           *
 *    iod_signal.c,v
 *    iod procedures to handle signals
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : iod_signal.c,v
 *  RCS Date     : 1996/05/30 17:34:18
 *  RCS Revision : 1.4
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts                                                 *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "iod_signal.c,v 1.4 1996/05/30 17:34:18 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern pid_t iodpid;

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern void  unregister _PH((void));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Handler for SIGCHLD signals                                               *
 * ************************************************************************* */
void iod_sigchld
#ifdef ANSI_C
(int signo)
#else  /* ANSI_C */
(signo)
int signo;
#endif /* ANSI_C */
{
  int state;
  int pid;

  while ((pid = waitpid(-1,&state,WNOHANG)) != 0)
  {
    if ((pid < 0) && (errno == ECHILD))
      break;

    if (pid < 0)
      perror("iod: _iod_sigchld(): waitpid()");
    else
    {
      if (WIFSIGNALED(state))
        fprintf(stderr,"iod: Child process %d terminated by signal %d\n",
                pid, WTERMSIG(state));
      else if (WIFEXITED(state))
      {
        if (WEXITSTATUS(state) != 0)
          fprintf(stderr,"iod: Child process %d terminated unsuccessfully with status %d\n",
                  pid, WEXITSTATUS(state));
#ifdef DEBUG
        else
          fprintf(stdout,"iod: Child process %d terminated successfully\n",
                  pid);
#endif /* DEBUG */
      }
      else if (WIFSTOPPED(state))
      {
        fprintf(stderr,"iod: Child process %d stopped by signal %d\n",
                pid,WSTOPSIG(state));
      }
      else 
      {
        fprintf(stderr,"iod: Received SIGCHLD for child process %d for unknown reason\n",
                pid);
      }
    }
  }

  return;
}


/* ************************************************************************* *
 * Signal handler for all signals causing a process termination              *
 * ************************************************************************* */
static void sighandler
#ifdef ANSI_C
(int signo)
#else  /* ANSI_C */
(signo)
int signo;
#endif /* ANSI_C */
{
#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction  sigact;
#endif

  fprintf(stderr,"iod: _sighandler(): Received signal: exiting\n");

  /* Reset the signal handler */
#ifdef BSD_SIG
  vec.sv_handler = void(*)()0;
  vec.sv_mask = 0;
  vec.sv_flags = SV_RESETHAND;
  if (sigvec(signo, &vec, (struct sigvec *)0) != 0)
  {
    perror("iod: _sighandler(): sigvec()");
    exit(1);
  }
#else  /* ! BSD_SIG */
  sigact.sa_handler = SIG_DFL;
  sigact.sa_flags = 0;
  if (sigaction(signo, &sigact, (struct sigaction *)0) != 0)
  {
    perror("iod: _sighandler(): sigaction()");
    exit(1);
  }
#endif /* BSD_SIG */
  
  if (iodpid == getpid())
  {
    unregister();
  }

  kill(getpid(),signo);
}


/* ************************************************************************* *
 * Activate the signal handler for all fatal signals                         *
 * ************************************************************************* */
static int catch_sigs
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  
#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction  sigact;
#endif
  int i;
  int sig[] =
  {
    SIGHUP,
    SIGINT,
    SIGQUIT,
    SIGILL,
    SIGTRAP,
    SIGABRT,
#ifndef LINUX
    SIGEMT,
#else
    SIGUNUSED,
#endif
    SIGFPE,
#ifndef LINUX
    SIGBUS,
#endif
    SIGSEGV,
#ifndef LINUX
    SIGSYS,
#endif
    SIGPIPE,
    SIGALRM,
    SIGTERM,
#ifdef LINUX
    SIGSTKFLT,
#endif
#ifndef HPPA
    SIGXCPU,
    SIGXFSZ,
#endif
    SIGVTALRM,
    SIGPROF,
    SIGUSR1,
    SIGUSR2,
#ifdef RS6K
    SIGMIGRATE,
    SIGPRE,
#endif
    -1
  };
  
  for (i=0; sig[i] > 0; i++)
  {
#ifdef BSD_SIG
    vec.sv_handler = sighandler;
    vec.sv_mask = sigmask(sig[i]);
    vec.sv_flags = 0;
    if (sigvec(sig[i], &vec, (struct sigvec *)0) != 0)
    {
      perror("pfsd: chatch_sigs(): sigvec()");
      return (-1);
    }
#else  /* ! BSD_SIG */
    sigact.sa_handler = sighandler;
    sigact.sa_flags = 0;
    if ((sigemptyset(&sigact.sa_mask) != 0) ||
        (sigaddset(&sigact.sa_mask,sig[i]) != 0) ||
        (sigaction(sig[i], &sigact, (struct sigaction *)0) != 0))
    {
      perror("pfsd: chatch_sigs():sigaction()");
      return (-1);
    }
#endif /* BSD_SIG */
  }

  return (0);
}



/* ************************************************************************* *
 * Activate the signal handlers                                              *
 * ************************************************************************* */
int iod_init_signal
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{

#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction  sigact;
#endif

#ifdef BSD_SIG
  vec.sv_handler = iod_sigchld;
  vec.sv_mask = sigmask(SIGCHLD);
  vec.sv_flags = 0;
  if (sigvec(SIGCHLD, &vec, (struct sigvec *)0) != 0)
  {
    perror("iod: _iod_init_signal(): sigvec()");
    return (-1);
  }
#else  /* ! BSD_SIG */
  sigact.sa_handler = iod_sigchld;
  sigact.sa_flags = 0;
  if ((sigemptyset(&sigact.sa_mask) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGCHLD) != 0) ||
      (sigaction(SIGCHLD, &sigact, (struct sigaction *)0) != 0))
  {
    perror("iod: _iod_init_signal():sigaction()");
    return (-1);
  }
#endif /* BSD_SIG */
  
  return (catch_sigs());
}
