/* ************************************************************************* *
 *                                                                           *
 *    iod_read.c,v
 *    iod procedures to read from a file
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : iod_read.c,v
 *  RCS Date     : 1996/07/18 10:21:08
 *  RCS Revision : 1.7
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "iod_read.c,v 1.7 1996/07/18 10:21:08 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#ifdef MMAP_READ
#include <sys/mman.h>
#endif /* MMAP_READ */

#include "iod.h"
#include "pfslib_errno.h"

#ifdef __GNUC__
#define alloca __builtin_alloca
#else  /* __GNUC__ */
#ifdef RS6K
#pragma alloca
#endif /* RS6K */
#ifdef SUN4
#include <alloca.h>
#endif /* SUN4 */
#endif /* __GNUC */

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Read data from a file and send it to the client                           *
 * ************************************************************************* */
iod_ioopres *iod_read_1
#ifdef ANSI_C
(iod_ioopargs *argp, struct svc_req *rqstp)
#else  /* ANSI_C */
(argp, rqstp)
iod_ioopargs *argp;
struct svc_req *rqstp;
#endif /* ANSI_C */
{
  static iod_ioopres result;
  char *buff;

  int fd;

  off_t eof;
  int  rlen;

  int pid = 1;
  
#ifdef MMAP_READ
  off_t mmapofs;
  size_t mmaplen;
  caddr_t mmapbuff;
#ifdef SGI
  long pagesize = sysconf(_SC_PAGESIZE);
#else  /* SGI */
  long pagesize = sysconf(_SC_PAGE_SIZE);
#endif /* SGI */
#else  /* MMAP_READ */
  off_t ofs;
#endif /* MMAP_READ */

#ifdef FILE_LOCKING
  struct flock fl;
#endif /* FILE_LOCKING */

#ifdef TCP_TRANSFER
  int tw;
  int wt;
  int rd;
  char go;
#endif /* TCP_TRANSFER */

  result.stat = RPC_OK;
  result.iod_ioopres_u.res.data.iod_filedata_len = 0;
  result.iod_ioopres_u.res.data.iod_filedata_val = NULL;

  if (argp->length == 0)
  {
    return (&result);
  }    

  /* Fork a process above threshold */
  if (argp->length >= argp->thresh)
  {
    if ((pid = fork()) < 0)
    {
      perror("iod: iod_read_1(): fork()");
      ERR_MACRO(iod_ioopres_u,errno);
      return(&result);
    }
    if (pid != 0)
    {
      /* Parent process returns */
#ifdef TCP_TRANSFER
      close(rqstp->rq_xprt->xp_sock);
#endif
#ifdef DEBUG
      fprintf(stdout,"iod: iod_read_1(): forked process %d\n",pid);
#endif /* DEBUG */
      return (NULL);
    }
  }
  
  /* Open the file */
  if ((fd = open(argp->fn, argp->oflags)) < 0)
  {
    perror("iod: iod_read_1(): open()");
    ERR_MACRO(iod_ioopres_u,errno);
  }

  /* Check if the file is long enough */
  else if ((eof = lseek(fd,0,SEEK_END)) < 0) /* Seek at EOF */
  {
    perror("iod: iod_read_1(): lseek()");
    ERR_MACRO(iod_ioopres_u,errno);
  }
  else
  {                             /* open() and lseek(END) ok */
    if (eof < argp->offset)
    {
      /* offset past eof */
      rlen = 0;
      buff = NULL;
    }
    else
    {                           /* rlen != 0 */
      if (eof < (argp->offset + argp->length))
        /* File is too short */
        rlen = eof - argp->offset;
      else
        rlen = argp->length;
      buff = NULL;

#ifdef MMAP_READ

      mmapofs = (argp->offset / pagesize) * pagesize;
      mmaplen = rlen + (argp->offset % pagesize);
    
      /* Map part of the file into memory */
      if ((mmapbuff = mmap(NULL,mmaplen,PROT_READ,
#ifdef SGI
                           MAP_SHARED,
#else  /* SGI */
                           MAP_FILE|MAP_VARIABLE|MAP_SHARED,
#endif /* SGI */
                           fd,mmapofs))
          == (caddr_t)-1)
      {
        perror("iod: iod_read_1(): mmap()");
        ERR_MACRO(iod_ioopres_u,errno);
      }
      else
      {
        buff = (char *)mmapbuff + (argp->offset % pagesize);
      }

#else  /* MMAP_READ */

      /* Seek to correct position */
      if ((ofs = lseek(fd,argp->offset, SEEK_SET)) < 0)
      {
        perror("iod: iod_read_1(): lseek()");
        ERR_MACRO(iod_ioopres_u,errno);
      }
      else if (ofs != argp->offset)
      {
        fprintf(stderr,"iod: iod_read_1(): Coudn't lseek() to correct position\n");
        ERR_MACRO(iod_ioopres_u,EPFSLSEEK);
      }
      else if ((buff = alloca((u_int)rlen)) == NULL)
      {
        perror("iod: iod_read_1(): alloca()");
        ERR_MACRO(iod_ioopres_u,errno);
      }

#endif /* MMAP_READ */

#ifdef FILE_LOCKING

      if (result.stat == RPC_OK)
      {                         /* mmap() or (lseek(SET) and alloca()) ok*/

        fl.l_type   = F_RDLCK;
        fl.l_whence = SEEK_SET;
        fl.l_start  = argp->offset;
        fl.l_len    = argp->length;

        if (fcntl(fd,F_SETLKW,&fl) < 0)
        {
          perror("iod: iod_read_1(): fcntl(F_RDLCK)");
          ERR_MACRO(iod_ioopres_u,errno);
        }
      }

#endif /* FILE_LOCKING */

#ifndef MMAP_READ

      /* Read into buffer */
      if (result.stat == RPC_OK)
      {                         /* (mmap() and fcntl(RDLCK)) or
                                   (lseek(SET) and alloca() and fcntl(RDLCK))
                                   ok */
        if ((rlen = read(fd,buff,(u_int)rlen)) < 0)
        {
          perror("iod: iod_read(): read()");
          ERR_MACRO(iod_ioopres_u,errno);
        }
      }

#endif /* !MMAP_READ */
    } /* else: rlen != 0 */
    
    if (result.stat == RPC_OK)
    {                           /* all calls so far ok */
#ifdef TCP_TRANSFER
      result.iod_ioopres_u.res.data.iod_filedata_val = NULL;
      result.iod_ioopres_u.res.data.iod_filedata_len = 0;
      result.iod_ioopres_u.res.length = rlen;
#else  /* TCP_TRANSFER */
      result.iod_ioopres_u.res.data.iod_filedata_val = buff;
      result.iod_ioopres_u.res.data.iod_filedata_len = rlen;
      result.iod_ioopres_u.res.length = rlen;
#endif /* TCP_TRANSFER */
    }
  } /* open() and lseek(fd,0,SEEK_END) ok */
  

  /* Send reply */
  if (!svc_sendreply(rqstp->rq_xprt, xdr_iod_ioopres, (char *)&result))
  {
    svcerr_systemerr(rqstp->rq_xprt);
  }
#ifdef TCP_TRANSFER
  else
  {
    while(((rd = read(rqstp->rq_xprt->xp_sock, &go, 1)) < 0) &&
          (errno == EWOULDBLOCK));
    if (rd < 0)
    {
      perror("iod: iod_read_1(): read(go)");
      ERR_MACRO(iod_ioopres_u,errno);
    }
    else if (rd == 0)
    {
      fprintf(stderr,"iod: iod_read_1(): read(go) returned 0\n");
      ERR_MACRO(iod_ioopres_u,ESHUTDOWN);
    }
    else
    {
      tw = rlen;
      while(tw > 0)
      {
        if ((wt = write(rqstp->rq_xprt->xp_sock,
                        (void *)buff, (unsigned int)tw)) < 0) 
        {
          if (errno != EWOULDBLOCK) /* If write() would block try again */
          {
            perror("iod: iod_read: write()");
            break;
          }
        }
        else if (wt == 0) 
        {
          fprintf(stderr,"iod: iod_read(): write() returned 0\n");
          break;
        }
        tw -= wt;
        buff += wt;
      }
    }
  }
#endif /* TCP_TRANSFER */

#ifdef MMAP_READ
  if (munmap(mmapbuff,mmaplen) < 0)
  {
    perror("iod: iod_read_1(): munmap()");
  }
#endif /* MMAP_READ */

  if (fd > 0)
    close(fd);
  
  if (pid == 0)
  {
    /* Child process */
    _exit(0);
  }
  
  /* Parent process */
  return(NULL);
}
