/*****************************************************************************
 *                                                                           *
 *  RCS Filename : meas.c,v
 *  RCS Date     : 1996/07/12 16:05:24
 *  RCS Revision : 1.4
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts                                                 *
 *                                                                           *
 *****************************************************************************/

#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <pfslib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <stdlib.h>
#include <math.h>
#include <nx.h>

#define USAGE "\
\t[-r <Resultfile>]\n\
\t[-f <Filename>]\n\
\t[-h <pfsdHost>]\n\
\t[-d <number of io Daemons, i.e. number of machines>]\n\
\t[-D {a|s}] /* -D a: iod forks , s: iod dosn't fork */\n\
\t[-c {a|s}] /* -c a: Client forks for asynchronous IO */\n\
\t[-s|-b]    /* -s Set O_SYNC flag in open */\n\
\t[-e <number of External iterations>]\n\
\t[-i <number of Internal iterations>]\n\
\t[-n <miNimum number of bytes>]\n\
\t[-x <maXimum number of bytes>]\n\
\t[-t <sTep of number of bytes>]\n\
\t[-m <Mode string of the form [glrsu]+>]\n\
\t[-o <Operations> ..... (must be last argument)]\n\
\t Operations:\n\
\tCWR    /* Starttime cwrite() Endtime */\n\
\tCRD    /* Starttime cread()  Endtime */\n\
\tIWRW   /* Starttime iwrite() iowait() Endtime */\n\
\tIRDW   /* Starttime iread()  iowait() Endtime */\n\
\tIWRIOW /* Starttime iwrite() Endtime Starttime iowait() Endtime */\n\
\tIRDIOW /* Starttime iread()  Endtime Starttime iowait() Endtime */\n\
\tIWR    /* Starttime iwrite() Endtime iowait() */\n\
\tIRD    /* Starttime iread()  Endtime iowait() */\n\
\tWIOW,  /* iwrite() Starttime iowait() Endtime */\n\
\tRIOW   /* iread()  Starttime iowait() Endtime */"

#define BUFFSIZE       2097152

#define OUTPUTLINESIZE 1000

#define MAXINTITER     20       /* Maximal outstanding io operations */

#define IOD_SYNC  0
#define IOD_ASYNC 1

#define CLNT_SYNC  0
#define CLNT_ASYNC 1

#define DFLTRESULTFILE "/tmp/pfslibresults"
#define DFLTFILENAME   "/tmp/pfslibtest"
#define DFLTPFSDHOST   "ibode3"
#define DFLTMACHNUMBER 0
#define DFLTIOD_SYNC   IOD_SYNC
#define DFLTCLNT_SYNC  CLNT_SYNC
#define DFLTEXTITER    1
#define DFLTINTITER    20
#define DFLTMINIMUM    1
#define DFLTMAXIMUM    100000
#define DFLTSTEP       0
#define DFLTMODES      "glrsu"
#define DFLTO_SYNC     0x0

char *dfltopargs[] = {"CWR","CRD","IWRW","IRDW","IWRIOW","IRDIOW",(char *)0};

extern char *optarg;
extern int optind;

int machnumber = -1;
int iod_sync   = -1;
int clnt_sync  = -1;
int o_sync     = -1;
int mode;

char buffer [BUFFSIZE];

int resultfd;

int filedes;


enum optype
{
  NO_OP,
  CWR,
  CRD,
  IWRW,
  IRDW,
  IWRIOW,
  IRDIOW,
  IWR,
  IRD,
  WIOW,
  RIOW
};
typedef enum optype optype;


#define EMEXIT 100

int inemexit=0;
int doemexit=1;

void recv_emexit
#ifdef ANSI_C
(long type, long count, long node, long ptype)
#else  /* ANSI_C */
(type, count, node, ptype)
long type;
long count;
long node;
long ptype;
#endif /* ANSI_C */
{
  inemexit=1;
  _dbgmsg("Received emergency exit message from (node %ld): exiting",node);
  exit(1);
}

void send_emexit
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  masktrap(1);
  if ((!inemexit) && (doemexit))
  {
    inemexit=1;
    _dbgmsg("Sending emergency exit messages");
    csend(EMEXIT,NULL,0,-1,0);
  }
}


int init_emexit
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  if ((_hrecv(EMEXIT,NULL,0,recv_emexit) < 0) ||
      (atexit(send_emexit) != 0))
    return (-1);

  gsync();

  return (0);
}
  

char *mode2str
#ifdef ANSI_C
(int mode)
#else
(mode)
int mode;
#endif
{
  
  return (mode == M_UNIX   ? "M_UNIX"   :
          mode == M_LOG    ? "M_LOG"    :
          mode == M_RECORD ? "M_RECORD" :
          mode == M_SYNC   ? "M_SYNC"   :
          mode == M_GLOBAL ? "M_GLOBAL" :
          "UNKOWN");
}

char mode2char
#ifdef ANSI_C
(int mode)
#else
(mode)
int mode;
#endif
{
  
  return (mode == M_UNIX   ? 'U' :
          mode == M_LOG    ? 'L' :
          mode == M_RECORD ? 'R' :
          mode == M_SYNC   ? 'S' :
          mode == M_GLOBAL ? 'G' :
          'X');
}

optype str2op
#ifdef ANSI_C
(char *str)
#else
(str)
char *str;
#endif
{
  if (!strcmp(str,"CWR"))
    return(CWR);
  if (!strcmp(str,"CRD"))
    return(CRD);
  if (!strcmp(str,"IWRW"))
    return(IWRW);
  if (!strcmp(str,"IRDW"))
    return(IRDW);
  if (!strcmp(str,"IWRIOW"))
    return(IWRIOW);
  if (!strcmp(str,"IRDIOW"))
    return(IRDIOW);
  if (!strcmp(str,"IWR"))
    return(IWR);
  if (!strcmp(str,"IRD"))
    return(IRD);
  if (!strcmp(str,"WIOW"))
    return(WIOW);
  if (!strcmp(str,"RIOW"))
    return(RIOW);

  return(NO_OP);
}

char *op2str
#ifdef ANSI_C
(optype op)
#else
(op)
optype op;
#endif
{
  switch (op)
  {
  case CWR:
    return("CWR");
  case CRD:
    return("CRD");
  case IWRW:
    return("IWRW");
  case IRDW:
    return("IRDW");
  case IWRIOW:
    return("IWRIOW");
  case IRDIOW:
    return("IRDIOW");
  case IWR:
    return("IWR");
  case IRD:
    return("IRD");
  case WIOW:
    return("WIOW");
  case RIOW:
    return("RIOW");
  }
  
  return("NO_OP");
}  

void write_head
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  char output[OUTPUTLINESIZE];

  setiomode(resultfd,M_GLOBAL);

  sprintf(output,"#\n");

  cwrite(resultfd,output,strlen(output));

  sprintf(output,"#  OP    : Operation\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  MA    : Number od Machines\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  NS    : Number of Nodes\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  ND    : Nuber of Node; if ND = NS All Nodes\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  C     : S=Client synchronous, A=Client asynchronous\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  I     : S=Iod synchronous, A=Iod asynchronous\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  O     : T=O_SYNC flags set, F=O_SYNC not set\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  M     : IO mode G=M_GLOBAL, L=M_LOG, R=M_RECORD, S=M_SYNC, U=M_UNIX\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  EI    : External Iterations\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  II    : Internal Iterations\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  BYTES : Number of bytes\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  MAX   : Maximum time in seconds\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  MIN   : Minimum time in seconds\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  AVG   : Average time in secongs\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#  STD   : Standard Diviation\n");
  cwrite(resultfd,output,strlen(output));
  
  sprintf(output,"#\n");
  cwrite(resultfd,output,strlen(output));

  sprintf(output,"# %4s %2s %2s %2s %c %c %c %c %4s %2s %7s %9s %9s %9s %9s\n",
          "OP","MA","NS","ND",'C','I','O','M',"EI","II","BYTES","MIN","MAX",
          "AVG","STD");
  cwrite(resultfd,output,strlen(output));

  sprintf(output,"#\n");
  cwrite(resultfd,output,strlen(output));

}


void write_res
#ifdef ANSI_C
(char *op, int extiter, int intiter, int size, double min, double max, 
 double total, double tsq, double nmin, double nmax, double ntotal, 
 double ntsq)
#else  /* ANSI_C */
(op, extiter, intiter, size, min, max, total, tsq, nmin, nmax, ntotal, 
 ntsq)
char *op;
int extiter;
int intiter;
int size;
double min;
double max;
double total;
double tsq;
double nmin;
double nmax;
double ntotal;
double ntsq;
#endif /* ANSI_C */
{
  double naver;
  double nstd;
  double aver;
  double std;
  
  char output[OUTPUTLINESIZE];
  
  naver = ntotal/(double)extiter;

  if (extiter == 1)
    nstd = 0.0;
  else
    nstd = sqrt((ntsq-(ntotal*naver))/(double)(extiter - 1));
  
  aver = (total/(double)extiter);
  if (extiter == 1)
    std = 0.0;
  else
    std = sqrt((tsq-(total*aver))/(double)(extiter - 1));

 /*               OP  MA  CLT  MYN CA IA OS M  EI  II  SIZ MIN   MAX   AVG   STD*/
  sprintf(output,"%6s %2d %2ld %2d %c %c %c %c %4d %2d %7d %9.4f %9.4f %9.4f %9.4f\n",
          op2str(op),
          machnumber,
          numnodes(),
          mynode(),
          (clnt_sync == CLNT_SYNC) ? 'S' : 'A',
          (iod_sync == IOD_SYNC) ? 'S' : 'A',
          (o_sync == O_SYNC) ? 'T' : 'F',
          mode2char(mode),
          extiter,
          intiter,
          size,
          min,
          max,
          aver,
          std);

  setiomode(resultfd,M_SYNC);
  cwrite(resultfd,output,strlen(output));

 /*               OP  MA  CLT  MYN CS IA OF M  EI  II  SIZ MIN   MAX   AVG   STD*/
  sprintf(output,"%6s %2d %2ld %2d %c %c %c %c %4d %2d %7d %9.4f %9.4f %9.4f %9.4f\n",
          op2str(op),
          machnumber,
          numnodes(),
          numnodes(),
          (clnt_sync == CLNT_SYNC) ? 'S' : 'A',
          (iod_sync == IOD_SYNC) ? 'S' : 'A',
          (o_sync == O_SYNC) ? 'T' : 'F',
          mode2char(mode),
          extiter,
          intiter,
          size,
          nmin,
          nmax,
          naver,
          nstd);

  setiomode(resultfd,M_GLOBAL);
  cwrite(resultfd,output,strlen(output));
  
  return;
}


void meas_op
#ifdef ANSI_C
(optype op, int extiter, int intiter, int size, int fpos)
#else  /* ANSI_C */
(op, extiter, intiter, size, fpos)
optype op;
int extiter;
int intiter;
int size;
int fpos;
#endif /* ANSI_C */
{
  int ei;
  int ii;
  long id[20];
  double work;
  double start;
  double time;
  double total = 0.0;
  double tsq = 0.0;
  double min = -1.0;
  double max = -1.0;
  double ntime;
  double ntotal = 0.0;
  double ntsq = 0.0;
  double nmin = -1.0;
  double nmax = -1.0;

  int sr;

  for (ei = 0; ei < extiter; ei++)
  {
    time = 0.0;
    for(ii = 0; ii < intiter; ii++)
    {

      gsync();
      if ((sr = lseek(filedes,fpos,SEEK_SET)) != fpos)
      {
        fprintf(stderr,"lseek(%d,%d,SEEK_SET) = %d\n",filedes,fpos,sr);
        perror("meas_op(): lseek()");
        exit(1);
      }
      gsync();

      switch (op)
      {
      case CWR:
        start = dclock();
        cwrite(filedes,buffer,size);
        time += (dclock() - start);
        break;
      case CRD:
        start = dclock();
        cread(filedes,buffer,size);
        time += (dclock() - start);
        break;
      case IWRW:
        start = dclock();
        id[0] = iwrite(filedes,buffer,size);
        iowait(id[0]);
        time += (dclock() - start);
        break;
      case IRDW:
        start = dclock();
        id[0] = iread(filedes,buffer,size);
        iowait(id[0]);
        time += (dclock() - start);
        break;
      case IWRIOW:
      case IWR:
        start  = dclock();
        id[ii] = iwrite(filedes,buffer,size);
        time  += (dclock() - start);
        break;
      case IRDIOW:
      case IRD:
        start = dclock();
        id[ii] = iread(filedes,buffer,size);
        time += (dclock() - start);
        break;
      case WIOW:
        id[ii] = iwrite(filedes,buffer,size);
        break;
      case RIOW:
        id[ii] = iread(filedes,buffer,size);
        break;
      }
    }
    for (ii = 0; ii < intiter; ii++)
    {
      switch (op)
      {
      case IWRIOW:
      case IRDIOW:
      case WIOW:
      case RIOW:
        start = dclock();
        iowait(id[ii]);
        time += (dclock() - start);
        break;
      case IWR:
      case IRD:
        iowait(id[ii]);
        break;
      }
    }
        
    total += time;

    tsq += time*time;
    if ((max == -1.0) || (time > max))
      max = time;
    if ((min == -1.0) || (time < min))
      min = time;

    ntime = time;
    gdsum(&ntime,1,&work);
    ntime = ntime/(double)numnodes();

    ntotal += ntime;
    ntsq += ntime*ntime;
    
    if ((nmax == -1.0) || (ntime > nmax)) nmax = ntime;
    if ((nmin == -1.0) || (ntime < nmin)) nmin = ntime;
    
  }
  
  write_res(op,extiter,intiter,size,min,max,total,tsq,nmin,nmax,ntotal,ntsq);
  
  gsync();

  return;
}

int main
#ifdef ANSI_C
(int argc, char **argv)
#else
(argc, argv)
int argc;
char **argv;
#endif
{
  char  *resultfile = NULL;
  char  *filename   = NULL;
  char  *pfsdhost   = NULL;
  int    extiter    = -1;
  int    intiter    = -1;
  int    minimum    = -1;
  int    maximum    = -1;
  int    step       = -1;
  char  *modes      = NULL;
  char **opargs     = NULL;
  int client_threshold;
  int server_threshold;

  int goerr = 0;
  
  int c;
  int i;
  char *mp;
  optype op;
  int size;
  char **opa;
  int pos;
  int sr;
  
  while (((c = getopt(argc,argv,"r:f:h:d:D:c:sbe:i:n:x:t:m:o")) != -1) &&
         (opargs == NULL))
  {
    switch (c)
    {
    case 'r':
      if (resultfile != NULL)
        fprintf(stderr,"%s: Warning: Overwriting -r flag\n",argv[0]);
      resultfile = optarg;
      break;
    case 'f':
      if (filename != NULL)
        fprintf(stderr,"%s: Warning: Overwriting -f flag\n",argv[0]);
      filename = optarg;
      break;
    case 'h':                   /* Hostname */
      if (pfsdhost != NULL)
        fprintf(stderr,"%s: Warning: Overwriting -h flag\n",argv[0]);
      pfsdhost = optarg;
      break;
    case 'd':
      if (machnumber != -1)
        fprintf(stderr,"%s: Warning: Overwriting -d flag\n",argv[0]);
      if ((machnumber = atoi(optarg)) < 0)
      {
        fprintf(stderr,"%s: Invalid -d option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 'D':
      if (iod_sync != -1)
        fprintf(stderr,"%s: Warning: Overwriting -D flag\n",argv[0]);
      if (strcmp(optarg,"a") == 0) /* iod asynchron */
        iod_sync = IOD_ASYNC;
      else if (strcmp(optarg,"s") == 0) /* iod synchron */
        iod_sync = IOD_SYNC;
      else
      {
        fprintf(stderr,"%s: Invalid -c option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 'c':
      if (clnt_sync != -1)
        fprintf(stderr,"%s: Warning: Overwriting -c flag\n",argv[0]);
      if (strcmp(optarg,"a") == 0) /* Client asynchron */
        clnt_sync = CLNT_ASYNC;
      else if (strcmp(optarg,"s") == 0) /* Client synchron */
        clnt_sync = CLNT_SYNC;
      else
      {
        fprintf(stderr,"%s: Invalid -c option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 's':
      if (o_sync != -1)
        fprintf(stderr,"%s: Warning: Overwriting -s or -b flag\n",argv[0]);
      o_sync = O_SYNC;
      break;
    case 'b':
      if (o_sync != -1)
        fprintf(stderr,"%s: Warning: Overwriting -s or -b flag\n",argv[0]);
      o_sync = 0x0;
      break;
    case 'e':
      if (extiter != -1)
        fprintf(stderr,"%s: Warning: Overwriting -e flag\n",argv[0]);
      if ((extiter = atoi(optarg)) <= 0)
      {
        fprintf(stderr,"%s: Invalid -e option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 'i':
      if (intiter != -1)
        fprintf(stderr,"%s: Warning: Overwriting -i flag\n",argv[0]);
      if ((intiter = atoi(optarg)) <= 0)
      {
        fprintf(stderr,"%s: Invalid -i option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 'n':
      if (minimum != -1)
        fprintf(stderr,"%s: Warning: Overwriting -n flag\n",argv[0]);
      if ((minimum = atoi(optarg)) < 0)
      {
        fprintf(stderr,"%s: Invalid -n option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 'x':
      if (maximum != -1)
        fprintf(stderr,"%s: Warning: Overwriting -x flag\n",argv[0]);
      if ((maximum = atoi(optarg)) < 0)
      {
        fprintf(stderr,"%s: Invalid -x option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 't':
      if (step != -1)
        fprintf(stderr,"%s: Warning: Overwriting -t flag\n",argv[0]);
      if ((step = atoi(optarg)) < 0)
      {
        fprintf(stderr,"%s: Invalid -t option argument %s\n",argv[0],optarg);
        goerr++;
      }
      break;
    case 'm':
      if (modes != NULL)
        fprintf(stderr,"%s: Warning: Overwriting -m flag\n",argv[0]);
      modes = optarg;
      for (mp = modes; *mp != '\0'; mp++)
        if ((*mp != 'g') &&     /* global */
            (*mp != 'l') &&     /* log */
            (*mp != 'r') &&     /* record */
            (*mp != 's') &&     /* sync */
            (*mp != 'u'))       /* unix */
        {
          fprintf(stderr,"%s: Invalid -m option argument %s: %c\n",
                  argv[0],optarg,*mp);
          goerr++;
        }
      break;
    case 'o':
      opargs = &argv[optind];
      for (i=optind; i < argc; i++)
      {
        if (str2op(argv[i]) == NO_OP)
        {
          fprintf(stderr,"%s: Invalid -o option argument %s\n",
                  argv[0],argv[i]);
          goerr++;
        }
      }
      break;
    case '?':
      goerr++;
      break;
    }
  }
  
  if (resultfile == NULL)
    resultfile = DFLTRESULTFILE;

  if (filename == NULL)
    filename = DFLTFILENAME;

  if (pfsdhost == NULL)
    pfsdhost = DFLTPFSDHOST;
  
  if (o_sync == -1)
    o_sync = DFLTO_SYNC;

  if (machnumber == -1)
    machnumber = DFLTMACHNUMBER;

  if (iod_sync == -1)
    iod_sync = DFLTIOD_SYNC;
  
  if (clnt_sync == -1)
    clnt_sync = DFLTCLNT_SYNC;
  
  if (extiter == -1)
    extiter = DFLTEXTITER;
  
  if (intiter == -1)
    intiter = DFLTINTITER;
  
  if (minimum == -1)
    minimum = DFLTMINIMUM;
  
  if (maximum == -1)
    maximum = DFLTMAXIMUM;
  
  if (step == -1)
    step = DFLTSTEP;
  
  if (modes == NULL)
    modes = DFLTMODES;
  
  if (opargs == NULL)
    opargs = dfltopargs;
  
  if (minimum > maximum)
  {
    fprintf(stderr,"%s: Error: Minimum greater than maximum\n",argv[0]);
    goerr++;
  }
  
  if (maximum > BUFFSIZE)
  {
    fprintf(stderr,
            "%s: Error: Maximum greater than internal Buffersize (%d)\n",
            argv[0],BUFFSIZE);
    goerr++;
  }
  
  if ((minimum == 0) && (step == 0))
  {
    fprintf(stderr,"%s: Error: Can't use exponential step starting with 0\n",
            argv[0]);
    goerr++;
  }
  
  if (intiter > MAXINTITER)
  {
    fprintf(stderr,
            "%s: Error: Internal iteration must be less than %d\n",
            argv[0], MAXINTITER);
    goerr++;
  }

  if (goerr)
  {
    fprintf(stderr,"Usage: %s %s\n",argv[0],USAGE);
    exit (1);
  }
  
  init_emexit();

  if (clnt_sync == CLNT_SYNC)
    client_threshold = maximum+1;
  else
    client_threshold = 0;
  
  if (iod_sync == IOD_SYNC)
    server_threshold = maximum+1;
  else
    server_threshold = 0;
  
  _dbgmsg("Initializing buffer");
  for (i=0;i < maximum;i++)
  {
    buffer[i] = mynode()+'0';
  }

  if (_pfslib_init(pfsdhost,numnodes(),mynode(),
                   server_threshold,client_threshold) < 0)
  {
    pfslib_perror("_pfslib_init()");
    exit(1);
  }

  _dbgmsg("Nach _pfslib_init()");

  /* Open result file */
  resultfd = gopen(resultfile,O_CREAT|O_WRONLY|O_APPEND,M_UNIX,0644);
  
  _dbgmsg("Nach gopen()");

  filedes  = gopen(filename,O_CREAT|O_RDWR|o_sync,M_UNIX,0644);
  
  _dbgmsg("Nach gopen()");

  write_head();
  
  for (mp = modes; *mp != '\0'; mp++)
  {
    switch (*mp)
    {
    case 'u':
      mode = M_UNIX;
      break;
    case 'g':
      mode = M_GLOBAL;
      break;
    case 'l':
      mode = M_LOG;
      break;
    case 's':
      mode = M_SYNC;
      break;
    case 'r':
      mode = M_RECORD;
      break;
    }
      
    setiomode(filedes,mode);
  
    fprintf(stderr,"\nIO-mode set to %s\n",mode2str(mode));
    fflush(stderr);

    for (opa = opargs; *opa != NULL; opa++)
    {
      if ((op = str2op(*opa)) == NO_OP)
      {
        fprintf(stderr,"%s: No such operation: %s; skipping\n",*opa);
        fflush(stderr);
        break;
      }
      
      fprintf(stderr,"Running operation %s\n",*opa);
      fflush(stderr);

      for (size = minimum, pos = 0;
           size <= maximum;
           size = (step != 0) ? size+step : 2*size)
      {
        fprintf(stderr,"%d",size);
        fflush(stderr);
        switch (op)
        {
        case CRD:
        case IRDW:
        case IRDIOW:
        case IRD:
        case RIOW:
          if ((sr = lseek(filedes,0,SEEK_END)) < 0)
          {
            _dbgmsg("lseek(%d,0,SEEK_SET) = %d\n",filedes,sr);
            perror("main(): lseek()");
            exit(1);
          }
          switch (mode)
          {
          case M_GLOBAL:
          case M_UNIX:
            if ((sr - pos) < size)
            {
              _dbgmsg("Inputfile to small: %d < %d\n",(sr-pos),size);
              exit(1);
            }
            break;
          case M_LOG:
          case M_RECORD:
          case M_SYNC:
            if ((sr - pos) < (size*numnodes()))
            {
              _dbgmsg("Inputfile to small: %d < %d\n",
                      (sr-pos),(size*numnodes()));
              exit(1);
            }
            break;
          }
          break;
        }
        meas_op(op, extiter, intiter, size, pos);
        switch (mode)
        {
        case M_GLOBAL:
        case M_UNIX:
          pos += size;
          break;
        case M_RECORD:
        case M_SYNC:
        case M_LOG:
          pos += (size*numnodes());
          break;
        }
        fprintf(stderr,"+",size);
        fflush(stderr);
      }
      fprintf(stderr,"\nDone operation %s\n",*opa);
      fflush(stderr);
    }
  }

  fprintf(stderr,"Closing result file\n");
  fflush(stderr);
  close(resultfd);

  fprintf(stderr,"Closing data file\n");
  fflush(stderr);
  close(filedes);
  
  fprintf(stderr,"Finished\n");
  fflush(stderr);

  doemexit=0;

  exit (0);
}

