/* ************************************************************************* *
 *                                                                           *
 *    clnt_ioop_iod.c,v
 *    Client procedures to perform an IO operation via an iod process
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : clnt_ioop_iod.c,v
 *  RCS Date     : 1996/05/30 17:28:11
 *  RCS Revision : 1.6
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "clnt_ioop_iod.c,v 1.6 1996/05/30 17:28:11 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <stdio.h>
#include <sys/types.h>
#include <rpc/rpc.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <memory.h>
#include <errno.h>
#include <unistd.h>

#include "iod.h"
#include "clnt_macros.h"
#include "rpc_defaults.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern CLIENT *_pfslib_clnt;
extern int     _pfslib_svr_thr;

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Create a RPC TCP client handle for a connection to an iod.                *
 * ************************************************************************* */
CLIENT *_iod_clnt_create
#ifdef ANSI_C
(int *sockp, char *ipaddr)
#else  /* ANSI_C */
(sockp, ipaddr)
int *sockp;
char *ipaddr;
#endif /* ANSI_C */
{
  CLIENT *clnt;
  struct sockaddr_in addr;
  int retry;
  int soopt;
  
  memset(&addr, 0, sizeof(struct sockaddr_in));
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = inet_addr(ipaddr);
  addr.sin_port = htons(0);

  retry = -1;
  do
  {
    retry++;
    *sockp = RPC_ANYSOCK;
    clnt = clnttcp_create(&addr, IOD, IOD_VERS,sockp,
                          RPC_TCP_BUFSZ, RPC_TCP_BUFSZ);
  }
  while((retry < CONNECT_RETRIES) &&
        (clnt == NULL) && (errno == ETIMEDOUT));
  if (clnt == NULL)
  {
    clnt_pcreateerror("iod_clnt_create(): clnttcp_create(IOD)");
    return(NULL);
  }

  clnt->cl_auth = _pfslib_clnt->cl_auth;
  
  /* TCP_NODELAY */
  soopt = 1;
#ifdef DEBUG
  fprintf(stdout,"_iod_clnt_create(): Setting TCP_NODELAY\n");
#endif /* DEBUG */
  if (setsockopt(*sockp,IPPROTO_TCP,TCP_NODELAY,(char *)&soopt,sizeof(soopt))<0)
  {
    perror("_iod_clnt_create(): setsockopt(..,IPPROTO_TCP,TCP_NODELAY,..)");
  }

  /* Set REUSEADDR */
  soopt = 1;
#ifdef DEBUG
  fprintf(stdout,"_iod_clnt_create(): Setting SO_REUSEADDR\n");
#endif /* DEBUG */
  if (setsockopt(*sockp,SOL_SOCKET,SO_REUSEADDR,(char *)&soopt,sizeof(soopt))< 0)
  {
    perror("_iod_clnt_create(): setsockopt(..,SOL_SOCKET,SO_REUSEADDR,..)");
  }
  
  return (clnt);
}


/* ************************************************************************* *
 * Perform an IO operation via an iod.                                       *
 * ************************************************************************* */
int _iod_ioop
#ifdef ANSI_C
(int op, char *filename, int oflags, int perm, long offset, 
 long length, char *buffer, CLIENT *clnt, int sock)
#else  /* ANSI_C */
(op, filename, oflags, perm, offset, length, buffer, clnt, sock)
int op;
char *filename;
int oflags;
int perm;
long offset;
long length;
char *buffer;
CLIENT *clnt;
int sock;
#endif /* ANSI_C */
{
        
  iod_ioopargs arg;
  iod_ioopres *result;

#ifdef TCP_TRANSFER
  int tr;
  int rd;
  char *bp;
  int wt;
  char go;
#endif /* TCP_TRANSFER */

  arg.fn     = filename;
  arg.oflags = oflags;
  arg.perm   = perm;
  arg.offset = offset;
  arg.length = length;
  arg.thresh = _pfslib_svr_thr;
  
  switch (op)
  {
  case WRITE_OP:
  case WRITEV_OP:
    arg.data.iod_filedata_len = length;
    arg.data.iod_filedata_val = buffer;
    if ((result = iod_write_1(&arg, clnt)) == NULL)
    {
      clnt_perror(clnt, "_iod_ioop(): iod_write_1()");
      return (-1);
    }
          
    ERR_RESULT(iod_ioopres_u, xdr_iod_ioopres);
          
    if (result->iod_ioopres_u.res.length != length)
    {
      errno = EPFSLNBYTES;
      xdr_free(xdr_iod_ioopres, (char *)result);
      return (-1);
    }
          
    xdr_free(xdr_iod_ioopres, (char *)result);
    break;
  case READ_OP:
  case READV_OP:
    arg.data.iod_filedata_len = 0;
    arg.data.iod_filedata_val = NULL;
    if ((result = iod_read_1(&arg, clnt)) == NULL)
    {
      clnt_perror(clnt, "_iod_ioop(): iod_read_1()");
      return (-1);
    }
          
    ERR_RESULT(iod_ioopres_u, xdr_iod_ioopres);
    
#ifdef TCP_TRANSFER
    while (((wt = write(sock, &go, 1)) < 0) && (errno == EWOULDBLOCK));
    if (wt < 0)
    {
      xdr_free(xdr_iod_ioopres, (char *)result);
      return (-1);
    }
    if (wt == 0)
    {
      xdr_free(xdr_iod_ioopres, (char *)result);
      errno = ESHUTDOWN;
      return (-1);
    }
    tr = result->iod_ioopres_u.res.length;
    bp = buffer;
    while (tr > 0)
    {
      if ((rd = read(sock, (void *)bp, (unsigned int)tr)) < 0)
      {
        if (errno != EWOULDBLOCK) /* If read() would block try again */
        {
          xdr_free(xdr_iod_ioopres, (char *)result);
          return (-1);
        }
      }
      else if (rd == 0)
      {
        xdr_free(xdr_iod_ioopres, (char *)result);
        errno = ESHUTDOWN;
        return (-1);
      }
      tr -= rd;
      bp += rd;
    }
#else  /* TCP_TRANSFER */
    if ((result->iod_ioopres_u.res.length != length) ||
        (result->iod_ioopres_u.res.data.iod_filedata_len != length))
    {
      errno = EPFSLNBYTES;
      xdr_free(xdr_iod_ioopres, (char *)result);
      return (-1);
    }
          
    memcpy(buffer,result->iod_ioopres_u.res.data.iod_filedata_val,(u_int)length);
#endif /* TCP_TRANSFER */

    xdr_free(xdr_iod_ioopres, (char *)result);
    break;
  default:
    errno = ENOSYS;
    return (-1);
  }
  return (0);
}


