/* ************************************************************************* *
 *                                                                           *
 *    clnt_ioop.c,v
 *    Client procedures to perform an IO operation
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : clnt_ioop.c,v
 *  RCS Date     : 1996/10/18 14:39:59
 *  RCS Revision : 1.8
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "clnt_ioop.c,v 1.8 1996/10/18 14:39:59 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/uio.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <signal.h>
#ifdef SUN4
#include <memory.h>
#else
#include <string.h>
#endif /* SUN4 */
#include <stdlib.h>
#include <unistd.h>

#include "pfsd.h"
#include "clnt_defines.h"
#include "clnt_macros.h"
#include "pfslib_errno.h"
#include "rpc_defaults.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern ioIDtabel       _pfslib_ioidtbl[];
extern int             _pfslib_maxfiles;
extern pfslib_fhdl    *_pfslib_ftbl;
extern int             _pfslib_clt_thr;
extern CLIENT         *_pfslib_clnt;
extern struct in_addr  _pfslib_pfsd_inaddr;
extern int             _pfslib_mynumber;

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern void    _pfslib_awaitresult  _PH((CLIENT *clnt));
extern void    _pfslib_resettimeout _PH((CLIENT *clnt));
extern void    _vectordistribute    _PH((char *buffer, struct iovec *iov, 
                                         int iovcnt));
extern long    _vectorlength        _PH((struct iovec *iov, int iovcnt));
extern void    _vectorcollect       _PH((char *buffer, struct iovec *iov,
                                         int iovcnt));
extern int     _getioID             _PH((int fd, int op, unsigned int iolen));
extern CLIENT *_iod_clnt_create     _PH((int *sockp, char *ipaddr));
extern bool_t  _pfslib_islocal      _PH((char *ipaddr));
extern int     _iod_ioop            _PH((int op,
                                         char *filename, int oflags, int perm,
                                         off_t offset, off_t length,
                                         char *buffer,
                                         CLIENT *clnt, int sock));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Copy data form a shard memory segment to the local buffer                 *
 * ************************************************************************* */
int _copy_child_shm
#ifdef ANSI_C
(long ioID)
#else  /* ANSI_C */
(ioID)
long ioID;
#endif /* ANSI_C */
{
  struct shmid_ds shmds;
  char *shmbuf;
  
  switch (_IDTE.op)
  {
  case READ_OP:
  case READV_OP:
    if ((shmbuf = (char *)shmat(_IDTE.shmid,(char *)0,0)) == (char *)-1)
    {
      perror("_copy_child_shm(): shmat()");
      return (-1);
    }
    if (_IDTE.op == READ_OP)
      memcpy(_IDTE.buffer,shmbuf,(u_int)_IDTE.iolen);
    else
      _vectordistribute(shmbuf,_IDTE.iov,_IDTE.iovcnt);
    
    if (shmdt(shmbuf) < 0)
    {
      perror("_copy_shild_shm(): shmdt()");
    }
    
    if (shmctl(_IDTE.shmid,IPC_RMID,&shmds) < 0)
    {
      perror("_copy_child_shm(): shmctl(..,IPC_RMID,..)");
    }    

    _IDTE.shmid = -1;
    break;
  }
  return (0);
}


/* ************************************************************************* *
 * Perform a basic iooperation.                                              *
 *   Get the distribution list from the pfsd.                                *
 *   Iterate over the list and transfer the data directly if the             *
 *   file is local.                                                          *
 *   Otherwise perform the operation via a iod.                              *
 * ************************************************************************* */
long basic_ioop
#ifdef ANSI_C
(long ioID, char *buffer, CLIENT *clnt)
#else  /* ANSI_C */
(ioID, buffer, clnt)
long ioID;
char *buffer;
CLIENT *clnt;
#endif /* ANSI_C */
{
  ioidstatargs arg;
  getdistres *result;
  int i,j;
  
  CLIENT *iodclnt;

  char *bufpt;
  char *bp;
  int oflags;
  int perm;
  int local;
  int fd;
  off_t ofs;
  int len;
  long totlen = 0;
  int sock;

#ifdef FILE_LOCKING
  struct flock fl;
#endif /* FILE_LOCKING */

  S_TIME(basic_ioop);

  /* Get distribution from pfsd */
  arg.fh   = _pfslib_ftbl[_IDTE.fd];
  arg.ioID = _IDTE.svrioID;
  
  if ((arg.fh.mode == M_SYNC) || (arg.fh.mode == M_GLOBAL))
    _pfslib_awaitresult(clnt);
  
  S_TIME(pfsd_getlist);
  if ((result = pfsd_getdist_1(&arg, clnt)) == NULL)
  {
    clnt_perror(clnt, "basic_ioop(): pfsd_getdist_1()");
    return (-1);
  }
  E_TIME(pfsd_getlist);

  _pfslib_resettimeout(clnt);
  
  ERR_RESULT(getdistres_u, xdr_getdistres);
  
  oflags = result->getdistres_u.res.oflags;
  perm   = result->getdistres_u.res.perm;

  /* Read or write data according to list */
  for (i = 0, bufpt = buffer;
       i < result->getdistres_u.res.list.DIST_LIST_len;
       bufpt += result->getdistres_u.res.list.DIST_LIST_val[i].length, i++)
  {
    if (result->getdistres_u.res.list.DIST_LIST_val[i].done) continue;
    if ((local = _pfslib_islocal(result->getdistres_u.res.list.DIST_LIST_val[i].ipaddr)) < 0)
      return (-1);
    if (local)
    {
      S_TIME(open);
      /* Open file */
      if ((fd=open(result->getdistres_u.res.list.DIST_LIST_val[i].filename,
                   oflags,perm)) < 0)
      {
        perror("basic_ioop() open()");
        return(-1);
      }
      E_TIME(open);
    }
    else
    {
      S_TIME(_iod_clnt_create);
      /* Open a connection to the approprite IOD */
      if ((iodclnt = _iod_clnt_create(&sock,result->getdistres_u.res.list.DIST_LIST_val[i].ipaddr)) == NULL)
        return (-1);
      _pfslib_resettimeout(iodclnt);
      E_TIME(_iod_clnt_create);
    }

    /* Transfer data to of from the same host */
    for(j=i, bp = bufpt;
        j < result->getdistres_u.res.list.DIST_LIST_len;
        bp += result->getdistres_u.res.list.DIST_LIST_val[j].length,j++)
    {
      if (result->getdistres_u.res.list.DIST_LIST_val[j].done ||
          (strcmp(result->getdistres_u.res.list.DIST_LIST_val[i].ipaddr,
                  result->getdistres_u.res.list.DIST_LIST_val[j].ipaddr)!=0) ||
          (strcmp(result->getdistres_u.res.list.DIST_LIST_val[i].filename,
                  result->getdistres_u.res.list.DIST_LIST_val[j].filename)!=0))
        continue;
      if (local)
      {
        S_TIME(lseek);
        if ((ofs = lseek(fd,
                         result->getdistres_u.res.list.DIST_LIST_val[j].offset,
                         SEEK_SET)) < 0)
        {
          perror("basic_ioop(): lseek()");
          close(fd);
          return (-1);
        }
        E_TIME(lseek);
        if (ofs != result->getdistres_u.res.list.DIST_LIST_val[j].offset)
        {
          fprintf(stderr,"basic_ioop(): PFSLib: Couldn't seek to specified offset");
          errno = EPFSLSEEK;
          close(fd);
          return (-1);
        }
          
        switch (_IDTE.op)
        {
        case WRITE_OP:
        case WRITEV_OP:
#ifdef DEBUG
          fprintf(stdout,"%d: Writing local %ld bytes to %s on %s\n",
                  _pfslib_mynumber,
                  result->getdistres_u.res.list.DIST_LIST_val[j].length,
                  result->getdistres_u.res.list.DIST_LIST_val[j].filename,
                  result->getdistres_u.res.list.DIST_LIST_val[j].ipaddr);
          fflush(stdout);
#endif /* DEBUG */
#ifdef FILE_LOCKING
          fl.l_type   = F_WRLCK;
          fl.l_whence = SEEK_SET;
          fl.l_start  = result->getdistres_u.res.list.DIST_LIST_val[j].offset;
          fl.l_len    = result->getdistres_u.res.list.DIST_LIST_val[j].length;

          if (fcntl(fd,F_SETLKW,&fl) < 0)
          {
            perror("basic_ioop(): fcntl(..,F_SETLKW,..)");
            close(fd);
            return (-1);
          }
#endif /* FILE_LOCKING */
          
          S_TIME(write);
          
          if ((len = write(fd,bp,(u_int)result->getdistres_u.res.list.DIST_LIST_val[j].length)) < 0)
          {
            perror("basic_ioop(): write()");
            close(fd);
            return (-1);
          }
          E_TIME(write);
          if (len != result->getdistres_u.res.list.DIST_LIST_val[j].length)
          {
            fprintf(stderr,"basic_ioop(): PFSLib: Read or written to few data");
            errno = EPFSLNBYTES;
            close(fd);
            return (-1);
          }
#ifdef FILE_LOCKING
          fl.l_type = F_UNLCK;
          fl.l_whence = SEEK_SET;
          fl.l_start  = result->getdistres_u.res.list.DIST_LIST_val[j].offset;
          fl.l_len    = result->getdistres_u.res.list.DIST_LIST_val[j].length;
          if (fcntl(fd,F_SETLKW,&fl) < 0)
          {
            perror("basic_ioop(): fcntl(..,F_SETLKW,..)");
            close(fd);
            return (-1);
          }
#endif /* FILE_LOCKING */
          break;
        case READ_OP:
        case READV_OP:
#ifdef DEBUG
          fprintf(stdout,"%d: Reading local %ld bytes form %s on %s\n",
                  _pfslib_mynumber,
                  result->getdistres_u.res.list.DIST_LIST_val[j].length,
                  result->getdistres_u.res.list.DIST_LIST_val[j].filename,
                  result->getdistres_u.res.list.DIST_LIST_val[j].ipaddr);
          fflush(stdout);
#endif /* DEBUG */
#ifdef FILE_LOCKING
          fl.l_type   = F_RDLCK;
          fl.l_whence = SEEK_SET;
          fl.l_start  = result->getdistres_u.res.list.DIST_LIST_val[j].offset;
          fl.l_len    = result->getdistres_u.res.list.DIST_LIST_val[j].length;

          if (fcntl(fd,F_SETLKW,&fl) < 0)
          {
            perror("basic_ioop(): fcntl(..,F_SETLKW,..)");
            close(fd);
            return (-1);
          }
#endif /* FILE_LOCKING */
          S_TIME(read);
          if ((len = read(fd,bp,(u_int)result->getdistres_u.res.list.DIST_LIST_val[j].length)) < 0)
          {
            perror("basic_ioop(): read()");
            close(fd);
            return (-1);
          }
          E_TIME(read);
          if (len != result->getdistres_u.res.list.DIST_LIST_val[j].length)
          {
            fprintf(stderr,"basic_ioop(): PFSLib: Read or written to few data");
            errno = EPFSLNBYTES;
            close(fd);
            return (-1);
          }
#ifdef FILE_LOCKING
          fl.l_type = F_UNLCK;
          fl.l_whence = SEEK_SET;
          fl.l_start  = result->getdistres_u.res.list.DIST_LIST_val[j].offset;
          fl.l_len    = result->getdistres_u.res.list.DIST_LIST_val[j].length;
          if (fcntl(fd,F_SETLKW,&fl) < 0)
          {
            perror("basic_ioop(): fcntl(..,F_SETLKW,..)");
            close(fd);
            return (-1);
          }
#endif /* FILE_LOCKING */
          break;
        default:
          errno = ENOSYS;
          close(fd);
          return(-1);
        }
      }
      else                    /* remote */
      {
#ifdef DEBUG
        switch (_IDTE.op)
        {
        case WRITE_OP:
        case WRITEV_OP:
          fprintf(stdout,"%d: Writing remote %ld bytes to %s on %s\n",
                  _pfslib_mynumber,
                  result->getdistres_u.res.list.DIST_LIST_val[j].length,
                  result->getdistres_u.res.list.DIST_LIST_val[j].filename,
                  result->getdistres_u.res.list.DIST_LIST_val[j].ipaddr);
          fflush(stdout);
          break;
        case READ_OP:
        case READV_OP:
          fprintf(stdout,"%d: Reading remote %ld bytes form %s on %s\n",
                  _pfslib_mynumber,
                  result->getdistres_u.res.list.DIST_LIST_val[j].length,
                  result->getdistres_u.res.list.DIST_LIST_val[j].filename,
                  result->getdistres_u.res.list.DIST_LIST_val[j].ipaddr);
          fflush(stdout);
          break;
        }
#endif /* DEBUG */
        S_TIME(_iod_ioop);
        if (_iod_ioop(_IDTE.op,
                      result->getdistres_u.res.list.DIST_LIST_val[j].filename,
                      oflags,perm,
                      result->getdistres_u.res.list.DIST_LIST_val[j].offset,
                      result->getdistres_u.res.list.DIST_LIST_val[j].length,
                      bp,iodclnt,sock) < 0)
          return (-1);
        E_TIME(_iod_ioop);
      }
      totlen += result->getdistres_u.res.list.DIST_LIST_val[j].length;
      result->getdistres_u.res.list.DIST_LIST_val[j].done = TRUE;
    }
    if (local)
    {
      S_TIME(close);
      close(fd);
      E_TIME(close);
    }
    else
    {
      S_TIME(clnt_destroy);
      /* Authentcation from _pfslib_clnt !!! Don't destroy it !!! */
      clnt_destroy(iodclnt);
      close(sock);
      E_TIME(clnt_destroy);
    }
  }
  
  S_TIME(xdr_free);
  xdr_free(xdr_getdistres, (char *)result);
  E_TIME(xdr_free);
  
  E_TIME(basic_ioop);

  return (totlen);
}

/* ************************************************************************* *
 * Perform a normal or vector IO operation                                   *
 * Fork a child process if necessary.                                        *
 * ************************************************************************* */
static int pfslib_ioop
#ifdef ANSI_C
(long ioID)
#else  /* ANSI_C */
(ioID)
long ioID;
#endif /* ANSI_C */
{
  ioopdoneargs arg;

  S_TIME(pfslib_ioop);

  arg.fh     = _pfslib_ftbl[_IDTE.fd];
  arg.ioID   = _IDTE.svrioID;

  if (_IDTE.csync)
  {
    /* Client works synchronously */

    switch (_IDTE.op)
    {
    case READV_OP:
      if ((_IDTE.buffer = malloc((size_t)_IDTE.iolen)) == NULL)
      {
        perror("pfslib_ioop(): malloc()");
        arg.res = -1;
        arg.errno = errno;
        if (pfsd_ioopdone_1(&arg, _pfslib_clnt) == NULL)
          clnt_perror(_pfslib_clnt, "pfslib_ioop(): pfsd_ioopdone_1()");
        return (-1);
      }
      break;
    case WRITEV_OP:
      if ((_IDTE.buffer = malloc((size_t)_IDTE.iolen)) == NULL)
      {
        perror("pfslib_ioop(): malloc()");
        arg.res = -1;
        arg.errno = errno;
        if (pfsd_ioopdone_1(&arg, _pfslib_clnt) == NULL)
          clnt_perror(_pfslib_clnt, "pfslib_ioop(): pfsd_ioopdone_1()");
        return (-1);
      }
      _vectorcollect(_IDTE.buffer,_IDTE.iov,_IDTE.iovcnt);
      break;
    }

    arg.res    = basic_ioop(ioID,_IDTE.buffer,_pfslib_clnt);
    arg.errno  = errno;
    
    S_TIME(pfsd_ioopdone);
    if (pfsd_ioopdone_1(&arg, _pfslib_clnt) == NULL)
    {
      clnt_perror(_pfslib_clnt, "pfslib_ioop(): pfsd_ioopdone_1()");
      return (-1);
    }
    E_TIME(pfsd_ioopdone);

    switch (_IDTE.op)
    {
    case READV_OP:
      if (arg.res == _IDTE.iolen)
        _vectordistribute(_IDTE.buffer,_IDTE.iov,_IDTE.iovcnt);
      /* No break by intention */
    case WRITEV_OP:
      free(_IDTE.buffer);
    }

    _IDTE.state = IDT_DONE;

    E_TIME(pfslib_ioop);
    return (0);
  }

  /* Client asynchron */
  switch (_IDTE.op)
  {
  case READV_OP:
  case READ_OP:
    /* get an shared memory segment for read operations */
    if ((_IDTE.shmid = shmget(IPC_PRIVATE,_IDTE.iolen,
                              0600|IPC_CREAT|IPC_EXCL)) < 0)
      return (-1);
    break;
  }
  
  S_TIME(fork);
  if ((_IDTE.childpid = fork()) < 0 )
  {
    /* LAMBO should free shm segment */
    perror("pfslib_ioop(): fork()");

    arg.res    = -1;
    arg.errno  = errno;
    
    if (pfsd_ioopdone_1(&arg, _pfslib_clnt) == NULL)
    {
      clnt_perror(_pfslib_clnt, "pfslib_ioop(): pfsd_ioopdone_1()");
      return (-1);
    }

    return (-1);
  }
  else if (_IDTE.childpid == 0)
  {
    /* Child process */
    char *buf;
    int ex = 0;
    CLIENT *cld_clnt;

    struct sockaddr_in addr;
    int sock;
    int soopt;
    int retry;
  
#ifdef CHILD_DEBUG
    int dbg = 1;
    while (dbg);
#endif /* CHILD_DEBUG */

/*    fprintf(stderr,"Child process BEGIN\n"); */

    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    memcpy(&addr.sin_addr, &_pfslib_pfsd_inaddr,sizeof(struct in_addr));
    addr.sin_port = htons(0);

    for (retry = 0, cld_clnt = NULL;
         (cld_clnt == NULL) && (retry < CLNT_TCP_CREATE_RETRIES);
         retry++)
    {
      sock = RPC_ANYSOCK;
      cld_clnt = clnttcp_create(&addr, PFSD, PFSD_VERS,
                                &sock, RPC_TCP_BUFSZ, RPC_TCP_BUFSZ);
    }
      
    if (cld_clnt == NULL)
    {
      clnt_pcreateerror("Child process: pfslib_ioop(): clnttcp_create(PFSD)");
      exit(1);
    }

    /* Set REUSEADDR */
    soopt = 1;
    setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,(char *)&soopt,sizeof(soopt));
    /* TCP_NODELAY */
    soopt = 1;
    setsockopt(sock,IPPROTO_TCP,TCP_NODELAY,(char *)&soopt,sizeof(soopt));

    cld_clnt->cl_auth = _pfslib_clnt->cl_auth;
      
    switch (_IDTE.op)
    {
    case READV_OP:
    case READ_OP:
      if ((buf = (char *)shmat(_IDTE.shmid,(char *)0,0))
          == (char *)-1)
      {
        perror("Child process: pfslib_ioop(): shmat()");
        arg.res = -1;
        arg.errno = errno;
        if (pfsd_ioopdone_1(&arg, cld_clnt) == NULL)
          clnt_perror(cld_clnt, "Child process: pfslib_ioop(): pfsd_ioopdone_1()");
        auth_destroy(cld_clnt->cl_auth);
        clnt_destroy(cld_clnt);
        exit (1);
      }
      break;
    case WRITE_OP:
      buf = _IDTE.buffer;
      break;
    case WRITEV_OP:
      if ((buf = malloc((size_t)_IDTE.iolen)) == NULL)
      {
        perror("Child process: pfslib_ioop(): malloc()");
        arg.res = -1;
        arg.errno = errno;
        if (pfsd_ioopdone_1(&arg, cld_clnt) == NULL)
          clnt_perror(cld_clnt, "Child process: pfslib_ioop(): pfsd_ioopdone_1()");
        auth_destroy(cld_clnt->cl_auth);
        clnt_destroy(cld_clnt);
        exit(1);
      }
      _vectorcollect(buf,_IDTE.iov,_IDTE.iovcnt);
      break;
    }
    
    arg.res    = basic_ioop(ioID,buf,cld_clnt);
    arg.errno  = errno;
    
    if (pfsd_ioopdone_1(&arg, cld_clnt) == NULL)
    {
      clnt_perror(cld_clnt, "Child process: pfslib_ioop(): pfsd_ioopdone_1()");
      ex = 1;
    }

    auth_destroy(cld_clnt->cl_auth);
    clnt_destroy(cld_clnt);

    switch (_IDTE.op)
    {
    case READV_OP:
    case READ_OP:
      if (shmdt(buf) < 0)
        perror("Child process: pfslib_ioop(): shmdt()");
      break;
    case WRITEV_OP:
      free(buf);
    }
    
/*    fprintf(stderr,"Child process EXITS\n"); */

    exit (ex);
    /* End of child process */
  }
  E_TIME(fork);

  E_TIME(pfslib_ioop);

  return (0);
}


/* ************************************************************************* *
 * Vector IO operation                                                       *
 *   Get an IO id and perform the operation                                  *
 * ************************************************************************* */
long _ioopv
#ifdef ANSI_C
(int op, int fd, struct iovec *iov, int iovcnt, bool_t ccall)
#else  /* ANSI_C */
(op, fd, iov, iovcnt, ccall)
int op;
int fd;
struct iovec *iov;
int iovcnt;
bool_t ccall;
#endif /* ANSI_C */
{
  long   ioID;
  unsigned int nbytes;
  
  fd -= VFDOFFSET;

  /* Check argument */
  if ((fd < 0) || (fd >= _pfslib_maxfiles) ||
      (_pfslib_ftbl[fd].vfd == -1))
  {
    errno = EPFSLBADF;
    return (-1);
  }

  nbytes = _vectorlength(iov,iovcnt);

  if ((ioID = _getioID(fd,op,nbytes)) < 0)
    return(-1);
  
  /* set values in iodtable */
  _IDTE.state  = IDT_PENDING;
  _IDTE.iov    = iov;
  _IDTE.iovcnt = iovcnt;
  
  if ((_IDTE.iolen < _pfslib_clt_thr) || ccall)
    _IDTE.csync = TRUE;
  else
    _IDTE.csync = FALSE;
  
  pfslib_ioop(ioID);

  return(ioID);
}


/* ************************************************************************* *
 * Normal IO operation                                                       *
 *   Get an IO id and perform the operation.                                 *
 * ************************************************************************* */
long _ioop
#ifdef ANSI_C
(int op, int fd, char *buffer, u_int nbytes, bool_t ccall)
#else  /* ANSI_C */
(op, fd, buffer, nbytes, ccall)
int op;
int fd;
char *buffer;
u_int nbytes;
bool_t ccall;
#endif /* ANSI_C */
{
  long  ioID;

  S_TIME(_ioop);

  fd -= VFDOFFSET;

  /* Check argument */
  if ((fd < 0) || (fd >= _pfslib_maxfiles) ||
      (_pfslib_ftbl[fd].vfd == -1))
  {
    errno = EPFSLBADF;
    return (-1);
  }

  if ((ioID = _getioID(fd,op,nbytes)) < 0)
    return(-1);
  
  /* set values in iodtable */
  _IDTE.state = IDT_PENDING;
  _IDTE.buffer = buffer;
  
  if ((_IDTE.iolen < _pfslib_clt_thr) || ccall)
    _IDTE.csync = TRUE;
  else
    _IDTE.csync = FALSE;
  
  pfslib_ioop(ioID);

  E_TIME(_ioop);

  return(ioID);
}
