/* ************************************************************************* *
 *                                                                           *
 *    clnt_iodone.c,v
 *    Client procedures to check the termination of an IO operation
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : clnt_iodone.c,v
 *  RCS Date     : 1996/05/29 15:04:58
 *  RCS Revision : 1.4
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "clnt_iodone.c,v 1.4 1996/05/29 15:04:58 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>

#include "pfsd.h"
#include "clnt_defines.h"
#include "clnt_macros.h"
#include "pfslib_errno.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

extern CLIENT      *_pfslib_clnt;
extern ioIDtabel    _pfslib_ioidtbl[];
extern pfslib_fhdl *_pfslib_ftbl;
extern long         _pfslib_ioopres;

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern int  _copy_child_shm _PH((long ioID));
extern void pfslib_perror   _PH((char *str));

#undef _PH

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Check whether an asynchronous IO operation completed.                     *
 *   If there is a child process for the operation check for its termination *
 *   Check the termiation at pfsd                                            *
 * ************************************************************************* */
long _iodone
#ifdef ANSI_C
(long ioID)
#else  /* ANSI_C */
(ioID)
long ioID;
#endif /* ANSI_C */
{
  ioopres         *result;
  ioidstatargs    arg;

  pid_t pid;
  int state;

  /* 
   * input correct?
   */

  if ((ioID < 0) || (ioID >= MAXIOID) ||
      ((_IDTE.state != IDT_DONE) && (_IDTE.state != IDT_PENDING)) ||
      ((_IDTE.state == IDT_PENDING) && (_IDTE.csync)))
  {
    errno  = EPFSLBADID;
    return (-1);
  }
  
  if (_IDTE.state != IDT_DONE)
  {

    if (_IDTE.childpid == -1)
    {
      errno = EPFSLNOCHLD;
      return (-1);
    }

    if ((pid = waitpid(_IDTE.childpid,&state,WNOHANG)) == 0)
    {
      return (0);
    }

    if (pid < 0)
    {
      perror("_iodone(): waitpid()");
      return (-1);
    }
 
    if (WIFSIGNALED(state) || WIFEXITED(state))
    {
      _IDTE.childpid  = -1;
      _IDTE.state     = IDT_DONE;
      _IDTE.childexit = state;

      if ((!WIFSIGNALED(state)) && (WEXITSTATUS(state) == 0))
        if (_copy_child_shm(ioID) < 0)
          return (-1);
    }
  }
  
  if (!_IDTE.csync)
  {
    if (WIFSIGNALED(_IDTE.childexit))
    {
      errno = EPFSLCHLDSIG;
      return (-1);
    }
    
    if (WEXITSTATUS(_IDTE.childexit) != 0)
    {
      errno = EPFSLCHLDEXIT;
      return (-1);
    }
  }
  
  arg.fh = _pfslib_ftbl[_IDTE.fd];
  arg.ioID = _IDTE.svrioID;

  if ((result = pfsd_iodone_1(&arg, _pfslib_clnt)) == NULL)
  {
    clnt_perror(_pfslib_clnt, "iodone(): pfsd_iodone_1():");
    return (-1);
  }

  ERR_RESULT(ioopres_u,xdr_ioopres);

  if (result->ioopres_u.res.done)
  {
    /* Reset element in ioidtable */
    _pfslib_ioopres = result->ioopres_u.res.res;
    errno = result->ioopres_u.res.errno;

    _IDTE.state = IDT_UNUSED;
    _IDTE.childpid = -1;
    _IDTE.shmid    = -1;
      
    xdr_free(xdr_ioopres, (char *)result);

    if (_pfslib_ioopres < 0)
      return (-1);
    else
      return (1);
  }

  xdr_free(xdr_ioopres, (char *)result);

  return (0);
}

/* ************************************************************************* *
 * Check whether an asynchronous IO operation completed.                     *
 * Exit on failure                                                           *
 * ************************************************************************* */
long iodone
#ifdef ANSI_C
(int ioID)
#else  /* ANSI_C */
(ioID)
int ioID;
#endif /* ANSI_C */
{
  long res;

  if ((res = _iodone(ioID)) < 0 )
  {
    pfslib_perror("iodone()");
    exit (1);
  }

  return(res);
}
