/* ************************************************************************* *
 *                                                                           *
 *    clnt_init.c,v
 *    Client procedures to initalize a client to use PFSLib
 *                                                                           *
 *    Copyright (C) 1995 A. Bode, S. Lamberts, T. Ludwig, C. R"oder          *
 *                                                                           *
 *    PFSLib (Parallel I/O on workstations)                                  *
 *                                                                           *
 *    PFSLib offers parallel access to files for a parallel application      *
 *    running on a cluster of workstations.                                  *
 *    It is intended but not restricted to be used in message passing        *
 *    applications based on PVM, NXLib, MPI, and other.                      *
 *                                                                           *
 *    PFSLib consists of a LIBRARY, deamon PROGRAMS, and utility PROGRAMS.   *
 *                                                                           *
 *    PFSLib is free software; you can redistribute the LIBRARY and/or       *
 *    modify it under the terms of the GNU Library General Public            *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *    You can redistribute the daemon PROGRAMS and utility PROGRAMS          *
 *    and/or modify them under the terms of the GNU General Public           *
 *    License as published by the Free Software Foundation; either           *
 *    version 2 of the License, or (at your option) any later version.       *
 *                                                                           *
 *    PFSLib is distributed in the hope that it will be useful,              *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *    Library General Public License and GNU General Public License          *
 *    for more details.                                                      *
 *                                                                           *
 *    You should have received a copy of the GNU Library General Public      *
 *    License and the GNU General Public License along with this             *
 *    library; if not, write to the Free                                     *
 *    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     *
 *                                                                           *
 *    Contact to the authors:                                                *
 *                                                                           *
 *    electronic mail: pfslib@informatik.tu-muenchen.de                      *
 *                                                                           *
 *    paper mail:      Prof. Dr. A. Bode                                     *
 *                     Lehrstuhl f"ur Rechnertechnik und                     *
 *                     Rechnerorganisation                                   *
 *                     Institut f"ur Informatik                              *
 *                     Technische Universit"at M"unchen                      *
 *                     80290 M"unchen                                        *
 *                     Germany                                               *
 *                                                                           *
 *    This project was partially funded by a research grant form Intel       *
 *    Corporation.                                                           *
 *                                                                           *
 * ************************************************************************* */


/* ************************************************************************* *
 *                                                                           *
 *  RCS Filename : clnt_init.c,v
 *  RCS Date     : 1996/12/27 15:46:44
 *  RCS Revision : 1.6
 *  RCS Author   : lamberts
 *  RCS State    : V2_0_B
 *                                                                           *
 *  Authors: Stefan Lamberts, Christian R"oder                               *
 *                                                                           *
 * ************************************************************************* */
#ifndef lint
static void *rcs_id = "clnt_init.c,v 1.6 1996/12/27 15:46:44 lamberts V2_0_B";
#endif

/* ************************************************************************* *
 * Include files                                                             *
 * ************************************************************************* */

#include <rpc/rpc.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <signal.h>
#ifdef SUN4
#include <memory.h>
#else
#include <string.h>
#endif /* SUN4 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "pfsd.h"
#include "clnt_defines.h"
#include "clnt_macros.h"
#include "rpc_defaults.h"

/* ************************************************************************* *
 * External declarations                                                     *
 * ************************************************************************* */

#ifdef ANSI_C
#define _PH(a) a
#else  /* ANSI_C */
#define _PH(a) ()
#endif /* ANSI_C */

extern int  pfslib_close          _PH((int filedes));
extern void _pfslib_resettimeout  _PH((CLIENT *clnt));
extern int  _get_my_ipaddr        _PH((void));

#undef _PH

/* ************************************************************************* *
 * Global variables                                                          *
 * ************************************************************************* */

CLIENT *_pfslib_clnt;           /* RPC client */

long           _pfslib_ioopres; /* Result of the last IO operation */
int            _pfslib_numclnts; /* Total number of clients */
int            _pfslib_mynumber; /* Number of this client */
int            _pfslib_maxfiles; /* max number of files */
int            _pfslib_clt_thr; /* Client threshold */
int            _pfslib_svr_thr; /* Server threshold */
struct in_addr _pfslib_pfsd_inaddr; /* INET address of pfsd host */
pid_t          _pfslib_mypid;   /* pid of the client process
                                 * NOT its children */
pfslib_fhdl   *_pfslib_ftbl;    /* File handle table */
ioIDtabel      _pfslib_ioidtbl[MAXIOID]; /* io id table */

/* ************************************************************************* *
 * Procedures                                                                *
 * ************************************************************************* */

/* ************************************************************************* *
 * Initialize the IO identifier table                                        *
 * ************************************************************************* */
static void init_ioidtable
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  int ioID;

  for (ioID=0;ioID<MAXIOID;ioID++)
  {
    _IDTE.state    = IDT_UNUSED;
    _IDTE.childpid = -1;
    _IDTE.shmid    = -1;
  }
}

/* ************************************************************************* *
 * PFSLib procedure on exit                                                  *
 *   Close all open PFSLib files                                             *
 *   Kill all forked child processes                                         *
 *   Remove shared memory segments                                           *
 * ************************************************************************* */
static void pfslib_exit
#ifdef ANSI_C
(void)
#else  /* ANSI_C */
()
#endif /* ANSI_C */
{
  pfslib_fhdl arg;
  void *result;

  int ioID;
  struct shmid_ds shmds;

  if ( _pfslib_mypid == getpid() )
  {
    arg.vfd      = -1;
    arg.how_many = _pfslib_numclnts;
    arg.which    = _pfslib_mynumber;
    arg.mode     = -1;
    arg.pid      = _pfslib_mypid;
    
    /* close all open files at the pfsd */
    if ((result = pfsd_cltexit_1(&arg, _pfslib_clnt)) == NULL)
    {
      clnt_perror(_pfslib_clnt,"pfslib_exit(): pfsd_cltexit_1()");
    }
  
    /* and destroy myself as a client */
    auth_destroy(_pfslib_clnt->cl_auth);
    clnt_destroy(_pfslib_clnt);

    /* kill all children and close shmids*/
    for (ioID=0;ioID<MAXIOID;ioID++)
    {
      if (_IDTE.childpid != -1)
        kill(_IDTE.childpid,SIGKILL);
      if (_IDTE.shmid != -1)
        shmctl(_IDTE.shmid,IPC_RMID,&shmds);
    }
  }
}


/* ************************************************************************* *
 * Create the RPC client handle                                              *
 * ************************************************************************* */
static CLIENT *pfslib_clnt_create
#ifdef ANSI_C
(char *host)
#else  /* ANSI_C */
(host)
char *host;
#endif /* ANSI_C */
{
  CLIENT *clnt;

  struct hostent *hp;
  struct sockaddr_in addr;
  int sock;

  int soopt;
  
  if ((hp = gethostbyname(host)) == (struct hostent *)0)
  {
    perror("pfslib_clnt_create(): gethostbyname()");
    return (NULL);
  }

  memcpy(&_pfslib_pfsd_inaddr, hp->h_addr, sizeof(struct in_addr));
  
  memset(&addr, 0, sizeof(struct sockaddr_in));
  addr.sin_family = AF_INET;
  memcpy(&addr.sin_addr,&_pfslib_pfsd_inaddr,sizeof(struct in_addr));
  addr.sin_port = htons(0);
  sock = RPC_ANYSOCK;

  if ((clnt = clnttcp_create(&addr, PFSD, PFSD_VERS,
                             &sock, RPC_TCP_BUFSZ, RPC_TCP_BUFSZ)) == NULL)
  {
    clnt_pcreateerror("pfslib_clnt_create(): clnttcp_create(PFSD)");
    return (NULL);
  }

  /* TCP_NODELAY */
  soopt = 1;
#ifdef DEBUG
  fprintf(stdout,"pfslib_clnt_create(): Setting TCP_NODELAY\n");
#endif /* DEBUG */
  if (setsockopt(sock,IPPROTO_TCP,TCP_NODELAY,(char *)&soopt,sizeof(soopt))<0)
  {
    perror("pfslib_clnt_create(): setsockopt(..,IPPROTO_TCP,TCP_NODELAY,..)");
  }
  

  /* Set REUSEADDR */
  soopt = 1;
#ifdef DEBUG
  fprintf(stdout,"pfslib_clnt_create(): Setting SO_REUSEADDR\n");
#endif /* DEBUG */
  if (setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,(char *)&soopt,sizeof(soopt))<0)
  {
    perror("pfslib_clnt_create(): setsockopt(..,SOL_SOCKET,SO_REUSEADDR,..)");
  }
  
  auth_destroy(clnt->cl_auth);
  clnt->cl_auth = authunix_create_default();

  _pfslib_resettimeout(clnt);

  return(clnt);
}



/* ************************************************************************* *
 * Initialize client for PFSLib                                              *
 *                                                                           *
 * arguments:                                                                *
 *      host:        hostname of the pfsd                                    *
 *      numofclnts:  number of clients attaching shared files                *
 *      whichnum:    my number in the parallel application                   *
 *      th_ios:      threshold to fork an IO-Server                          *
 *      th_ioc:      in case of ixxx() calls threshold to fork an IO-Client  *
 * ************************************************************************* */
int _pfslib_init
#ifdef ANSI_C
(char *host, int numofclnts, int whichnum, int svr_thr, int clt_thr)
#else  /* ANSI_C */
(host, numofclnts, whichnum, svr_thr, clt_thr)
char *host;
int numofclnts;
int whichnum;
int svr_thr;
int clt_thr;
#endif /* ANSI_C */
{
  int    *numfd;
  int    i;
  char   numfarg;

  if (svr_thr < 0)
    svr_thr = S_DFLTIOLEN;
  
  if (clt_thr < 0)
    clt_thr = C_DFLTIOLEN;
  
  /*
   * set or change the number of clients
   * set or change my client number
   */

  _pfslib_numclnts =  numofclnts;
  _pfslib_mynumber =  whichnum;
  _pfslib_mypid    =  getpid();

  /* get my own ip addresses */
  if (_get_my_ipaddr() < 0)
    return (-1);

  /* set up as PFSLib client */
  if ((_pfslib_clnt = pfslib_clnt_create(host)) == NULL)
    return (-1);
  
  /* homany files are possible? */
  if ((numfd = pfsd_numfiles_1(&numfarg,_pfslib_clnt)) == NULL)
  {
    clnt_perror(_pfslib_clnt, "pfslib_init(): pfsd_numfiles_1()");
    return (-1);
  }

  _pfslib_maxfiles = *numfd;

  /* No xdr_free necessary */

  /* allocate memory for file handle table */

  if ((_pfslib_ftbl=(struct pfslib_fhdl *)malloc(sizeof(struct pfslib_fhdl) *
                                                _pfslib_maxfiles))
      == NULL )
  {
    perror("_pfslib_init(): malloc()");
    return (-1);
  }

  for (i = 0; i < _pfslib_maxfiles; i++ )
  {
    _pfslib_ftbl[i].vfd  = -1;
  }
  
  /*
   * set up exit-handler
   */

#ifdef SUN4
  on_exit(pfslib_exit);
#else
  atexit(pfslib_exit);
#endif

  /* Initialize ioidtabke */
  init_ioidtable();
  
  /*
   * set thresholds for asynchronous IO
   */

  _pfslib_clt_thr = clt_thr;    /* fork() an IO-Client with this threshold */
  _pfslib_svr_thr = svr_thr;    /* fork() an IO-Server with this threshold */

  return (0);
}

/* ************************************************************************* *
 * Initialize PFSLib with default thresholds                                 *
 * ************************************************************************* */
void pfslib_init
#ifdef ANSI_C
(char *host, int numofclnts, int whichnum)
#else  /* ANSI_C */
(host, numofclnts, whichnum)
char *host;
int numofclnts;
int whichnum;
#endif /* ANSI_C */
{
  if (_pfslib_init(host, numofclnts, whichnum, -1, -1) < 0)
  {
    perror("pfslib_init() failed");
    exit(1);
  }
}

