/*
 * urc.c
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: urc.c,v 1.8 1995/08/22 21:27:49 tuna Exp $
 */

/*
 * This file contains functions and structures for the unmapped region
 * cache, which keeps track of region data after a region has been
 * unmapped.
 */

#include "crl_int.h"

#define NumEntries (1<<10)

typedef struct
{
  int     size;
  int     scan;
  Region *entry[NumEntries];
} URC;

URC the_urc;


void urc_insert(Region *rgn)
{
  int      size;
  int      scan;
  Region **slot;
  Region  *victim;
  int      reenable;
  URC     *urc;
  int      oldmask;

  urc = &the_urc;
  size = urc->size;

  if (size < NumEntries)
  {
    /* easy case: urc isn't already full, so we can just insert the
     * new region.
     */
    urc->entry[size] = rgn;
    urc->size = size + 1;

    /* use tx_arg3 to remember urc slot index
     */
    rgn->tx_arg3 = size;
  }
  else
  {
    /* slightly more complicated case: urc is already full, so we need
     * to evict one of the current members to make room for the new
     * region.
     *
     * note that before deleting and freeing the chosen victim, we
     * need to wait until any sends in progress have finished.
     */
    scan      = urc->scan;
    slot      = &(urc->entry[scan]);
    victim    = *slot;
    *slot     = rgn;
    urc->scan = (scan + 1) & (NumEntries-1);

    /* use tx_arg3 to remember urc slot index
     */
    rgn->tx_arg3 = scan;

    rgn_flush(UserFromMeta(victim));

#if defined(CM5) || defined(TCPUNIX)
    if (victim->send_cnt > 0)
    {
#if defined(CM5)
      reenable = CMAML_disable_interrupts();
      while (*((volatile int *) &(victim->send_cnt)) > 0)
	CMAML_poll();
      if (reenable) CMAML_enable_interrupts();
#elif defined(TCPUNIX)
      oldmask = crl_sigblock(sigmask(SIGIO));
      while (*((volatile int *) &(victim->send_cnt)) > 0)
        sigpause(0);
      crl_sigsetmask(oldmask);
#endif
    }
#elif defined(ALEWIFE)
    wait_for_launch();
#endif

    rtable_delete(victim);
    safe_free((void *) victim);
  }
}


void urc_delete(Region *rgn)
{
  int     idx;
  int     size;
  URC    *urc;
  Region *tmp;

  urc  = &the_urc;
  size = urc->size;

  /* figure out where region is in urc (by looking in tx_arg3)
   */
  idx = rgn->tx_arg3;
  sanity(urc->entry[idx] == rgn);

  /* remove region from set
   */
  size -= 1;
  if (idx < size)
  {
    tmp = urc->entry[size];
    urc->entry[idx] = tmp;

    /* update tx_arg3 to remember new urc slot index
     */
    tmp->tx_arg3 = idx;
  }

#if defined(CRL_SANITY)
  urc->entry[size] = NULL;
#endif

  urc->size = size;
}


void crl_flush_urc(void)
{
  int     i;
  int     size;
  int     reenable;
  int     oldmask;
  Region *rgn;
  URC    *urc;

  urc  = &the_urc;
  size = urc->size;

  for (i=0; i<size; i++)
  {
    rgn = urc->entry[i];
    rgn_flush(UserFromMeta(rgn));

#if defined(CM5) | defined(TCPUNIX)
    if (rgn->send_cnt > 0)
    {
#if defined(CM5)
      reenable = CMAML_disable_interrupts();
      while (*((volatile int *) &(rgn->send_cnt)) > 0)
	CMAML_poll();
      if (reenable) CMAML_enable_interrupts();
#elif defined(TCPUNIX)
        oldmask = crl_sigblock(sigmask(SIGIO));
        while (*((volatile int *) &(rgn->send_cnt)) > 0)
          sigpause(0);
        crl_sigsetmask(oldmask);
#endif
    }
#elif defined(ALEWIFE)
    wait_for_launch();
#endif

    rtable_delete(rgn);
    safe_free((void *) rgn);

    urc->entry[i] = NULL;
  }

  urc->size = 0;
  urc->scan = 0;
}
