/*
 * tcp_setup.c
 * june 1995
 * sandeep gupta
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: tcp_setup.c,v 1.7 1995/08/22 21:27:49 tuna Exp $
 */

/* This file contains code to setup the TCP connections, to shutdown
 * the TCP connections, and to improve the cost of checking the status
 * of the SIGIO signal.
 * Much of the networking code is adapted from W. Richard Stevens's book
 * "UNIX Network Programming" (publisher: Prentice Hall, copyright 1990),
 * pp. 267-304.
 */

#include "crl_int.h"
#include <netdb.h>
#include <netinet/tcp.h>

#define TCPBUFFERSIZE 32768
#define MAXHOSTNAMELEN 64

int      node_fd[MaxNodes];    /* node_fd's used to contact each host */
int      shuttingdown = 0;

int      current_sigmask=0;    /* the current signal mask (an optimization) */

int TCP_interrupt_status(void)
{
  /* hack to optimize TCP performance, because system calls for signals are
   * too expensive:
   */
  return(!(current_sigmask & sigmask(SIGIO)));
}

int crl_sigblock(int mask)
{
  /* hack to optimize checking interrupt status */
  int      oldmask;

  oldmask         = sigblock(mask);
  current_sigmask = mask;
  return(oldmask);
}

int crl_sigsetmask(int mask)
{
  /* hack to optimize checking interrupt status */
  int      prevmask;

  prevmask        = sigsetmask(mask);
  current_sigmask = mask;
  return(prevmask);
}


/* Broadcast the full socket address to all other nodes */
static void send_socket_address(int socket)
{
  struct hostent     *info;
  char                myname[MAXHOSTNAMELEN];  /* my own name */
  u_long              addr;                    /* my address */
  int                 retval;
  struct sockaddr_in  socket_addr;             /* my socket address */
  int                 namelen;                 /* required by getsockname() */
  int                 portnum;                 /* port number of socket */

  /* Get information on socket */
  namelen = sizeof(socket_addr);           /* sockets magic */
  retval = getsockname(socket,
                       (struct sockaddr *) &socket_addr,
                       &namelen);
  if (retval <0)
    err_sys("getsockname returned %d\n", retval);
  portnum=socket_addr.sin_port;        /* the socket's port number */

#if defined(CRL_DEBUG)
  printf("port number is %d\n",portnum);
  fflush(stdout);
#endif

  /* Get information on host */
  if(retval = gethostname(myname,MAXHOSTNAMELEN)) {
    fprintf(stderr, "gethostname() returned %d\n",retval);
    exit(1);
  }
  info = gethostbyname(myname);
  if(info == NULL) {
    fprintf(stderr, "unknown host %s\n", myname);
    exit(1);
  }
  addr = (*((u_long *) info->h_addr_list[0]));  /* host's IP address */

  /* Broadcast the information to all the other nodes */
  pvm_initsend(PvmDataRaw);
  retval = pvm_packf("%d%lud",portnum,addr);
  if(retval < 0) {
    printf("pvm_packf returned %d\n",retval);
    exit(1);
  }
  retval = pvm_bcast(group,0);
  if(retval < 0) {
    printf("pvm_bcast returned %d\n",retval);
    exit(1);
  }
}

/* Set the no_delay option for the socket */
static void set_socket_no_delay(int socket)
{
  int no_delay;        /* to set tcp_nodelay */

  no_delay = 1;
  if(setsockopt(socket,
                IPPROTO_TCP,
                TCP_NODELAY,
                (char *) &no_delay,
                sizeof(no_delay))
     < 0)
    err_sys("setsockopt to TCP_NODELAY failed.\n",0);
}

/* Set the buffer size for the socket */
static void set_socket_buffer_size(int socket,int bufsize)
{
  if(setsockopt(socket,
                SOL_SOCKET,
                SO_RCVBUF,
                (char *) &bufsize,
                sizeof(bufsize))
     < 0)
    err_sys("setsockopt to SO_RCVBUF failed.\n",0);

  if(setsockopt(socket,
                SOL_SOCKET,
                SO_SNDBUF,
                (char *) &bufsize,
                sizeof(bufsize))
     < 0)
    err_sys("setsockopt to SO_SNDBUF failed.\n",0);
}

/* Receive the full remote socket address */
static void get_socket_address(int node, int *portnum, u_long *addr)
{
  int     retval;

  retval = pvm_recv(pvm_gettid(group,node),0);
  if(retval < 0) {
    printf("pvm_precv returned %d\n",retval);
    exit(1);
  }
  retval = pvm_unpackf("%d%lud",portnum,addr);
  if(retval < 0) {
    printf("pvm_unpackf returned %d\n",retval);
    exit(1);
  }
}

/* Create and setup server socket */
static int setup_server_socket(void)
{
  int                 server_socketfd;
  struct sockaddr_in  peer;
  int                 retval;

  /* Create a socket */
  server_socketfd = socket(AF_INET,SOCK_STREAM,0);
  if(server_socketfd < 0)
    err_sys("Cannot get server socket\n",0);

  peer.sin_family = AF_INET;
  peer.sin_addr.s_addr = htonl(INADDR_ANY);
  peer.sin_port = htons(0);       /* Allow system to assign port number */

  retval = bind(server_socketfd, (struct sockaddr *) &peer, sizeof(peer));
  if(retval < 0)
    err_sys("bind returned %d\n",retval);

  retval = listen(server_socketfd, 5);
  if(retval < 0)
    err_sys("listen returned %d\n",retval);

  return server_socketfd;
}

/* Sets all the socket options */
static void set_socket_options(int sock)
{
  int retval;

  /*  eliminate tcp delay to coalesce packets */
    set_socket_no_delay(sock);

  /* Increase buffer size - a kludge to stop buffers from filling */
    set_socket_buffer_size(sock,TCPBUFFERSIZE);

    FD_SET(sock,&readfds);
    if(maxfds <= sock)
      maxfds = sock + 1;

    retval = fcntl(sock, F_SETOWN, getpid());
    if(retval < 0)
      err_sys("fcntl SETOWN returned %d\n",retval);

    retval = fcntl(sock, F_SETFL, FASYNC | FNDELAY);
    if(retval < 0)
      err_sys("fcntl SETFL returned %d\n",retval);
}

/* Accept and setup client connections */
static void accept_client_connections(int server_socketfd)
{
  struct sockaddr_in  peer;
  int                 peerlen;    /* Length of peer structure */
  int                 sock;       /* Newly created sockets */
  unsigned            peerhost;   /* Node number of peer */
  int                 i;
  int                 retval;

  peerlen = sizeof(peer);

  for(i=0;i<crl_self_addr;i++) {
#if defined(CRL_DEBUG)
    printf("Waiting for connection number %d\n",i);
    fflush(stdout);
#endif

    /* Wait for a connection */
    sock = accept(server_socketfd, (struct sockaddr *) &peer, &peerlen);
    if(sock < 0)
      err_sys("accept returned %d\n",sock);

#if defined(CRL_DEBUG)
    printf("Successfully accepted connection number %d\n",i);
    fflush(stdout);
#endif

    /* Get peer node number */
    retval = recv(sock, &peerhost, sizeof(unsigned), 0);
    if(retval < 0)
      err_sys("recv returned %d\n",retval);

#if defined(CRL_DEBUG)
    printf("Connected with host %d\n",peerhost);
    fflush(stdout);
#endif

    node_fd[peerhost] = sock;

    set_socket_options(sock);
  }
}

static void make_client_connections(void)
{
  struct sockaddr_in  peer;
  int                 other_portnum;   /* port number of other host */
  u_long              other_addr;      /* network address of other host */
  int                 i;
  int                 retval;

  for(i=crl_self_addr+1;i<crl_num_nodes;i++) {
    /* Create a socket */
    node_fd[i] = socket(AF_INET,SOCK_STREAM,0);
    if(node_fd[i] < 0)
      err_sys("Cannot get socket\n",0);

    get_socket_address(i,&other_portnum,&other_addr);
#if defined(CRL_DEBUG)
    printf("other_portnum is %d, other_addr is %d\n",other_portnum,other_addr);
    fflush(stdout);
#endif

    peer.sin_port = htons(other_portnum);
    peer.sin_addr.s_addr = other_addr;
    peer.sin_family = AF_INET;

    /* Send a connection request to the server */
    retval = connect(node_fd[i], (struct sockaddr *) &peer, sizeof(peer));
    if(retval < 0)
      err_sys("connect returned %d\n",retval);

#if defined(CRL_DEBUG)
    printf("Successfully made connection with server %d\n",i);
    fflush(stdout);
#endif

    /* Send node number */
    retval = send(node_fd[i],&crl_self_addr,sizeof(unsigned),0);
    if(retval < 0)
      err_sys("send returned %d\n",retval);

    set_socket_options(node_fd[i]);
  }
}

void tcp_setup(void)
{
  int     server_socketfd;
  int     i;
  int     other_portnum;   /* port number of other host */
  u_long  other_addr;      /* network address of other host */
  int     retval;


#if defined(CRL_DEBUG)
  printf("starting setup\n");
  fflush(stdout);
#endif

  /* we need to intialize some variables needed for select() calls: */
  FD_ZERO(&readfds);
  FD_ZERO(&writefds);
  FD_ZERO(&exceptfds);
  maxfds = 0;

  /* initialize some CRL variables */
  crl_self_addr = pvm_getinst(group,pvm_mytid());
  crl_num_nodes = pvm_gsize(group);

#if defined(CRL_DEBUG)
  printf("crl_self_addr is %d, crl_num_nodes is %d\n",
          crl_self_addr,crl_num_nodes);
  fflush(stdout);
#endif

  /* Setup some connections as server */
  server_socketfd = setup_server_socket();

  /* Throw away addresses broadcasted by nodes 1 to (crl_self_addr - 1) */
  for(i=1;i<crl_self_addr;i++)
    get_socket_address(i,&other_portnum,&other_addr);

  if(crl_self_addr != 0)
    send_socket_address(server_socketfd);

  accept_client_connections(server_socketfd);

  retval = close(server_socketfd);
  if(retval < 0)
    err_sys("close returned %d\n",retval);

  /* Setup the rest of the connections as client */
  make_client_connections();

  signal(SIGIO, tcp_am_handler);
}

void tcp_shutdown(void)
{
  int i;

  /* Close each socket */
  for(i=0;i<crl_num_nodes;i++)
    if(i != crl_self_addr)
      close(node_fd[i]);
}
