/*
 * tcp_comm.c
 * june 1995
 * sandeep gupta
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: tcp_comm.c,v 1.15 1995/08/22 21:27:49 tuna Exp $
 */

 /*
  * This file contains communication functions specific to the TCP/UNIX
  * version of CRL.  It is included by comm.c.  Among other things,
  * the active message send and receive functions, and the region data
  * send and receive functions are here.
  */

#include "crl_int.h"

#define QueueChunkSize (16)

typedef struct send_msg_element
{
  char                    *msg;
  int                      msg_size;
  struct send_msg_element *next;
} SendMsgElement;

typedef struct AM_Header
{
  handler     h;
  u_long      sending_node_id;
  u_long      lngth;
#if defined(CRL_SANITY)
  u_long      nonce;
#endif
} am_header;

typedef struct CMAM_Msg
{
  handler    h;
  u_long     data1;
  u_long     data2;
  u_long     data3;
  u_long     data4;
} cmam_msg;

ActMsgQueue incoming_queue;
ProtMsg    *prot_msg_free;

static void rcv_data_msg(int, int, char *);

/* Used for asynchronous I/O */
fd_set                 readfds, writefds, exceptfds;
int                    maxfds = 0;
static char           *partial_data[MaxNodes];
static int             curlen[MaxNodes];
static int             datalen[MaxNodes];
static int             partial_data_sender[MaxNodes];
static handler         partial_data_handler[MaxNodes];
static SendMsgElement *SendMsgQueue[MaxNodes];

#if defined(CRL_SANITY)
static int partial_data_msgno[MaxNodes];
static int msgno[MaxNodes];
#endif

/* the following is used to allow non-blocking sending (set to 1 if a
 * send is in progress so that we don't disable interrupts)
 */
static int send_in_progress = 0;


void init_comm(void)
{
  int     i;
  ActMsg *new;

  /* allocate message buffers
   */
  new = (ActMsg *) safe_malloc(sizeof(ActMsg) * QueueChunkSize);
  assert(new != NULL);

  /* link them together 
   */
  for (i=0; i<(QueueChunkSize-1); i++)
    new[i].next = &(new[i+1]);
  new[i].next = &(new[0]);

  /* initialize incoming_queue
   */
  incoming_queue.enabled = 0;
  incoming_queue.head    = new;
  incoming_queue.tail    = new;

  /* protocol message free list is initially empty
   */
  prot_msg_free = NULL;

  /* initialize partial data storage area */
  for(i=0;i<crl_num_nodes;i++)
  {
    partial_data[i] = NULL;
    SendMsgQueue[i] = NULL;
#if defined(CRL_SANITY)
    msgno[i] = (i * crl_num_nodes + crl_num_nodes) << 10;
    /* more unique numbers, less confusion */
#endif
  }
}

void exit_comm(void)
{
  /* Will put some cleanup code here later */
}

void actmsg_enqueue(void *proc, unsigned a0, unsigned a1,
		    unsigned a2, unsigned a3)
{
  int          i;
  ActMsgQueue *q;
  ActMsg      *m;
  ActMsg      *next;

  /* actmsg_enqueue() should only be called
   * when interrupts are disabled
   */
  sanity(TCP_interrupt_status() == FALSE);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsgQueued] += 1;
#endif

  q = &incoming_queue;
  m = q->tail;

  next = m->next;
  if (next == q->head)
  {
    /* allocate more message buffers
     */
    next = (ActMsg *) malloc(sizeof(ActMsg) * QueueChunkSize);
    assert(next != NULL);

    /* link them in 
     */
    m->next = next;
    for (i=0; i<(QueueChunkSize-1); i++)
      next[i].next = &(next[i+1]);
    next[i].next = q->head;
  }
  q->tail = next;

  m->proc = proc;
  m->arg0 = a0;
  m->arg1 = a1;
  m->arg2 = a2;
  m->arg3 = a3;
}


/* drain the incoming message queue; conditionally
 * disable message queueing when done
 */
void actmsg_poll(unsigned new_enable)
{
  int         oldmask;
  int         reenable;
  int         enabled;
  ActMsgQueue *q;
  ActMsg      *m;
  void       (*f)(unsigned, unsigned, unsigned, unsigned);

  reenable = 0;
  enabled  = TCP_interrupt_status();

  q = &incoming_queue;

  while (1)
  {
    /* walk through the message queue invoking handlers
     */
    m = q->head;

    if (m != q->tail)
    {
      if (enabled)
      {
        oldmask = crl_sigblock(sigmask(SIGIO));
        reenable = 1;
        enabled  = 0;
      }

      do
      {
        sanity(q->enabled);
        f = m->proc;
        f(m->arg0, m->arg1, m->arg2, m->arg3);
        m = m->next;
        q->head = m;
      } while (m != q->tail);
    }

    /* turn message queueing off. if (m == q->tail) is still true, no
     * messages arrived and queued themselves sneakily between the
     * time we exited the inner loop and when message queuing got
     * turned off, so we can safely exit the outer loop (note clever
     * [hah!] use of volatile to make sure the compiled code does the
     * right thing)
     */
    *((volatile unsigned *) &(q->enabled)) = new_enable;
    if (m == *((ActMsg * volatile *) &(q->tail)))
      break;

    /* no such luck, so lather, rinse, repeat
     */
    q->enabled = 1;
  }

  if (reenable)
    crl_sigsetmask(oldmask);
}

/* like actmsg_poll(), but assumes interrupts are already disabled
 */
void actmsg_poll_no_intr(unsigned new_enable)
{
  ActMsgQueue *q;
  ActMsg      *m;
  void       (*f)(unsigned, unsigned, unsigned, unsigned);

  sanity(TCP_interrupt_status() == FALSE);

  q = &incoming_queue;

  /* walk through the message queue invoking handlers
   */
  m = q->head;

  while (m != q->tail)
  {
    sanity(q->enabled);
    f = m->proc;
    f(m->arg0, m->arg1, m->arg2, m->arg3);
    m = m->next;
    q->head = m;
  }

  q->enabled = new_enable;
  sanity(q->head == q->tail);
}

void retry_blocked_msgs(HomeRegion *rgn)
{
  ProtMsgQueue *q;
  ProtMsg      *m;
  ProtMsg      *m2;

  /* since retry_blocked_msgs() calls protocol message handlers, make sure
   * the incoming message queue is enabled
   */
  sanity(TCP_interrupt_status() == FALSE);
  sanity(incoming_queue.enabled);

  q = &(rgn->blocked_msgs);
  sanity(ProtMsgQueuePeek(q) != NULL);

  while (1)
  {
    m = ProtMsgQueuePeek(q);

    /* stop if there aren't any more blocked messages
     */
    if (m == NULL) break;

    /* stop if the first message is still blocked
     */
    if (MsgEvent(m->type, (Region *) rgn, m->src, m->vers) == MessageBlocked)
      break;

    /* message was successfully processed, so remove it from the queue,
     * double check that it's the same message we got when we peeked,
     * and free it
     */
    ProtMsgQueueGet(m2, q);
    sanity(m == m2);
    ProtMsgFree(m2);
  }
}


static void rgn_msg(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
  int      rtn;
  ProtMsg *msg;

  /* since rgn_msg() calls protocol message handlers, make sure the
   * incoming message queue is enabled
   */
  sanity(TCP_interrupt_status() == FALSE);
  sanity(incoming_queue.enabled);

  rtn = MsgEvent(type, rgn, src, vers);

  if (rtn == MessageBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    ProtMsgAlloc(msg);
    msg->type = type;
    msg->src  = src;
    msg->vers = vers;
    ProtMsgQueuePut(&((HomeRegion *) rgn)->blocked_msgs, msg);
  }
  else if (rtn == RetryBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    if (ProtMsgQueuePeek(&(((HomeRegion *) rgn)->blocked_msgs)) != NULL)
      retry_blocked_msgs((HomeRegion *) rgn);
  }
}


static void rgn_inv(unsigned type, rid_t rgn_id, unsigned src, unsigned vers)
{
  Region *rgn;

  rgn = rtable_lookup(rgn_id);

  /* if rtable_lookup() returns NULL, the region in question has been
   * flushed and removed from the rtable; this invalidate message must
   * have been sent from the home node before the flush message
   * arrived there, so it can be safely ignored.
   */
  if (rgn != NULL)
    rgn_msg(type, rgn, src, vers);
}


void rgn_msg_stub(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    actmsg_enqueue(rgn_msg, type, (unsigned) rgn, src, vers);
  }
  else
  {
    incoming_queue.enabled = 1;
    rgn_msg(type, rgn, src, vers);
    actmsg_poll_no_intr(0);
  }
}


void rgn_inv_stub(unsigned type, rid_t rgn_id, unsigned src, unsigned vers)
{
#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    actmsg_enqueue(rgn_inv, type, rgn_id, src, vers);
  }
  else
  {
    incoming_queue.enabled = 1;
    rgn_inv(type, rgn_id, src, vers);
    actmsg_poll_no_intr(0);
  }
}


/* Called after all data has been sent successfully */
static void send_data_msg_done(Region *rgn)
{
  rgn->send_cnt -= 1;
}

void send_data_msg(unsigned node, unsigned type, Region *dst,
		   unsigned vers, Region *src)
{
  void    *src_buf;
  char    *sendbuf;

  sanity(src->send_cnt > 0);

  src_buf = UserFromMeta(src);

  /* We can probably improve this later to save a copying operation */

  /* Message contains a type, remote region address, own node address,
     region version, region size, region data */
  sendbuf = (char *)safe_malloc(sizeof(int) * 5 + sizeof(char) * src->size);
  *((unsigned *)(&sendbuf[0])) = type;
  *((Region **)(&sendbuf[4])) = dst;
  *((unsigned *)(&sendbuf[8])) = crl_self_addr;
  *((unsigned *)(&sendbuf[12])) = vers;
  *((unsigned *)(&sendbuf[16])) = src->size;
  memcpy(sendbuf+20,src_buf,src->size);

  tcp_am_send(node, rcv_data_msg, (sizeof(int) * 5 + sizeof(char) * src->size),
	      sendbuf);

  send_data_msg_done(src);

  safe_free(sendbuf);
}


static void rcv_data_msg(int node, int buflen, char *buffer)
{
  int      rtn;
  ProtMsg *msg;
  unsigned type;
  Region *rgn;
  unsigned src;
  unsigned vers;
  unsigned size;
  char *data;

  type = *((unsigned *)(&buffer[0]));
  rgn = *((Region **)(&buffer[4]));
  src = *((unsigned *)(&buffer[8]));
  vers = *((unsigned *)(&buffer[12]));
  size = *((unsigned *)(&buffer[16]));
  data = buffer + 20;

  memcpy(UserFromMeta(rgn), data, size);

  rgn_msg_stub(type, rgn, src, vers);
}


static void rgn_info_ack(unsigned vers, unsigned size, void *addr, RgnInfo *rtn)
{
  rtn->vers      = vers - 1;
  rtn->size      = size;
  rtn->home_addr = addr;
  rtn->valid     = 1;
}


static void rgn_info_req(unsigned src, rid_t rgn_id, RgnInfo *rtn)
{
  Region *rgn;

  rgn = rtable_lookup(rgn_id);
  sanity(rgn != NULL);

  sanity(src < crl_num_nodes);
  tcp_am_send_cm(src, rgn_info_ack, rgn->vers, rgn->size, (int) rgn, (int) rtn); 
#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRgnInfoAck] += 1;
#endif
}


unsigned get_region_info(rid_t rgn_id, RgnInfo *info)
{
  int      oldmask;
  unsigned home;
  unsigned self;

  /* figure out where home node is
   */
  home = RegionIdNode(rgn_id);
  sanity(home < crl_num_nodes);

  /* this procedure should only be called with region ids for remote
   * regions, so crash and burn if the home is the local node
   */
  self = crl_self_addr;
  sanity(home != self);

  /* query home node for info
   */
  info->valid = 0;
  oldmask = crl_sigblock(sigmask(SIGIO));
  /*  note: the final argument to this function is there to pass this
   *  message through the compiler; it has no meaning or other purpose.
   */
  tcp_am_send_cm(home, rgn_info_req, self, rgn_id, (int) info, 0);


#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRgnInfoReq] += 1;
#endif

  while (*((volatile unsigned *) &(info->valid)) == 0)
    sigpause(0);

  crl_sigsetmask(oldmask);

  return home;
}


/* generalized active message interface
 */
static void send_block(int sockfd, char *msg, int msg_size)
{
  /* an abstraction for sending a large message: */
  int        bytes_sent;
  char      *tmp;
  
  bytes_sent = send(sockfd, msg, msg_size, 0);

  while (bytes_sent != msg_size) {
    if(bytes_sent < 0) {
      if(errno != EWOULDBLOCK)
	err_sys("in send_block: send call failed.\n",0);
      else
	tcp_am_handler();
    } else {
      msg_size -= bytes_sent;
      msg += bytes_sent;
    } /* if-else */
    bytes_sent = send(sockfd, msg, msg_size, 0);
  } /* while */
}

void tcp_am_send(int destination, handler h, int lngth, char *inpt)
{
  int          i;
  char        *msg;
  char        *tmp;
  int          sockfd;
  int          msg_size;
  int          bytes_sent;
  int          oldmask;
  int          unset = 0;  /* 1 if supposed to unset send_in_progress on exit */
  am_header    header;
  SendMsgElement *sme;

  if(send_in_progress == 0) {
    oldmask = crl_sigblock(sigmask(SIGIO));
    send_in_progress = 1;
    unset = 1;
  }
  else {
#if defined(CRL_DEBUG)
    fprintf(stderr, "Send in progress,,,,,,,,,,\n");
#endif
    sanity(TCP_interrupt_status() == FALSE);
  }

#if defined(CRL_DEBUG)
#if defined(CRL_SANITY)
  fprintf(stderr, "Sending message number %d to %d of length %d\n",
	  msgno[destination], destination, lngth);
#else
  fprintf(stderr, "Sending message to %d of length %d\n",
	  destination, lngth);
#endif
#endif

  /* Needs to be safe? */
  msg = (char *)malloc(sizeof(am_header)+lngth);
  
#if defined(CRL_SANITY)
  /* putting in a unique message number (nonce?) */
  header.nonce = htonl(msgno[destination]++);
#endif
  header.h               = h;
  header.sending_node_id = htonl(crl_self_addr);
  header.lngth           = htonl(lngth);

  /* marshalling the header and data into the message: */
  memcpy(msg, (char *) &header, sizeof(am_header));  
  memcpy(msg + sizeof(am_header), inpt, lngth);
  msg_size = sizeof(am_header) + lngth;
  
  /* send the message: */

  if(SendMsgQueue[destination] == NULL) { /* If currently not sending to dest */
  /* Needs to be safe? */
    sme = (SendMsgElement *)malloc(sizeof(SendMsgElement));
    sme->msg = msg;
    sme->msg_size = msg_size;
    sme->next = NULL;
    SendMsgQueue[destination] = sme;
  
    /* figure out what fd corresponds to node_id destination */
    sockfd = node_fd[destination];

    send_block(sockfd, msg, msg_size);
    
    free(msg);
#if defined(CRL_DEBUG)
    fprintf(stderr, "Sent message to %d\n",destination);
#endif

    SendMsgQueue[destination] = sme->next;
    free(sme);
    sme = SendMsgQueue[destination];
    while(sme != NULL) {
      msg = sme->msg;
      msg_size = sme->msg_size;

      send_block(sockfd, msg, msg_size);

      free(msg);
      SendMsgQueue[destination] = sme->next;
      free(sme);

#if defined(CRL_DEBUG)
      fprintf(stderr, "Sent queued message to %d\n", destination);
#endif

      sme = SendMsgQueue[destination];
    }
  }
  else { /* already sending a message to destination, so queue msg */
    sme = (SendMsgElement *)malloc(sizeof(SendMsgElement));

    sme->msg = msg;
    sme->msg_size = msg_size;
    /*  Put message right after first message. (Should probably put at end
     *  of queue, but good enough for now)
     */
    sme->next = SendMsgQueue[destination]->next;
    SendMsgQueue[destination]->next = sme;

#if defined(CRL_DEBUG)
    fprintf(stderr, "Queued message to %d.\n", destination);
#endif
  }

  if (unset) {
    send_in_progress = 0;
    crl_sigsetmask(oldmask);
  }
}

void tcp_am_handler(void)
{
  /*  on a SIGIO signal, this procedure processes the active message and
   *  calls the appropriate handler
   */

  int              i, fd;
  fd_set           rfds;
  struct timeval   timeout;
  am_header        header;
  char             junk_buffer[4];
  char            *data_buffer;
  char            *data_buffer_temp;
  int              oldmask;
  handler          h;
  int              total_lngth;
  int              retval;
  long             num_to_read = 0;
  int              lngth;
  int              lngth_tmp;
  int              sending_node;
  int              bytes_received;
  int              total_bytes_received;
#if defined(CRL_SANITY)
  int              rcv_msgno;
#endif

  while(1) { /* continue processing messages until there are no more waiting */
    rfds = readfds;

    timeout.tv_sec  = 0;
    timeout.tv_usec = 0;

    num_to_read = select(maxfds, &rfds, (fd_set *) 0, (fd_set *) 0, &timeout);
    if (num_to_read == 0) {
      /* one of the other nodes must have exited */
      return;
    } else if (num_to_read < 0) {
  /* err_sys(">> num_to_read = %d, maxfds = %d <<\n", num_to_read, maxfds); */
      return;
    }

    for (i=0; (i < crl_num_nodes); i++) {
      if (i != crl_self_addr) {
        fd = node_fd[i];
        if (FD_ISSET(fd, &rfds)) {
          /*  there is a message waiting on file descriptor fd: */
          ioctl(fd, FIONREAD, &num_to_read);
          if (num_to_read <= 0) {
            /* one of the other nodes must have exited; 
             * We need to clear the receive bit by reading 0 characters */
            retval = recv(fd, junk_buffer, 1, MSG_PEEK);
            if ((retval < 0)&&(errno != EWOULDBLOCK))
              err_sys("in tcp_am_handler: failed to clear read bit.\n",0);
            else if (retval == 0) {
            /* Assume that this is EOF since nothing left to read and clear
             *  permanently */
              FD_CLR(fd, &readfds);
#if defined(CRL_SANITY)
              printf("Just cleared fd %d\nExiting\n",fd);
              fflush(stdout);
              if(!shuttingdown)
                exit(1);
#endif
            }
            break;
          }
          if (partial_data[i] == NULL) {
            /* We assume that the full header is sent in one shot */
            if (recv(fd, (char *) &header, sizeof(am_header),0) 
		!= sizeof(am_header))
              err_sys("in tcp_am_handler: failed to get header information from active message.\n",0);

	    h            = header.h;
	    sending_node = ntohl(header.sending_node_id);
	    lngth        = ntohl(header.lngth);	    
	    
#if defined(CRL_SANITY)
            if(sending_node != i) {
#if defined(CRL_DEBUG)
              fprintf(stderr, "%d != %d\n", sending_node, i);
#endif
              fflush(stdout);
            }

            rcv_msgno = ntohl(header.nonce);  /* retrieve message number: */
#if defined(CRL_DEBUG)
            fprintf(stderr, "New message number %d from %d of length %d\n",
		    rcv_msgno, sending_node, lngth);
#endif
#else
#if defined(CRL_DEBUG)
            fprintf(stderr, "Message from %d\n",sending_node);
#endif
#endif
         
            if(lngth == 0) {
              /* call active message handler: */
              (h)(sending_node,lngth,NULL);
              continue;
            }

            /* retrieve the rest of the message: */
            data_buffer = (char *) malloc(lngth);
            bytes_received = recv(fd, data_buffer, lngth, 0);

            /* assume that at least part of the message will arrive */ 
            if( bytes_received < 0)
              err_sys("in tcp_am_handler: failed to receive message.\n",0);

            total_bytes_received = bytes_received;
            data_buffer_temp = data_buffer + bytes_received;
            lngth_tmp = lngth - bytes_received;
            while (total_bytes_received != lngth) {
              bytes_received = recv(fd, data_buffer_temp, lngth_tmp, 0);
              if (bytes_received < 0)
                if(errno != EWOULDBLOCK)
                  err_sys("in tcp_am_handler: failed to receive message.\n",0);
                else
                  break;
              total_bytes_received += bytes_received;
              data_buffer_temp     += bytes_received;
              lngth_tmp            -= bytes_received;
            }

            /* If whole message was not read, then store partial info,
               and continue */
            if(total_bytes_received != lngth) {
              partial_data[i]         = data_buffer;
              partial_data_handler[i] = h;
              curlen[i]               = total_bytes_received;
              datalen[i]              = lngth;
              partial_data_sender[i]  = sending_node;
#if defined(CRL_SANITY)
              partial_data_msgno[i]   = rcv_msgno;
              DEBUG("Continuing later\n");
#endif
              continue;
            }

            /* whole message received, so call active message handler: */
            (h)(sending_node,lngth,data_buffer);

            free(data_buffer);
	  }
          else {  /* Part of the data has already been read */
            sanity(partial_data_sender[i] == i);
            lngth                = datalen[i];
            total_bytes_received = curlen[i];
            data_buffer_temp     = partial_data[i] + total_bytes_received;
            lngth_tmp            = lngth - total_bytes_received;
#if defined(CRL_SANITY) && defined(CRL_DEBUG)
            fprintf(stderr,
		    "Continuing message number %d from %d of length %d\n",
		    partial_data_msgno[i], i, lngth);
#endif
            while (total_bytes_received != lngth) {
              bytes_received = recv(fd, data_buffer_temp, lngth_tmp, 0);
              if (bytes_received < 0)
                if(errno != EWOULDBLOCK)
                  err_sys("in tcp_am_handler: failed to receive message.\n",0);
                else
                  break;
              total_bytes_received += bytes_received;
              data_buffer_temp     += bytes_received;
              lngth_tmp            -= bytes_received;
            }

            /* If whole message was not read, then store partial info,
               and continue */
            if(total_bytes_received != lngth) {
              curlen[i] = total_bytes_received;
              continue;
            }

            /* whole message received, so call active message handler: */
#if defined(CRL_SANITY) && defined(CRL_DEBUG)
	    fprintf(stderr,
		    ">> Finished receiving message number %d from node %d\n",
		    partial_data_msgno[i], partial_data_sender[i]);
#endif
	    data_buffer = partial_data[i];
	    partial_data[i]= NULL;
            (partial_data_handler[i])(partial_data_sender[i],lngth,data_buffer);

            free(data_buffer);
          }
        } /* if */
      }   /* if */
    }     /* for */
  } /* while */  
}
  
static void cm_am_handler(int sender, int lngth, char *msg) {
  /* processes cmaml style active messages: */

  handler     h;
  int         data1, data2, data3, data4;

  /* retrieve handler address: */
  h     = (*(cmam_msg *)msg).h;
  
  /* retrieve handler arguments: */
  data1 = (int) ntohl((*(cmam_msg *)msg).data1);
  data2 = (int) ntohl((*(cmam_msg *)msg).data2);
  data3 = (int) ntohl((*(cmam_msg *)msg).data3);
  data4 = (int) ntohl((*(cmam_msg *)msg).data4);

  /* call active message handler procedure: */
  (h)(data1,data2,data3,data4);
}


void tcp_am_send_cm(int destination, handler h, int data1,
		    int data2, int data3, int data4)
{
  /* Sends a CMMD style active message to the node specified by destination */
  
  cmam_msg   msg;
  int        sockfd;

  /* marshalling the information above into an active message: */
  msg.h      = h;
  msg.data1  = htonl(data1);
  msg.data2  = htonl(data2);
  msg.data3  = htonl(data3);
  msg.data4  = htonl(data4);

  /* send the generalized active message: */
  tcp_am_send(destination, cm_am_handler, sizeof(cmam_msg), (char *) &msg);
}
