/*
 * stats.c
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: stats.c,v 1.6 1995/08/22 21:27:49 tuna Exp $
 */

 /*
  * This file contains functions and structures for keeping track of
  * CRL statistics (e.g. number of times that rgn_map() is called).
  */

#include "crl_int.h"

#if (ALEWIFE_PROFILING != ProfileNone)
#include <math.h>
extern unsigned _msg_intr[2];
#endif

#if defined(STATISTICS)

static char *crl_stats_label[NumStatCntrs] =
{
  "rgn_create",
  " (destroy)",
  "rgn_map",
  " (remote)",
  " (miss)",
  " (unmap)",
  "rgn_start_read",
  " (remote)",
  " (miss)",
  " (block)",
  " (end)",
  "rgn_start_write",
  " (remote)",
  " (miss)",
  " (block)",
  " (end)",
  "rgn_flush",
  "rgn_info_req",
  " (ack)",
  "protocol msg",
  " (queued)",
  " shared req",
  " shared ack data",
  " exclusive req",
  " exclusive ack data",
  " modify req",
  " modify ack",
  " modify ack data",
  " R invalidate",
  " W invalidate",
  " invalidate ack",
  " invalidate ack data",
  " release",
  " release data",
  " flush",
  " flush data",
  "rtable_lookup",
  " (iters)",
};

unsigned crl_stat_cntrs[NumStatCntrs];


void crl_stats_reset(void)
{
  int i;

  for (i=0; i<NumStatCntrs; i++)
    crl_stat_cntrs[i] = 0;

  rgn_barrier();
}


void crl_stats_print(void)
{
  int    i;
  double val;
  double min;
  double max;
  double avg;

  for (i=0; i<NumStatCntrs; i++)
  {
    val = crl_stat_cntrs[i];
    min = rgn_reduce_dmin(val);
    max = rgn_reduce_dmax(val);
    avg = rgn_reduce_dadd(val) / crl_num_nodes;

    if (crl_self_addr == 0)
      printf(" %-24s %12.1f | %10.0f %10.0f\n",
	     crl_stats_label[i], avg, min, max);
  }

  rgn_barrier();
}


#else

void crl_stats_reset(void)
{
}

void crl_stats_print(void)
{
}

#endif


void crl_prof_reset(void)
{
#if (ALEWIFE_PROFILING != ProfileNone)
  set_statcount(0, 0);
  _msg_intr[0] = 0;
  _msg_intr[1] = 0;
#endif
}

void crl_prof_print(void)
{
#if (ALEWIFE_PROFILING != ProfileNone)
  int    i;
  double t1;
  double t2;
  double avg;
  double std;

  t1 = get_statcount(0);
  t2 = t1 * t1;

  t1 = rgn_reduce_dadd(t1);
  t2 = rgn_reduce_dadd(t2);

  avg = t1 / crl_num_nodes;
  std = sqrt((t2 / crl_num_nodes) - (avg * avg));

  if (crl_self_addr == 0)
    printf(" prof: %.3f +- %.3f  %u %.6f %.6f\n",
	   avg, std, crl_num_nodes, t1, t2);

  for (i=0; i<2; i++)
  {
    t1 = _msg_intr[i];
    t2 = t1 * t1;

    t1 = rgn_reduce_dadd(t1);
    t2 = rgn_reduce_dadd(t2);

    avg = t1 / crl_num_nodes;
    std = sqrt((t2 / crl_num_nodes) - (avg * avg));

    if (crl_self_addr == 0)
      printf(" int%d: %.3f +- %.3f  %u %.6f %.6f\n",
	     i, avg, std, crl_num_nodes, t1, t2);
  }
#endif
}
