/*
 * rtable.c
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: rtable.c,v 1.8 1995/08/22 21:27:49 tuna Exp $
 */

 /*
  * This file has functions and structures for the region table.  The
  * region table contains information about regions mapped on this
  * node.
  */

#include "crl_int.h"

#define RtableBits (13)
#define RtableSize (1<<RtableBits)
#define RtableMask (RtableSize-1)

static Region **rtable;
static unsigned rgn_seqno;

#if defined(CRL_DEBUG)
HomeRegion   **home_rtable;
RemoteRegion **remote_rtable;
#endif


void init_rtable(void)
{
  rtable = (Region **) safe_malloc(sizeof(Region *) * RtableSize);
  assert(rtable != NULL);
  bzero((char *) rtable, sizeof(Region *) * RtableSize);

  rgn_seqno = 1;

#if defined(CRL_DEBUG)
  home_rtable   = (HomeRegion **) rtable;
  remote_rtable = (RemoteRegion **) rtable;
#endif
}

void exit_rtable(void)
{
  /* Will put some cleanup code here later */
}

rid_t allocate_rgn_id(void)
{
  unsigned rslt;

  rslt = MakeRegionId(crl_self_addr, rgn_seqno);
  rgn_seqno += 1;

  /* by convention, 0 is not a valid region identifier
   */
  sanity(rslt != 0);

  /* if this ever triggers, then we need to think about implementing
   * (global) region id garbage collection ...
   */
  assert(rgn_seqno < MaxSeqno);

  return rslt;
}


Region *rtable_lookup(rid_t rgn_id)
{
  Region *rslt;
  Region *scan;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrRtableLookup] += 1;
#endif

  /* search appropriate bucket for matching rgn_id
   */
  rslt = NULL;
  scan = rtable[HashRegionId(rgn_id) & RtableMask];
  while (scan != NULL)
  {
#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrRtableLookupLoop] += 1;
#endif

    if (scan->rgn_id == rgn_id)
    {
      rslt = scan;
      break;
    }

    scan = scan->rtable_link;
  }

  return rslt;
}


void rtable_insert(Region *rgn)
{
  unsigned idx;

  idx = HashRegionId(rgn->rgn_id) & RtableMask;

  rgn->rtable_link = rtable[idx];
  rtable[idx] = rgn;
}


void rtable_delete(Region *rgn)
{
  unsigned idx;
  Region  *scan;
  Region  *next;

  idx  = HashRegionId(rgn->rgn_id) & RtableMask;
  scan = rtable[idx];
  sanity(scan != NULL);

  if (scan == rgn)
  {
    /* region to be deleted is at head of list
     */
    rtable[idx] = scan->rtable_link;
  }
  else
  {
    /* region to be deleted is not at head of list
     */
    while (1)
    {
      next = scan->rtable_link;
      sanity(next != NULL);

      if (next == rgn)
	break;

      scan = next;
    }

    /* next points to region to be deleted; scan points to previous
     * region in list; delete next by setting scan->link to the value
     * of next->link.
     */
    scan->rtable_link = next->rtable_link;
  }

  rgn->rtable_link = NULL;
}
