/*
 * ptr.c
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: ptr.c,v 1.6 1995/08/22 21:27:49 tuna Exp $
 */

 /*
  * This file contains functions and structures which keep track of
  * which nodes have a copy of a particular region.
  */

#include "crl_int.h"

Pointer *ptr_free_list = NULL;


Pointer *ptr_insert(HomeRegion *rgn, unsigned node, Region *addr, unsigned vers)
{
  unsigned num_ptrs;
  Pointer *rslt;

  num_ptrs = rgn->num_ptrs;
  rslt     = &(rgn->ptrs);

  if (num_ptrs == 0)
  {
    /* case (1) - no existing pointers
     */
    rslt->node = node;
    rslt->addr = addr;
    rslt->vers = vers;
    sanity(rslt->next == NULL);

    rgn->num_ptrs = 1;
  }
  else
  {
    do
    {
      if (rslt->node == node)
	break;
      rslt = rslt->next;
    } while (rslt != NULL);

    if (rslt != NULL)
    {
      /* case (2) - found a matching pointer
       */
      rslt = NULL;
    }
    else
    {
      /* case (3) - no matching pointers
       */
      PtrAlloc(rslt);

      rslt->node = node;
      rslt->addr = addr;
      rslt->vers = vers;
      rslt->next = rgn->ptrs.next;

      rgn->ptrs.next = rslt;
      rgn->num_ptrs  = num_ptrs + 1;
    }
  }

  return rslt;
}


void ptr_delete(HomeRegion *rgn, unsigned node, unsigned vers)
{
  unsigned num_ptrs;
  Pointer *prev;
  Pointer *next;

  num_ptrs = rgn->num_ptrs;
  prev     = &(rgn->ptrs);

  sanity(num_ptrs > 0);

  if (prev->node == node)
  {
    if (num_ptrs == 1)
    {
      /* case (1) - matching pointer is only pointer
       */
      sanity(prev->vers == vers);

#if defined(CRL_SANITY)
      prev->node = 0xdeadbeef;
      prev->addr = (Region *) 0xdeadbeef;
      prev->vers = 0xdeadbeef;
      sanity(prev->next == NULL);
#endif
    }
    else
    {
      /* case (2) - more than one pointer,
       * matching pointer is at head of list
       */
      sanity(prev->vers == vers);
      next = prev->next;

      /* since storage for the head of the list is not dynamically
       * allocated (it's allocated as a part of the HomeRegion
       * structure), to delete the element at the head of the list we
       * copy the 2nd element into the head and then free the storage
       * that had been allocated for the 2nd element.
       */
      *prev = *next;
      PtrFree(next);
    }
  }
  else
  {
    /* case (3) - more than one pointer,
     * matching pointer is not at head of list
     */
    while (1)
    {
      next = prev->next;
      sanity(next != NULL);

      if (next->node == node)
	break;

      prev = next;
    }

    /* next points at the pointer to be deleted; prev points to the
     * previous pointer in the list; delete next by setting prev->link
     * to the value of next->link, then free the storage that had been
     * allocated for next.
     */
    sanity(next->vers == vers);
    prev->next = next->next;
    PtrFree(next);
  }

  rgn->num_ptrs = num_ptrs - 1;
}
