/*
 * proto.c
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: proto.c,v 1.11 1995/08/24 08:25:10 tuna Exp $
 */

 /*
  * This file contains the protocol code for CRL.  Each functions defines
  * what action should be taken based on the current state of a region
  * and on what request has just been made.  The end of this file defines
  * some state structures which group the actions for a particular state.
  */

#include "crl_int.h"

#if defined(CRL_DEBUG)
#define DebugMarker(rgn, xtra)                 \
 do {                                          \
  (rgn)->core.info3 = (rgn)->core.info2;       \
  (rgn)->core.info2 = (rgn)->core.info1;       \
  (rgn)->core.info1 = (rgn)->core.info0;       \
  (rgn)->core.info0.state = (rgn)->core.state; \
  (rgn)->core.info0.file  = __FILE__;          \
  (rgn)->core.info0.line  = __LINE__;          \
  (rgn)->core.info0.extra = (xtra);            \
 } while (0)
#else
#define DebugMarker(rgn, xtra) \
 do { } while (0)
#endif

#if defined(CM5)

/* emperically, the performance of the small and large data transfer
 * mechanisms crosses over at around 256 bytes.
 *
 * the position of the crossoverpoint almost certainly depends on
 * things like hardware (CM-5 vs CM-5E), libraries (CMMD version), and
 * O/S (CMOST version). the value used here was obtained on a CM-5
 * platform (32 MHz Ross processors) running CMOST 7.4 Final and CMMD
 * version 3.3.
 */
#define DataXover (256)

#define SendMsgToRemote(type, ptr, rgn)        \
 do {                                          \
  sanity((ptr)->node < crl_num_nodes);         \
  sanity(CMAML_interrupt_status() == FALSE);   \
  CMAML_rpc((ptr)->node, rgn_msg_stub, (type), \
            (ptr)->addr, crl_self_addr,        \
            (ptr)->vers);                      \
 } while (0)

#define SendMsgInvToRemote(type, ptr, rgn)     \
 do {                                          \
  sanity((ptr)->node < crl_num_nodes);         \
  sanity(CMAML_interrupt_status() == FALSE);   \
  CMAML_rpc((ptr)->node, rgn_inv_stub, (type), \
            (rgn)->core.rgn_id, crl_self_addr, \
            (ptr)->vers);                      \
 } while (0)

#define SendMsgDataToRemote(type, ptr, rgn)    \
 do {                                          \
  if ((rgn)->core.size > DataXover) {          \
    (rgn)->core.send_cnt += 1;                 \
    sanity((ptr)->node < crl_num_nodes);       \
    sanity(CMAML_interrupt_status() == FALSE); \
    CMAML_rpc((ptr)->node,                     \
              rgn_data_rts,                    \
              (type),                          \
              (ptr)->addr,                     \
              crl_self_addr,                   \
              (ptr)->vers);                    \
  } else {                                     \
    fast_data_msg((ptr)->node,                 \
                  (type),                      \
                  (Region *) (ptr)->addr,      \
                  (ptr)->vers,                 \
                  (Region *) (rgn));           \
  }                                            \
 } while (0)

#define SendMsgToHome(type, rgn)               \
 do {                                          \
  sanity((rgn)->home < crl_num_nodes);         \
  sanity(CMAML_interrupt_status() == FALSE);   \
  CMAML_rpc((rgn)->home, rgn_msg_stub, (type), \
            (rgn)->home_addr, crl_self_addr,   \
            (rgn)->core.vers);                 \
 } while (0)

#define SendMsgReqToHome(type, rgn)            \
 do {                                          \
  sanity((rgn)->home < crl_num_nodes);         \
  sanity(CMAML_interrupt_status() == FALSE);   \
  CMAML_rpc((rgn)->home, rgn_msg_stub, (type), \
            (rgn)->home_addr, crl_self_addr,   \
            (rgn));                            \
 } while (0)

#define SendMsgDataToHome(type, rgn)           \
 do {                                          \
  if ((rgn)->core.size > DataXover) {          \
    (rgn)->core.send_cnt += 1;                 \
    sanity((rgn)->home < crl_num_nodes);       \
    sanity(CMAML_interrupt_status() == FALSE); \
    CMAML_rpc((rgn)->home,                     \
              rgn_data_rts,                    \
              (type),                          \
              (rgn)->home_addr,                \
              crl_self_addr,                   \
              (rgn)->core.vers);               \
  } else {                                     \
    fast_data_msg((rgn)->home,                 \
                  (type),                      \
                  (Region *) (rgn)->home_addr, \
                  (rgn)->core.vers,            \
                  (Region *) (rgn));           \
  }                                            \
 } while (0)

#elif defined(ALEWIFE)

#define SendMsgToRemote(type, ptr, rgn)         \
 do {                                           \
  sanity((ptr)->node < crl_num_nodes);          \
  user_do_on((ptr)->node, rgn_msg_stub, (type), \
             (ptr)->addr, crl_self_addr,        \
             (ptr)->vers);                      \
 } while (0)

#define SendMsgInvToRemote(type, ptr, rgn)      \
 do {                                           \
  sanity((ptr)->node < crl_num_nodes);          \
  user_do_on((ptr)->node, rgn_inv_stub, (type), \
             (rgn)->core.rgn_id, crl_self_addr, \
             (ptr)->vers);                      \
 } while (0)

#define SendMsgDataToRemote(type, ptr, rgn)     \
 send_data_msg((ptr)->node,                     \
               (type),                          \
               (Region *) (ptr)->addr,          \
               (ptr)->vers,                     \
               (Region *) (rgn))

#define SendMsgToHome(type, rgn)                \
 do {                                           \
  sanity((rgn)->home < crl_num_nodes);          \
  user_do_on((rgn)->home, rgn_msg_stub, (type), \
             (rgn)->home_addr, crl_self_addr,   \
             (rgn)->core.vers);                 \
 } while (0)

#define SendMsgReqToHome(type, rgn)             \
 do {                                           \
  sanity((rgn)->home < crl_num_nodes);          \
  user_do_on((rgn)->home, rgn_msg_stub, (type), \
             (rgn)->home_addr, crl_self_addr,   \
             (rgn));                            \
 } while (0)

#define SendMsgDataToHome(type, rgn)            \
 send_data_msg((rgn)->home,                     \
               (type),                          \
               (Region *) (rgn)->home_addr,     \
               (rgn)->core.vers,                \
               (Region *) (rgn))

#elif defined(TCPUNIX)

#define SendMsgToRemote(type, ptr, rgn)             \
 do {                                               \
  sanity((ptr)->node < crl_num_nodes);              \
  sanity(TCP_interrupt_status() == FALSE);          \
  tcp_am_send_cm((ptr)->node, rgn_msg_stub, (type), \
            (ptr)->addr, crl_self_addr,             \
            (ptr)->vers);                           \
 } while (0)

#define SendMsgInvToRemote(type, ptr, rgn)          \
 do {                                               \
  sanity((ptr)->node < crl_num_nodes);              \
  sanity(TCP_interrupt_status() == FALSE);          \
  tcp_am_send_cm((ptr)->node, rgn_inv_stub, (type), \
            (rgn)->core.rgn_id, crl_self_addr,      \
            (ptr)->vers);                           \
 } while (0)

#define SendMsgDataToRemote(type, ptr, rgn)     \
 do {                                           \
    sanity((ptr)->node < crl_num_nodes);        \
    sanity(TCP_interrupt_status() == FALSE);    \
    (rgn)->core.send_cnt += 1;                  \
    send_data_msg((ptr)->node,                  \
              (type),                           \
              (ptr)->addr,                      \
              (ptr)->vers,                      \
              (Region *)rgn);                   \
 } while (0)

#define SendMsgToHome(type, rgn)                    \
 do {                                               \
  sanity((rgn)->home < crl_num_nodes);              \
  sanity(TCP_interrupt_status() == FALSE);          \
  tcp_am_send_cm((rgn)->home, rgn_msg_stub, (type), \
            (rgn)->home_addr, crl_self_addr,        \
            (rgn)->core.vers);                      \
 } while (0)

#define SendMsgReqToHome(type, rgn)                 \
 do {                                               \
  sanity((rgn)->home < crl_num_nodes);              \
  sanity(TCP_interrupt_status() == FALSE);          \
  tcp_am_send_cm((rgn)->home, rgn_msg_stub, (type), \
            (rgn)->home_addr, crl_self_addr,        \
            (rgn));                                 \
 } while (0)

#define SendMsgDataToHome(type, rgn)            \
 do {                                           \
    sanity((rgn)->home < crl_num_nodes);        \
    sanity(TCP_interrupt_status() == FALSE);    \
    (rgn)->core.send_cnt += 1;                  \
    send_data_msg((rgn)->home,                  \
              (type),                           \
              (rgn)->home_addr,                 \
              (rgn)->core.vers,                 \
              (Region *)rgn);                   \
 } while (0)

#endif

static int HomeShared_CallStartWrite_cont(HomeRegion *);
static int HomeShared_MsgExclusiveReq_cont(HomeRegion *);
static int HomeShared_MsgModifyReq_cont(HomeRegion *);
static int HomeInvalid_CallStartRead_cont(HomeRegion *);
static int HomeInvalid_CallStartWrite_cont(HomeRegion *);
static int HomeInvalid_MsgSharedReq_cont(HomeRegion *);
static int HomeInvalid_MsgExclusiveReq_cont(HomeRegion *);
static int HomeInvalid_MsgModifyReq_cont(HomeRegion *);

static void protocol_error(Region *rgn);


/* ---------------- HomeExclusive ----------------
 */

static void HomeExclusive_CallStartRead(HomeRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeExclusiveRip;
  rgn->core.read_cnt = 1;

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void HomeExclusive_CallStartWrite(HomeRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeExclusiveWip;

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static int HomeExclusive_MsgSharedReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* ABSTRACTION VIOLATION: since we're in the HomeExclusive state, we
   * know that num_ptrs == 0, so instead of calling ptr_insert() to
   * add the new pointer, we just bash it in directly
   *
   * insert pointer and increment version number
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);
  rgn->num_ptrs = 1;

  /* send ack+data
   */
  SendMsgDataToRemote(MsgSharedAckData, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgSharedAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeShared;

  return NormalReturn;
}


static int HomeExclusive_MsgExclusiveReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* ABSTRACTION VIOLATION: since we're in the HomeExclusive state, we
   * know that num_ptrs == 0, so instead of calling ptr_insert() to
   * add the new pointer, we just bash it in directly
   *
   * insert pointer and increment version number
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);
  rgn->num_ptrs = 1;

  /* send ack+data
   */
  SendMsgDataToRemote(MsgExclusiveAckData, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgExclusiveAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeInvalid;

  return NormalReturn;
}


static int HomeExclusive_MsgModifyReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* ABSTRACTION VIOLATION: since we're in the HomeExclusive state, we
   * know that num_ptrs == 0, so instead of calling ptr_insert() to
   * add the new pointer, we just bash it in directly
   *
   * insert pointer and increment version number
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);
  rgn->num_ptrs = 1;

  /* send ack+data
   */
  SendMsgDataToRemote(MsgModifyAckData, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgModifyAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeInvalid;

  return NormalReturn;
}


/* ---------------- HomeExclusiveRip ----------------
 */

static void HomeExclusiveRip_CallStartRead(HomeRegion *rgn)
{
  rgn->core.read_cnt += 1;

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void HomeExclusiveRip_CallEndRead(HomeRegion *rgn)
{
  unsigned cnt;
#if defined(CM5)
  int      reenable;
#elif defined(TCPUNIX)
  int      oldmask;
#endif

  cnt = rgn->core.read_cnt - 1;

  if (cnt > 0)
  {
    rgn->core.read_cnt = cnt;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else if (ProtMsgQueuePeek(&(rgn->blocked_msgs)) == NULL)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeExclusive;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeExclusive;
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    retry_blocked_msgs(rgn);

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


static int HomeExclusiveRip_MsgSharedReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* ABSTRACTION VIOLATION: since we're in the HomeExclusiveRip state,
   * we know that num_ptrs == 0, so instead of calling ptr_insert() to
   * add the new pointer, we just bash it in directly
   *
   * insert pointer and increment version number
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);
  rgn->num_ptrs = 1;

  /* send ack+data
   */
  SendMsgDataToRemote(MsgSharedAckData, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgSharedAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeSharedRip;

  return NormalReturn;
}


/* ---------------- HomeExclusiveWip ----------------
 */

static void HomeExclusiveWip_CallEndWrite(HomeRegion *rgn)
{
#if defined(CM5)
  int reenable;
#elif defined(TCPUNIX)
  int oldmask = crl_sigblock(0);
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeExclusive;

  if (ProtMsgQueuePeek(&(rgn->blocked_msgs)) == NULL)
  {
#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    retry_blocked_msgs(rgn);

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)    
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


/* ---------------- HomeShared ----------------
 */

static void HomeShared_CallStartRead(HomeRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeSharedRip;
  rgn->core.read_cnt = 1;

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void HomeShared_CallStartWrite(HomeRegion *rgn)
{
  Pointer *ptr;
  Pointer *last;

#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
  incoming_queue.enabled = 1;
  user_atomic_release();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  /* send out invalidate messages
   */
  sanity(rgn->num_ptrs > 0);
  ptr = &(rgn->ptrs);
  do
  {
    SendMsgInvToRemote(MsgWInvalidate, ptr, rgn);
    last = ptr;
    ptr = ptr->next;
#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgWInvalidate] += 1;
#endif
  } while (ptr != NULL);

  PtrFreeMany(rgn, last);

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeShared_CallStartWrite_cont;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIip;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteMiss] += 1;
#endif

  /* enable message delivery and wait until transaction is done
   */
#if defined(CM5)
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
  crl_sigsetmask(oldmask);
#endif

  sanity(rgn->core.state == &state_HomeExclusiveWip);
}


static int HomeShared_CallStartWrite_cont(HomeRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeExclusiveWip;

  return NormalReturn;
}


static int HomeShared_MsgSharedReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region  *addr;
  Pointer *ptr;
  int      rslt;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* attempt to insert pointer
   */
  vers = rgn->core.vers;
  ptr  = ptr_insert(rgn, src, addr, vers);

  if (ptr != NULL)
  {
    /* pointer inserted successfully; send ack+data and update version
     */
    SendMsgDataToRemote(MsgSharedAckData, ptr, rgn);
    rgn->core.vers = vers + 1;

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgSharedAckData] += 1;
#endif

    rslt = NormalReturn;
  }
  else
  {
    /* queue protocol message for later processing
     */
    rslt = MessageBlocked;
  }

  return rslt;
}


static int HomeShared_MsgExclusiveReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region  *addr;
  Pointer *ptr;
  Pointer *last;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* send out invalidate messages
   */
  sanity(rgn->num_ptrs > 0);
  ptr = &(rgn->ptrs);
  do
  {
    SendMsgInvToRemote(MsgWInvalidate, ptr, rgn);
    last = ptr;
    ptr = ptr->next;
#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgWInvalidate] += 1;
#endif
  } while (ptr != NULL);

  PtrFreeMany(rgn, last);

  /* prepare new pointer and increment the version number
   *
   * ABSTRACTION VIOLATION: note that we bash the new pointer in
   * directly instead of calling ptr_insert(); this is both for
   * performance (avoiding the function call) and correctness --
   * rgn->num_ptrs is being used to count the returning acks, so we
   * don't want to bash it
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeShared_MsgExclusiveReq_cont;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIip;

  return NormalReturn;
}


static int HomeShared_MsgExclusiveReq_cont(HomeRegion *rgn)
{
  SendMsgDataToRemote(MsgExclusiveAckData, &(rgn->ptrs), rgn);
  rgn->num_ptrs = 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgExclusiveAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeInvalid;

  return RetryBlocked;
}


static int HomeShared_MsgModifyReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region  *addr;
  unsigned ack_with_data;
  unsigned num_ptrs;
  Pointer *ptr;
  Pointer *last;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  ack_with_data = 1;
  num_ptrs = rgn->num_ptrs;
  sanity(num_ptrs > 0);

  /* send out invalidate messages
   */
  ptr = &(rgn->ptrs);
  do
  {
    if (ptr->node != src)
    {
      SendMsgInvToRemote(MsgWInvalidate, ptr, rgn);
#if defined(STATISTICS)
      crl_stat_cntrs[StatCntrMsgWInvalidate] += 1;
#endif
    }
    else
    {
      /* since we found a matching pointer, clear ack_with_data and
       * decrement num_ptrs so it matches the number of invalidations
       * that get sent out
       */
      ack_with_data = 0;
      num_ptrs     -= 1;
    }
    last = ptr;
    ptr = ptr->next;
  } while (ptr != NULL);

  PtrFreeMany(rgn, last);

  if (num_ptrs == 0)
  {
    /* the only way (num_ptrs == 0) should be possible is if there was
     * already a pointer for the requesting node, thus we executed the
     * conditional branch that clears ack_with_data and decrements
     * num_ptrs; sanity check that here
     */
    sanity(ack_with_data == 0);

    /* ABSTRACTION VIOLATION: since we know there's only one pointer,
     * instead of clearing it and then calling ptr_insert() to insert
     * the new one, just bash the new pointer in directly
     */
    rgn->ptrs.node = src;
    rgn->ptrs.addr = addr;
    rgn->ptrs.vers = (rgn->core.vers)++;
    sanity(rgn->ptrs.next == NULL);
    sanity(rgn->num_ptrs == 1);

    /* send ack
     */
    SendMsgToRemote(MsgModifyAck, &(rgn->ptrs), rgn);
#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgModifyAck] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeInvalid;
  }
  else
  {
    /* prepare new pointer and increment the version number
     *
     * ABSTRACTION VIOLATION: note that we bash the new pointer in
     * directly instead of calling ptr_insert(); this is both for
     * performance (avoiding the function call) and correctness --
     * rgn->num_ptrs is being used to count the returning acks, so we
     * don't want to bash it
     */
    rgn->ptrs.node = src;
    rgn->ptrs.addr = addr;
    rgn->ptrs.vers = (rgn->core.vers)++;
    sanity(rgn->ptrs.next == NULL);

    /* prepare for transaction completion
     */
    rgn->core.tx_done = 0;
    rgn->core.tx_cont = (TxCont) HomeShared_MsgModifyReq_cont;

    /* use .tx_arg1 to hold ack_with_data (so the continuation knows
     * whether or not to send data along with the ack)
     */
    rgn->core.tx_arg1 = ack_with_data;

    if (ack_with_data == 0)
    {
      /* if we cleared ack_with_data, we also decremented num_ptrs, so make
       * sure to store the new value back into rgn->num_ptrs
       */
      rgn->num_ptrs = num_ptrs;
    }

    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeIip;
  }

  return NormalReturn;
}


static int HomeShared_MsgModifyReq_cont(HomeRegion *rgn)
{
  if (rgn->core.tx_arg1)
  {
    SendMsgDataToRemote(MsgModifyAckData, &(rgn->ptrs), rgn);
#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgModifyAckData] += 1;
#endif
  }
  else
  {
    SendMsgToRemote(MsgModifyAck, &(rgn->ptrs), rgn);
#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgModifyAck] += 1;
#endif
  }

  rgn->num_ptrs = 1;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeInvalid;

  return RetryBlocked;
}


static int HomeShared_MsgFlush(HomeRegion *rgn, unsigned src, unsigned vers)
{
  ptr_delete(rgn, src, vers);

  if (rgn->num_ptrs == 0)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeExclusive;
  }

  return RetryBlocked;
}


/* ---------------- HomeSharedRip ----------------
 */

static void HomeSharedRip_CallStartRead(HomeRegion *rgn)
{
  rgn->core.read_cnt += 1;

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void HomeSharedRip_CallEndRead(HomeRegion *rgn)
{
  unsigned cnt;
#if defined(CM5)
  int      reenable;
#elif defined(TCPUNIX)
  int      oldmask;
#endif

  cnt = rgn->core.read_cnt - 1;

  if (cnt > 0)
  {
    rgn->core.read_cnt = cnt;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else if (ProtMsgQueuePeek(&(rgn->blocked_msgs)) == NULL)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeShared;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeShared;
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    retry_blocked_msgs(rgn);

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


static int HomeSharedRip_MsgSharedReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region  *addr;
  Pointer *ptr;
  int      rslt;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* attempt to insert pointer
   */
  vers = rgn->core.vers;
  ptr  = ptr_insert(rgn, src, addr, vers);

  if (ptr != NULL)
  {
    /* pointer inserted successfully; send ack+data and update version
     */
    SendMsgDataToRemote(MsgSharedAckData, ptr, rgn);
    rgn->core.vers = vers + 1;

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgSharedAckData] += 1;
#endif

    rslt = NormalReturn;
  }
  else
  {
    /* queue protocol message for later processing
     */
    rslt = MessageBlocked;
  }

  return rslt;
}


static int HomeSharedRip_MsgFlush(HomeRegion *rgn, unsigned src, unsigned vers)
{
  ptr_delete(rgn, src, vers);

  if (rgn->num_ptrs == 0)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeExclusiveRip;
  }

  return RetryBlocked;
}


/* ---------------- HomeIip ----------------
 */

/* also used to handle CallStartRead in HomeIipSpecial state
 */
static void HomeIip_CallStartRead(HomeRegion *rgn)
{
#if defined(CM5)
  int reenable;
#elif defined(TCPUNIX)
  int oldmask;
#endif

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadBlock] += 1;
#endif

  /* reenable message delivery and wait
   * until current transaction is done
   */
#if defined(CM5)
  reenable = CMAML_disable_interrupts();
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
#elif defined(ALEWIFE)
  user_atomic_release();
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  oldmask = crl_sigblock(sigmask(SIGIO));
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
#endif

  /* retry call
   */
#if defined(ALEWIFE)
  user_atomic_request();
#else  
  incoming_queue.enabled = 1;
#endif
  CallEvent(CallStartRead, ((Region *) rgn));

#if defined(CM5)
  if (reenable) CMAML_enable_interrupts();
#elif defined(TCPUNIX)
  crl_sigsetmask(oldmask);
#endif
}


/* also used to handle CallStartWrite in HomeIipSpecial state
 */
static void HomeIip_CallStartWrite(HomeRegion *rgn)
{
#if defined(CM5)
  int reenable;
#elif defined(TCPUNIX)
  int oldmask;
#endif

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteBlock] += 1;
#endif

  /* reenable message delivery and wait
   * until current transaction is done
   */
#if defined(CM5)
  reenable = CMAML_disable_interrupts();
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
#elif defined(ALEWIFE)
  user_atomic_release();
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  oldmask = crl_sigblock(sigmask(SIGIO));
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
#endif

  /* retry call
   */
#if defined(ALEWIFE)
  user_atomic_request();
#else
  incoming_queue.enabled = 1;
#endif
  CallEvent(CallStartWrite, ((Region *) rgn));

#if defined(CM5)
  if (reenable) CMAML_enable_interrupts();
#elif defined(TCPUNIX)
  crl_sigsetmask(oldmask);
#endif
}


/* also used for MsgInvalidateAckData,
 * MsgFlush, and MsgFlushData messages
 */
static int HomeIip_MsgInvalidateAck(HomeRegion *rgn, unsigned src, unsigned vers)
{
  unsigned num_ptrs;
  int      rslt;

  num_ptrs = rgn->num_ptrs - 1;

  if (num_ptrs == 0)
  {
    /* mark transaction complete and call continuation
     */
    rgn->core.tx_done = 1;
    rslt = (rgn->core.tx_cont)(rgn);
  }
  else
  {
    rgn->num_ptrs = num_ptrs;
    rslt = NormalReturn;
  }

  return rslt;
}


/* ---------------- HomeIipSpecial ----------------
 */

/* also used for MsgInvalidateAckData,
 * MsgFlush, and MsgFlushData messages
 */
static int HomeIipSpecial_MsgInvalidateAck(HomeRegion *rgn, unsigned src, unsigned vers)
{
  int rslt;

  /* since HomeIipSpecial can only be entered via a CallStartRead or
   * a MsgSharedReq from the HomeInvalid state, rgn->num_ptrs should
   * definitely be 1 here
   */
  sanity(rgn->num_ptrs == 1);

  /* mark transaction complete and call continuation (set .tx_arg1 to 0 so
   * continuation can determine it was called due to the arrival of a
   * MsgInvalidateAck)
   */
  rgn->core.tx_done = 1;
  rgn->core.tx_arg1 = 0;
  rslt = (rgn->core.tx_cont)(rgn);

  return rslt;
}


static int HomeIipSpecial_MsgRelease(HomeRegion *rgn, unsigned src, unsigned vers)
{
  int rslt;

  /* since HomeIipSpecial can only be entered via a CallStartRead or
   * a MsgSharedReq from the HomeInvalid state, rgn->num_ptrs should
   * definitely be 1 here
   */
  sanity(rgn->num_ptrs == 1);

  /* with out-of-order message delivery, it is possible for a MsgFlush
   * sent later (in time) than a MsgRelease to arrive out of order --
   * MsgFlush before MsgRelease. when the MsgFlush shows up, the
   * MsgInvalidateAck handler (above) will do the right thing, so the
   * only thing that remains is to sink the stale MsgRelease that will
   * show up eventually.
   *
   * this is accomplished with the following policy:
   *
   * (1) in any state other than HomeIipSpecial, ignore MsgRelease
   *     messages
   *
   * (2) in HomeIipSpecial, check a MsgRelease message's version
   *     number against that of the single pointer; if they don't
   *     match, the MsgRelease message is stale and can be safely
   *     ignored
   */
  if (rgn->ptrs.vers == vers)
  {
    /* mark transaction complete and call continuation (set .tx_arg1 to 1
     * so continuation can determine it was called due to the arrival of a
     * MsgRelease)
     */
    rgn->core.tx_done = 1;
    rgn->core.tx_arg1 = 1;
    rslt = (rgn->core.tx_cont)(rgn);
  }
  else
  {
    rslt = NormalReturn;
  }

  return rslt;
}


/* ---------------- HomeInvalid ---------------- */

static void HomeInvalid_CallStartRead(HomeRegion *rgn)
{
#if defined(CRL_SANITY)
  State *st;
#endif
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
  incoming_queue.enabled = 1;
  user_atomic_release();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  /* send out invalidate message
   */
  sanity(rgn->num_ptrs == 1);
  SendMsgInvToRemote(MsgRInvalidate, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRInvalidate] += 1;
#endif

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeInvalid_CallStartRead_cont;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIipSpecial;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadMiss] += 1;
#endif

  /* enable message delivery and wait until transaction is done
   */
#if defined(CM5)
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
  crl_sigsetmask(oldmask);
#endif

#if defined(CRL_SANITY)
  /* make sure rgn->core.state is only loaded once so there's no
   * possibility of the sanity check loading it twice and getting
   * different values each time.
   */
  st = rgn->core.state;
  sanity((st == &state_HomeExclusiveRip) ||
	 (st == &state_HomeSharedRip));
#endif
}


static int HomeInvalid_CallStartRead_cont(HomeRegion *rgn)
{
  if (rgn->core.tx_arg1)
  {
    /* called because of a MsgRelease
     */
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeSharedRip;
    rgn->core.read_cnt = 1;
  }
  else
  {
    /* called because of a MsgInvalidateAck
     */
    DebugMarker(rgn, 0);
    rgn->core.state = &state_HomeExclusiveRip;
    rgn->core.read_cnt = 1;
  }

  /* ??? */
  return RetryBlocked;
}


static void HomeInvalid_CallStartWrite(HomeRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
  incoming_queue.enabled = 1;
  user_atomic_release();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  /* send out invalidate message
   */
  sanity(rgn->num_ptrs == 1);
  SendMsgInvToRemote(MsgWInvalidate, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgWInvalidate] += 1;
#endif

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeInvalid_CallStartWrite_cont;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIip;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteMiss] += 1;
#endif

  /* enable message delivery and wait until transaction is done
   */
#if defined(CM5)
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
  crl_sigsetmask(oldmask);
#endif

  sanity(rgn->core.state == &state_HomeExclusiveWip);
}


static int HomeInvalid_CallStartWrite_cont(HomeRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeExclusiveWip;

  return NormalReturn;
}


static int HomeInvalid_MsgSharedReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* send out invalidate message
   */
  sanity(rgn->num_ptrs == 1);
  SendMsgInvToRemote(MsgRInvalidate, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRInvalidate] += 1;
#endif

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeInvalid_MsgSharedReq_cont;

  /* use .tx_arg2 and .tx_arg3 to hold src and addr of request
   * (.tx_arg1 already in use by HomeIipSpecial handlers)
   */
  rgn->core.tx_arg2 = src;
  rgn->core.tx_arg3 = (unsigned) addr;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIipSpecial;

  return NormalReturn;
}


static int HomeInvalid_MsgSharedReq_cont(HomeRegion *rgn)
{
  unsigned node;
  void    *addr;
  unsigned vers;
  Pointer *ptr;

  /* src and addr of request are kept in .tx_arg2 and .tx_arg3
   * (see above)
   */
  node = rgn->core.tx_arg2;
  addr = (void *) rgn->core.tx_arg3;
  vers = rgn->core.vers;

  if (rgn->core.tx_arg1)
  {
    /* called because of a MsgRelease
     */
    ptr = ptr_insert(rgn, node, addr, vers);
    sanity(ptr != NULL);
    sanity(rgn->num_ptrs == 2);
  }
  else
  {
    /* called because of a MsgInvalidateAck
     */
    ptr = &(rgn->ptrs);

    /* ABSTRACTION VIOLATION: since we know there's only one pointer
     * and we don't want it any more, just bash the new pointer in
     * directly instead of messing around with ptr_delete() and
     * ptr_insert()
     *
     * insert pointer
     */
    ptr->node = node;
    ptr->addr = addr;
    ptr->vers = vers;
    sanity(ptr->next == NULL);
    sanity(rgn->num_ptrs == 1);
  }

  /* send ack+data and update version
   */
  SendMsgDataToRemote(MsgSharedAckData, ptr, rgn);
  rgn->core.vers = vers + 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgSharedAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeShared;

  return RetryBlocked;
}


static int HomeInvalid_MsgExclusiveReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* send out invalidate message
   */
  sanity(rgn->num_ptrs == 1);
  SendMsgInvToRemote(MsgWInvalidate, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgWInvalidate] += 1;
#endif

  /* prepare new pointer and increment the version number
   *
   * ABSTRACTION VIOLATION: note that we bash the new pointer in
   * directly instead of calling ptr_insert(); this is both for
   * performance (avoiding the function call) and correctness --
   * rgn->num_ptrs is being used to count the returning acks, so we
   * don't want to bash it
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeInvalid_MsgExclusiveReq_cont;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIip;

  return NormalReturn;
}


static int HomeInvalid_MsgExclusiveReq_cont(HomeRegion *rgn)
{
  SendMsgDataToRemote(MsgExclusiveAckData, &(rgn->ptrs), rgn);
  rgn->num_ptrs = 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgExclusiveAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeInvalid;

  return RetryBlocked;
}


static int HomeInvalid_MsgModifyReq(HomeRegion *rgn, unsigned src, unsigned vers)
{
  Region *addr;

  /* in all Request messages, the vers argument is actually the
   * address of the requesting RemoteRegion
   */
  addr = (Region *) vers;

  /* send out invalidate message
   */
  sanity(rgn->num_ptrs == 1);
  SendMsgInvToRemote(MsgWInvalidate, &(rgn->ptrs), rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgWInvalidate] += 1;
#endif

  /* prepare new pointer and increment the version number
   *
   * ABSTRACTION VIOLATION: note that we bash the new pointer in
   * directly instead of calling ptr_insert(); this is both for
   * performance (avoiding the function call) and correctness --
   * rgn->num_ptrs is being used to count the returning acks, so we
   * don't want to bash it
   */
  rgn->ptrs.node = src;
  rgn->ptrs.addr = addr;
  rgn->ptrs.vers = (rgn->core.vers)++;
  sanity(rgn->ptrs.next == NULL);

  /* save continuation
   */
  rgn->core.tx_done = 0;
  rgn->core.tx_cont = (TxCont) HomeInvalid_MsgModifyReq_cont;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeIip;

  return NormalReturn;
}


static int HomeInvalid_MsgModifyReq_cont(HomeRegion *rgn)
{
  SendMsgDataToRemote(MsgModifyAckData, &(rgn->ptrs), rgn);
  rgn->num_ptrs = 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgModifyAckData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeInvalid;

  return RetryBlocked;
}


/* also used for MsgFlushData messages
 */
static int HomeInvalid_MsgFlush(HomeRegion *rgn, unsigned src, unsigned vers)
{
  sanity(rgn->num_ptrs == 1);

  /* ABSTRACTION VIOLATION: since we know that there is only one
   * pointer, we just bash rgn->num_ptrs to zero instead of calling
   * ptr_delete()
   */
  rgn->num_ptrs = 0;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_HomeExclusive;

  return NormalReturn;
}


/* ---------------- RemoteInvalid ----------------
 */

static void RemoteInvalid_CallStartRead(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
  incoming_queue.enabled = 1;
  user_atomic_release();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  /* send request message
   */
  SendMsgReqToHome(MsgSharedReq, rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgSharedReq] += 1;
#endif

  /* save continuation
   */
  rgn->core.tx_done = 0;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteInvalidReq;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
  crl_stat_cntrs[StatCntrStartReadMiss]   += 1;
#endif

  /* enable message delivery and wait until transaction is done
   */
#if defined(CM5)
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
  crl_sigsetmask(oldmask);
#endif

  sanity(rgn->core.state == &state_RemoteSharedRip);
}


static void RemoteInvalid_CallStartWrite(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
  incoming_queue.enabled = 1;
  user_atomic_release();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  /* send request message
   */
  SendMsgReqToHome(MsgExclusiveReq, rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgExclusiveReq] += 1;
#endif

  /* save continuation
   */
  rgn->core.tx_done = 0;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteInvalidReq;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteRemote] += 1;
  crl_stat_cntrs[StatCntrStartWriteMiss]   += 1;
#endif

  /* enable message delivery and wait until transaction is done
   */
#if defined(CM5)
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
  crl_sigsetmask(oldmask);
#endif  

  sanity(rgn->core.state == &state_RemoteModifiedWip);
}


/* handler for MsgRInvalidate and MsgWInvalidate messages
 */
static int RemoteInvalid_MsgInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  /* invalidate message can be safely ignored
   * if delta is less than or equal to zero
   */
  if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }

  return NormalReturn;
}


/* ---------------- RemoteInvalidReq ----------------
 */

/* handler for MsgRInvalidate and MsgWInvalidate messages
 */
static int RemoteInvalidReq_MsgInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  /* invalidate message can be safely ignored
   * if delta is less than or equal to zero
   */
  if (((int) delta) > 0)
  {
    /* set received-invalidate flag (and remember
     * version number for later reference)
     */
    sanity(rgn->received_invalidate == 0);
    rgn->received_invalidate = 1;
    rgn->core.vers = vers;
  }

  return NormalReturn;
}


static int RemoteInvalidReq_MsgSharedAckData(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  if (rgn->received_invalidate)
    sanity(rgn->core.vers == vers);
  else
    rgn->core.vers = vers;

  /* mark transaction complete
   */
  rgn->core.tx_done = 1;

  /* update state
   */
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteSharedRip;
  rgn->core.read_cnt = 1;

  return NormalReturn;
}


static int RemoteInvalidReq_MsgExclusiveAckData(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  if (rgn->received_invalidate)
    sanity(rgn->core.vers == vers);
  else
    rgn->core.vers = vers;

  /* mark transaction complete
   */
  rgn->core.tx_done = 1;

  /* update state
   */
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteModifiedWip;

  return NormalReturn;
}


static int RemoteInvalidReq_MsgModifyAckData(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  if (rgn->received_invalidate)
    sanity(rgn->core.vers == vers);
  else
    rgn->core.vers = vers;

  /* mark transaction complete
   */
  rgn->core.tx_done = 1;

  /* update state
   */
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteModifiedWip;

  return NormalReturn;
}


/* ---------------- RemoteShared ----------------
 */

static void RemoteShared_CallStartRead(RemoteRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteSharedRip;
  rgn->core.read_cnt = 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteShared_CallStartWrite(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
  incoming_queue.enabled = 1;
  user_atomic_release();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  /* send request message
   */
  SendMsgReqToHome(MsgModifyReq, rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgModifyReq] += 1;
#endif

  /* save continuation
   */
  rgn->core.tx_done = 0;

  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteSharedReq;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteRemote] += 1;
  crl_stat_cntrs[StatCntrStartWriteMiss]   += 1;
#endif

  /* enable message delivery and wait until transaction is done
   */
#if defined(CM5)
  actmsg_poll_no_intr(0);
  CMAML_poll_while_lt(&(rgn->core.tx_done), 1);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  while (*((volatile unsigned *) &(rgn->core.tx_done)) == 0)
    sigpause(0);
  crl_sigsetmask(oldmask);
#endif

  sanity(rgn->core.state == &state_RemoteModifiedWip);
}


static void RemoteShared_CallFlush(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  SendMsgToHome(MsgFlush, rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgFlush] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
  actmsg_poll_no_intr(0);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  crl_sigsetmask(oldmask);
#endif
}


static int RemoteShared_MsgWInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* ack invalidate message and change state
     */
    SendMsgToHome(MsgInvalidateAck, rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAck] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate message
     */
  }

  return NormalReturn;
}


/* ---------------- RemoteSharedReq ----------------
 */

/* handler for MsgRInvalidate and MsgWInvalidate messages
 */
static int RemoteSharedReq_MsgInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* ack invalidate message and change state
     */
    SendMsgToHome(MsgInvalidateAck, rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAck] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalidReq;
  }
  else if (((int) delta) > 0)
  {
    /* set received-invalidate flag (and remember
     * version number for later reference)
     */
    sanity(rgn->received_invalidate == 0);
    rgn->received_invalidate = 1;
    rgn->core.vers = vers;
  }
  else
  {
    /* ignore invalidate mesasge
     */
  }

  return NormalReturn;
}


/* handler for MsgModifyAck and MsgModifyAckData messages
 */
static int RemoteSharedReq_MsgModifyAck(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  if (rgn->received_invalidate)
    sanity(rgn->core.vers == vers);
  else
    rgn->core.vers = vers;

  /* mark transaction complete
   */
  rgn->core.tx_done = 1;

  /* update state
   */
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteModifiedWip;

  return NormalReturn;
}


/* ---------------- RemoteSharedRip ----------------
 */

static void RemoteSharedRip_CallStartRead(RemoteRegion *rgn)
{
  rgn->core.read_cnt += 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteSharedRip_CallEndRead(RemoteRegion *rgn)
{
  unsigned cnt;
#if defined(CM5)
  int      reenable;
#elif defined(TCPUNIX)
  int      oldmask;
#endif

  cnt = rgn->core.read_cnt - 1;

  if (cnt > 0)
  {
    rgn->core.read_cnt = cnt;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else if (rgn->received_invalidate == 0)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteShared;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    SendMsgToHome(MsgInvalidateAck, rgn);
    rgn->received_invalidate = 0;

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAck] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


static int RemoteSharedRip_MsgWInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* set received-invalidate flag
     */
    sanity(rgn->received_invalidate == 0);
    rgn->received_invalidate = 1;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate message
     */
  }

  return NormalReturn;
}


/* ---------------- RemoteExclusive ----------------
 */

static void RemoteExclusive_CallStartRead(RemoteRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state    = &state_RemoteExclusiveRip;
  rgn->core.read_cnt = 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteExclusive_CallStartWrite(RemoteRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteModifiedWip;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteExclusive_CallFlush(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  SendMsgToHome(MsgFlush, rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgFlush] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
  actmsg_poll_no_intr(0);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  crl_sigsetmask(oldmask);
#endif
}


/* handler for MsgRInvalidate and MsgWInvalidate messages
 */
static int RemoteExclusive_MsgInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* ack invalidate message and change state
     */
    SendMsgToHome(MsgInvalidateAck, rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAck] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate mesasge
     */
  }

  return NormalReturn;
}


/* ---------------- RemoteExclusiveRip ----------------
 */

static void RemoteExclusiveRip_CallStartRead(RemoteRegion *rgn)
{
  rgn->core.read_cnt += 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteExclusiveRip_CallEndRead(RemoteRegion *rgn)
{
  unsigned cnt;
#if defined(CM5)
  int      reenable;
#elif defined(TCPUNIX)
  int      oldmask;
#endif

  cnt = rgn->core.read_cnt - 1;

  if (cnt > 0)
  {
    rgn->core.read_cnt = cnt;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else if (rgn->received_invalidate == 0)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteExclusive;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    SendMsgToHome(MsgInvalidateAck, rgn);
    rgn->received_invalidate = 0;

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAck] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


static int RemoteExclusiveRip_MsgRInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* inform home node that it's ok to proceed
     */
    SendMsgToHome(MsgRelease, rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgRelease] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteSharedRip;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate message
     */
  }

  return NormalReturn;
}


static int RemoteExclusiveRip_MsgWInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* set received-invalidate flag
     */
    sanity(rgn->received_invalidate == 0);
    rgn->received_invalidate = 1;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate message
     */
  }

  return NormalReturn;
}


/* ---------------- RemoteModified ----------------
 */

static void RemoteModified_CallStartRead(RemoteRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteModifiedRip;
  rgn->core.read_cnt = 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteModified_CallStartWrite(RemoteRegion *rgn)
{
  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteModifiedWip;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWriteRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteModified_CallFlush(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  SendMsgDataToHome(MsgFlushData, rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgFlushData] += 1;
#endif

  DebugMarker(rgn, 0);
  rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
  actmsg_poll_no_intr(0);
  if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
  actmsg_poll(0);
#elif defined(TCPUNIX)
  actmsg_poll_no_intr(0);
  crl_sigsetmask(oldmask);
#endif
}


/* handler for MsgRInvalidate and MsgWInvalidate messages
 */
static int RemoteModified_MsgInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* ack invalidate message and change state
     */
    SendMsgDataToHome(MsgInvalidateAckData, rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAckData] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate mesasge
     */
  }

  return NormalReturn;
}


/* ---------------- RemoteModifiedRip ----------------
 */

static void RemoteModifiedRip_CallStartRead(RemoteRegion *rgn)
{
  rgn->core.read_cnt += 1;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartReadRemote] += 1;
#endif

#if defined(ALEWIFE)
  user_atomic_release();
#else
  actmsg_poll(0);
#endif
}


static void RemoteModifiedRip_CallEndRead(RemoteRegion *rgn)
{
  unsigned cnt;
#if defined(CM5)
  int      reenable;
#elif defined(TCPUNIX)
  int      oldmask;
#endif

  cnt = rgn->core.read_cnt - 1;

  if (cnt > 0)
  {
    rgn->core.read_cnt = cnt;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else if (rgn->received_invalidate == 0)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteModified;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    SendMsgDataToHome(MsgInvalidateAckData, rgn);
    rgn->received_invalidate = 0;

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAckData] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


static int RemoteModifiedRip_MsgRInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* inform home node that it's ok to proceed
     */
    SendMsgDataToHome(MsgRelease, rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgReleaseData] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteSharedRip;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate mesasge
     */
  }

  return NormalReturn;
}


static int RemoteModifiedRip_MsgWInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* set received-invalidate flag
     */
    sanity(rgn->received_invalidate == 0);
    rgn->received_invalidate = 1;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate mesasge
     */
  }

  return NormalReturn;
}


/* ---------------- RemoteModifiedWip ----------------
 */

static void RemoteModifiedWip_CallEndWrite(RemoteRegion *rgn)
{
#if defined(CM5)
  int reenable;
#elif defined(TCPUNIX)
  int oldmask = crl_sigblock(0);
#endif

  if (rgn->received_invalidate == 0)
  {
    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteModified;

#if defined(ALEWIFE)
    user_atomic_release();
#else
    actmsg_poll(0);
#endif
  }
  else
  {
#if defined(CM5)
    reenable = CMAML_disable_interrupts();
#elif defined(ALEWIFE)
    incoming_queue.enabled = 1;
    user_atomic_release();
#elif defined(TCPUNIX)
    oldmask = crl_sigblock(sigmask(SIGIO));
#endif

    SendMsgDataToHome(MsgInvalidateAckData, rgn);
    rgn->received_invalidate = 0;

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMsgInvalidateAckData] += 1;
#endif

    DebugMarker(rgn, 0);
    rgn->core.state = &state_RemoteInvalid;

#if defined(CM5)
    actmsg_poll_no_intr(0);
    if (reenable) CMAML_enable_interrupts();
#elif defined(ALEWIFE)
    actmsg_poll(0);
#elif defined(TCPUNIX)
    actmsg_poll_no_intr(0);
    crl_sigsetmask(oldmask);
#endif
  }
}


/* handler for MsgRInvalidate and MsgWInvalidate messages
 */
static int RemoteModifiedWip_MsgInvalidate(RemoteRegion *rgn, unsigned src, unsigned vers)
{
  unsigned delta;

  delta = vers - rgn->core.vers;

  if (delta == 0)
  {
    /* set received-invalidate flag
     */
    sanity(rgn->received_invalidate == 0);
    rgn->received_invalidate = 1;
  }
  else if (((int) delta) > 0)
  {
    /* signal protocol error
     */
    protocol_error((Region *) rgn);
  }
  else
  {
    /* ignore invalidate mesasge
     */
  }

  return NormalReturn;
}


/* generic call/message handlers
 */

static void call_error(Region *rgn)
{
#if defined(CM5)
  CMMD_error("CRL version %s: call error, node %u (rgn_id %u, state %d)\n",
	     CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
#elif defined(ALEWIFE)
  fprintf(stderr,
	  "CRL version %s: call error, node %u (rgn_id %u, state %d)\n",
	  CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
  fflush(stderr);
  abort();
#elif defined(TCPUNIX)
  fprintf(stderr,
	  "CRL version %s: call error, node %u (rgn_id %u, state %d)\n",
	  CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
  fflush(stderr);
  abort();
#endif
}


static void call_no_op(Region *rgn)
{
  actmsg_poll(0);
}


static int msg_no_op(Region *rgn, unsigned src, unsigned vers)
{
  return NormalReturn;
}


static int msg_error(Region *rgn, unsigned src, unsigned vers)
{
#if defined(CM5)
  CMMD_error("CRL version %s: message error, node %u (rgn_id %u, state %d)\n",
	     CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
#elif defined(ALEWIFE)
  fprintf(stderr,
	  "CRL version %s: message error, node %u (rgn_id %u, state %d)\n",
	  CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
  fflush(stderr);
  abort();
#elif defined(TCPUNIX)
  fprintf(stderr,
	  "CRL version %s: message error, node %u (rgn_id %u, state %d)\n",
	  CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
  fflush(stderr);
  abort();
#endif

  /* keep th' compiler happy
   */
  return NormalReturn;
}


static int msg_queue_self(HomeRegion *rgn, unsigned src, unsigned vers)
{
  return MessageBlocked;
}


static void protocol_error(Region *rgn)
{
#if defined(CM5)
  CMMD_error("CRL version %s: protocol error, node %u (rgn_id %u, state %d)\n",
	     CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
#elif defined(ALEWIFE)
  fprintf(stderr,
	  "CRL version %s: protocol error, node %u (rgn_id %u, state %d)\n",
	  CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
  fflush(stderr);
  abort();
#elif defined(TCPUNIX)
  fprintf(stderr,
	  "CRL version %s: protocol error, node %u (rgn_id %u, state %d)\n",
	  CRL_VERSION, crl_self_addr, rgn->rgn_id, rgn->state->state);
  fflush(stderr);
  abort();
#endif
}


/* ---------------- top-level State structures ----------------
 */

State state_HomeExclusive =
{
  HomeExclusive,
{
  (CallHandler) HomeExclusive_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) HomeExclusive_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) HomeExclusive_MsgSharedReq,
  (MsgHandler) HomeExclusive_MsgExclusiveReq,
  (MsgHandler) HomeExclusive_MsgModifyReq,
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_HomeExclusiveRip =
{
  HomeExclusiveRip,
{
  (CallHandler) HomeExclusiveRip_CallStartRead,
  (CallHandler) HomeExclusiveRip_CallEndRead,
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) HomeExclusiveRip_MsgSharedReq,
  (MsgHandler) msg_queue_self,		/* MsgExclusiveReq */
  (MsgHandler) msg_queue_self,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_HomeExclusiveWip =
{
  HomeExclusiveWip,
{
  (CallHandler) call_error,		/* CallStartRead */
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) HomeExclusiveWip_CallEndWrite,
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_queue_self,		/* MsgSharedReq */
  (MsgHandler) msg_queue_self,		/* MsgExclusiveReq */
  (MsgHandler) msg_queue_self,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_HomeShared =
{
  HomeShared,
{
  (CallHandler) HomeShared_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) HomeShared_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) HomeShared_MsgSharedReq,
  (MsgHandler) HomeShared_MsgExclusiveReq,
  (MsgHandler) HomeShared_MsgModifyReq,
  (MsgHandler) HomeShared_MsgFlush,
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_HomeSharedRip =
{
  HomeSharedRip,
{
  (CallHandler) HomeSharedRip_CallStartRead,
  (CallHandler) HomeSharedRip_CallEndRead,
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) HomeSharedRip_MsgSharedReq,
  (MsgHandler) msg_queue_self,		/* MsgExclusiveReq */
  (MsgHandler) msg_queue_self,		/* MsgModifyReq */
  (MsgHandler) HomeSharedRip_MsgFlush,
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_HomeIip =
{
  HomeIip,
{
  (CallHandler) HomeIip_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) HomeIip_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) HomeIip_MsgInvalidateAck,
  (MsgHandler) HomeIip_MsgInvalidateAck, /* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_queue_self,		/* MsgSharedReq */
  (MsgHandler) msg_queue_self,		/* MsgExclusiveReq */
  (MsgHandler) msg_queue_self,		/* MsgModifyReq */
  (MsgHandler) HomeIip_MsgInvalidateAck, /* MsgFlush */
  (MsgHandler) HomeIip_MsgInvalidateAck, /* MsgFlushData */
} };


State state_HomeIipSpecial =
{
  HomeIipSpecial,
{
  (CallHandler) HomeIip_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) HomeIip_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) HomeIipSpecial_MsgInvalidateAck,
  (MsgHandler) HomeIipSpecial_MsgInvalidateAck,	/* MsgInvalidateAckData */
  (MsgHandler) HomeIipSpecial_MsgRelease, /* MsgRelease */
  (MsgHandler) msg_queue_self,		/* MsgSharedReq */
  (MsgHandler) msg_queue_self,		/* MsgExclusiveReq */
  (MsgHandler) msg_queue_self,		/* MsgModifyReq */
  (MsgHandler) HomeIipSpecial_MsgInvalidateAck,	/* MsgFlush */
  (MsgHandler) HomeIipSpecial_MsgInvalidateAck,	/* MsgFlushData */
} };


State state_HomeInvalid =
{
  HomeInvalid,
{
  (CallHandler) HomeInvalid_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) HomeInvalid_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) msg_error,		/* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) HomeInvalid_MsgSharedReq,
  (MsgHandler) HomeInvalid_MsgExclusiveReq,
  (MsgHandler) HomeInvalid_MsgModifyReq,
  (MsgHandler) HomeInvalid_MsgFlush,
  (MsgHandler) HomeInvalid_MsgFlush,	/* MsgFlushData */
} };


State state_RemoteInvalid =
{
  RemoteInvalid,
{
  (CallHandler) RemoteInvalid_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) RemoteInvalid_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) RemoteInvalid_MsgInvalidate, /* MsgRInvalidate */
  (MsgHandler) RemoteInvalid_MsgInvalidate, /* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteInvalidReq =
{
  RemoteInvalidReq,
{
  (CallHandler) call_error,		/* CallStartRead */
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_error,		/* CallFlush */
}, {
  (MsgHandler) RemoteInvalidReq_MsgInvalidate, /* MsgRInvalidate */
  (MsgHandler) RemoteInvalidReq_MsgInvalidate, /* MsgWInvalidate */
  (MsgHandler) RemoteInvalidReq_MsgSharedAckData,
  (MsgHandler) RemoteInvalidReq_MsgExclusiveAckData,
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) RemoteInvalidReq_MsgModifyAckData,
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteShared =
{
  RemoteShared,
{
  (CallHandler) RemoteShared_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) RemoteShared_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) RemoteShared_CallFlush,
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) RemoteShared_MsgWInvalidate,
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteSharedReq =
{
  RemoteSharedReq,
{
  (CallHandler) call_error,		/* CallStartRead */
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_error,		/* CallFlush */
}, {
  (MsgHandler) RemoteSharedReq_MsgInvalidate, /* MsgRInvalidate */
  (MsgHandler) RemoteSharedReq_MsgInvalidate, /* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) RemoteSharedReq_MsgModifyAck,
  (MsgHandler) RemoteSharedReq_MsgModifyAck, /* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteSharedRip =
{
  RemoteSharedRip,
{
  (CallHandler) RemoteSharedRip_CallStartRead,
  (CallHandler) RemoteSharedRip_CallEndRead,
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) msg_error,		/* MsgRInvalidate */
  (MsgHandler) RemoteSharedRip_MsgWInvalidate,
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteExclusive =
{
  RemoteExclusive,
{
  (CallHandler) RemoteExclusive_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) RemoteExclusive_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) RemoteExclusive_CallFlush,
}, {
  (MsgHandler) RemoteExclusive_MsgInvalidate, /* MsgRInvalidate */
  (MsgHandler) RemoteExclusive_MsgInvalidate, /* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteExclusiveRip =
{
  RemoteExclusiveRip,
{
  (CallHandler) RemoteExclusiveRip_CallStartRead,
  (CallHandler) RemoteExclusiveRip_CallEndRead,
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) RemoteExclusiveRip_MsgRInvalidate,
  (MsgHandler) RemoteExclusiveRip_MsgWInvalidate,
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteModified =
{
  RemoteModified,
{
  (CallHandler) RemoteModified_CallStartRead,
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) RemoteModified_CallStartWrite,
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) RemoteModified_CallFlush,
}, {
  (MsgHandler) RemoteModified_MsgInvalidate, /* MsgRInvalidate */
  (MsgHandler) RemoteModified_MsgInvalidate, /* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteModifiedRip =
{
  RemoteModifiedRip,
{
  (CallHandler) RemoteModifiedRip_CallStartRead,
  (CallHandler) RemoteModifiedRip_CallEndRead,
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) call_error,		/* CallEndWrite */
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) RemoteModifiedRip_MsgRInvalidate,
  (MsgHandler) RemoteModifiedRip_MsgWInvalidate,
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };


State state_RemoteModifiedWip =
{
  RemoteModifiedWip,
{
  (CallHandler) call_error,		/* CallStartRead */
  (CallHandler) call_error,		/* CallEndRead */
  (CallHandler) call_error,		/* CallStartWrite */
  (CallHandler) RemoteModifiedWip_CallEndWrite,
  (CallHandler) call_no_op,		/* CallFlush */
}, {
  (MsgHandler) RemoteModifiedWip_MsgInvalidate, /* MsgRInvalidate */
  (MsgHandler) RemoteModifiedWip_MsgInvalidate, /* MsgWInvalidate */
  (MsgHandler) msg_error,		/* MsgSharedAckData */
  (MsgHandler) msg_error,		/* MsgExclusiveAckData */
  (MsgHandler) msg_error,		/* MsgModifyAck */
  (MsgHandler) msg_error,		/* MsgModifyAckData */
  (MsgHandler) msg_error,		/* MsgInvalidateAck */
  (MsgHandler) msg_error,		/* MsgInvalidateAckData */
  (MsgHandler) msg_no_op,		/* MsgRelease */
  (MsgHandler) msg_error,		/* MsgSharedReq */
  (MsgHandler) msg_error,		/* MsgExclusiveReq */
  (MsgHandler) msg_error,		/* MsgModifyReq */
  (MsgHandler) msg_error,		/* MsgFlush */
  (MsgHandler) msg_error,		/* MsgFlushData */
} };
