/*
 * crl_int.h
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: crl_int.h,v 1.14 1995/08/22 21:27:49 tuna Exp $
 */

 /* This header file contains prototypes, structures, etc. only for use
  * within the CRL implementation code.  There are many #define's for
  * state numbers and many macros.
  */

#ifndef _CRL_INT_H_
#define _CRL_INT_H_

/*#define STATISTICS*/
/*#define CRL_SANITY*/
/*#define CRL_DEBUG*/

#define ProfileNone       (0)
#define ProfileMap        (1)
#define ProfileAll        (2)
#define ALEWIFE_PROFILING (ProfileNone)

#include <stdio.h>
#include <stdlib.h>

#if defined(CM5)
#include <cm/cmmd.h>
#elif defined(ALEWIFE)
#include <parallel.h>
#elif defined(TCPUNIX)
#include "tcp_setup.h"
#endif

#if (ALEWIFE_PROFILING != ProfileNone)
#include <primops.h>
#include <profile.h>
#endif

#if defined(TCPUNIX)
extern char *group;
#endif

#include "crl.h"

#if defined(TCPUNIX)
#include "pvm3.h"
#endif

#define Max(a,b) (((a)>(b))?(a):(b))

#if defined(CM5)
#define assert(x)                                                    \
 do {                                                                \
  if (!(x))                                                          \
    CMMD_error("CRL version %s: failed assertion, node %d, %s:%d\n", \
	       CRL_VERSION, crl_self_addr, __FILE__, __LINE__);      \
 } while (0)
#elif defined(ALEWIFE) || defined(TCPUNIX)
#define assert(x)                                                         \
 do {                                                                     \
  if (!(x)) {                                                             \
    fprintf(stderr, "CRL version %s: failed assertion, node %d, %s:%d\n", \
	    CRL_VERSION, crl_self_addr, __FILE__, __LINE__);              \
    exit(1); }                                                            \
 } while (0)
#endif

#if defined(CRL_SANITY)
#define sanity(x) assert(x)
#else
#define sanity(x)
#endif

#define err_sys(str, arg) err_system(__FILE__, __LINE__, str, arg)

/* assume 32-bit wordsize
 */
#define MaxNodesBits (8)
#define MaxSeqnoBits (32-MaxNodesBits)

#define MaxNodes     (1<<MaxNodesBits)
#define MaxSeqno     (1<<MaxSeqnoBits)

#if defined(TCPUNIX)
extern int node_fd[MaxNodes];    /* node_fd's used to contact each host */
#endif

#define MakeRegionId(node, seqno) (((seqno)<<MaxNodesBits)|(node))
#define RegionIdNode(rgn_id)      ((rgn_id)&((1<<MaxNodesBits)-1))
#define HashRegionId(rgn_id)      (((rgn_id)>>MaxNodesBits)^(rgn_id))

/* call types
 */
#define CallStartRead        (0)
#define CallEndRead          (1)
#define CallStartWrite       (2)
#define CallEndWrite         (3)
#define CallFlush            (4)

/* home-to-remote message types
 */
#define MsgRInvalidate       (0)
#define MsgWInvalidate       (1)
#define MsgSharedAckData     (2)
#define MsgExclusiveAckData  (3)
#define MsgModifyAck         (4)
#define MsgModifyAckData     (5)

/* remote-to-home message types
 */
#define MsgInvalidateAck     (6)
#define MsgInvalidateAckData (7)
#define MsgRelease           (8)
#define MsgSharedReq         (9)
#define MsgExclusiveReq      (10)
#define MsgModifyReq         (11)
#define MsgFlush             (12)
#define MsgFlushData         (13)

#define NumCallTypes         (5)
#define NumMsgTypes          (14)

/* home states
 */
#define HomeExclusive        (0)
#define HomeExclusiveRip     (1)
#define HomeExclusiveWip     (2)
#define HomeShared           (3)
#define HomeSharedRip        (4)
#define HomeIip              (5)
#define HomeIipSpecial       (6)
#define HomeInvalid          (7)

/* remote states
 */
#define RemoteInvalid        (8)
#define RemoteInvalidReq     (9)
#define RemoteShared         (10)
#define RemoteSharedReq      (11)
#define RemoteSharedRip      (12)
#define RemoteExclusive      (13)
#define RemoteExclusiveRip   (14)
#define RemoteModified       (15)
#define RemoteModifiedRip    (16)
#define RemoteModifiedWip    (17)

#define NumHomeStates        (8)
#define NumStates            (18)
#define IsHomeState(s)       ((s) < NumHomeStates)

/* message handler return codes
 */
#define NormalReturn   (0)
#define RetryBlocked   (1)
#define MessageBlocked (2)

/* size of meta-data area, rounded up to the next doubleword
 * (on alewife, add an extra dword to the metadata size to work around
 * a bug in the first-spin CMMU silicon)
 */
#if defined(ALEWIFE)
#define MetaDataSize \
  (((Max(sizeof(HomeRegion), sizeof(RemoteRegion)) + 0x7) & (~0x7)) + 8)
#else
#define MetaDataSize \
  ((Max(sizeof(HomeRegion), sizeof(RemoteRegion)) + 0x7) & (~0x7))
#endif

#define MetaFromUser(rgn) \
  ((Region *) (((char *) rgn) - MetaDataSize))

#define UserFromMeta(rgn) \
  ((void *) (((char *) rgn) + MetaDataSize))

#define CallEvent(type, rgn) \
  (((rgn)->state->calls[type])(rgn))

#define MsgEvent(type, rgn, src, vers) \
  (((rgn)->state->msgs[type])(rgn, src, vers))

/* pointer storage management
 */
#define PtrAlloc(rslt)                              \
  do {                                              \
    (rslt) = ptr_free_list;                         \
    if ((rslt) == NULL)                             \
      (rslt) = (Pointer *) malloc(sizeof(Pointer)); \
    else                                            \
      ptr_free_list = (rslt)->next;                 \
    (rslt)->next = NULL;                            \
  } while (0)

#define PtrFree(p)             \
  do {                         \
    (p)->next = ptr_free_list; \
    ptr_free_list = (p);       \
  } while (0)

#define PtrFreeMany(rgn, last)      \
  do {                              \
    Pointer *first;                 \
    first = &((rgn)->ptrs);         \
    if (first != (last)) {          \
      (last)->next = ptr_free_list; \
      ptr_free_list = first->next;  \
      first->next = NULL;           \
    }                               \
  } while (0)

/* ProtMsgAlloc() should only be used within message
 * handlers or when interrupts are disabled!
 */
#define ProtMsgAlloc(rslt)                          \
  do {                                              \
    (rslt) = prot_msg_free;                         \
    if ((rslt) == NULL)                             \
      (rslt) = (ProtMsg *) malloc(sizeof(ProtMsg)); \
    else                                            \
      prot_msg_free = (rslt)->next;                 \
    (rslt)->next = NULL;                            \
  } while (0)

/* ProtMsgFree() should only be used within message
 * handlers or when interrupts are disabled!
 */
#define ProtMsgFree(m)       \
  do {                       \
    m->next = prot_msg_free; \
    prot_msg_free = m;       \
  } while (0)

#define ProtMsgQueuePeek(q) ((q)->head)

#define ProtMsgQueuePut(q, m) \
  do {                        \
    (m)->next = NULL;         \
    if ((q)->head == NULL)    \
      (q)->head = (m);        \
    else                      \
      (q)->tail->next = (m);  \
    (q)->tail = (m);          \
  } while (0)

#define ProtMsgQueueGet(rslt, q) \
  do {                           \
    (rslt) = (q)->head;          \
    if ((rslt) != NULL) {        \
      (q)->head = (rslt)->next;  \
      (rslt)->next = NULL;       \
    }                            \
  } while (0)

typedef void (*CallHandler)(void *);
typedef int  (*MsgHandler)(void *, unsigned, unsigned);
typedef int  (*TxCont)(void *);

typedef struct state
{
  unsigned    state;			/* state number  */
  CallHandler calls[NumCallTypes];	/* call handlers */
  MsgHandler  msgs[NumMsgTypes];	/* msg handlers  */
} State;

typedef struct pointer
{
  unsigned        node;		/* remote node number */
  void           *addr;		/* remote address     */
  unsigned        vers;		/* version number     */
  struct pointer *next;		/* (list/queue)       */
} Pointer;

typedef struct protmsg
{
  unsigned        type;		/* message type   */
  unsigned        src;		/* sending node   */
  unsigned        vers;		/* version number */
  struct protmsg *next;		/* (list/queue)   */
} ProtMsg;

typedef struct
{
  ProtMsg *head;
  ProtMsg *tail;
} ProtMsgQueue;

#if defined(CRL_DEBUG)
typedef struct
{
  State   *state;
  char    *file;
  unsigned line;
  unsigned extra;
} DebugInfo;
#endif

typedef struct region
{
  State   *state;		/* region state      */
  rid_t    rgn_id;		/* region identifier */
  unsigned vers;		/* version number    */
  unsigned size;		/* region size       */

  unsigned recv_in_progress;	/* hooks for delivery of protocol */
  unsigned recv_type;		/* messages that carry data       */
  unsigned recv_src;
  unsigned recv_vers;

  unsigned tx_done;		/* tx done flag    */
  TxCont   tx_cont;		/* tx continuation */
  unsigned tx_arg1;
  unsigned tx_arg2;		/* extra tx args */
  unsigned tx_arg3;

  unsigned read_cnt;		/* read counter     */
  unsigned map_cnt;		/* mapping counter  */
#if defined(CM5) || defined(TCPUNIX)
  unsigned send_cnt;		/* send counter     */
#endif

  struct region *rtable_link;
  struct region *queue_link;

#if defined(CRL_DEBUG)
  DebugInfo info0;		/* debugging hooks */
  DebugInfo info1;
  DebugInfo info2;
  DebugInfo info3;
#endif
} Region;

typedef struct home_region
{
  Region       core;
  unsigned     num_ptrs;
  Pointer      ptrs;
  ProtMsgQueue blocked_msgs;
} HomeRegion;

typedef struct
{
  Region   core;
  unsigned home;
  void    *home_addr;
  unsigned received_invalidate;
} RemoteRegion;

typedef struct
{
  Region *head;
  Region *tail;
} RgnQueue;

typedef struct
{
  unsigned valid;
  unsigned vers;
  unsigned size;
  void    *home_addr;
} RgnInfo;

typedef struct actmsg
{
  void          *proc;
  unsigned       arg0;
  unsigned       arg1;
  unsigned       arg2;
  unsigned       arg3;
  struct actmsg *next;
} ActMsg;

typedef struct
{
  unsigned enabled;
  ActMsg  *head;
  ActMsg  *tail;
} ActMsgQueue;

/* comm.c
 */
extern ActMsgQueue incoming_queue;
extern ProtMsg    *prot_msg_free;

extern void     init_comm(void);
#if defined(TCPUNIX)
extern void     exit_comm(void);
#endif
extern void     actmsg_enqueue(void *, unsigned, unsigned, unsigned, unsigned);
extern void     actmsg_poll(unsigned);
extern void     retry_blocked_msgs(HomeRegion *);
extern unsigned get_region_info(rid_t, RgnInfo *);

#if defined(CM5)
extern void actmsg_poll_no_intr(unsigned);
extern void rgn_msg_stub(unsigned, Region *, unsigned, unsigned);
extern void rgn_inv_stub(unsigned, rid_t, unsigned, unsigned);
extern void rgn_data_rts(unsigned, Region *, unsigned, unsigned);
extern void fast_data_msg(unsigned, unsigned, Region *, unsigned, Region *);
#elif defined(ALEWIFE)
extern void rgn_msg_stub(void);
extern void rgn_inv_stub(void);
extern void send_data_msg(unsigned, unsigned, Region *, unsigned, Region *);
extern void rgn_data_msg_stub(void);
#elif defined(TCPUNIX)
extern void actmsg_poll_no_intr(unsigned);
extern void rgn_msg_stub(unsigned, Region *, unsigned, unsigned);
extern void rgn_inv_stub(unsigned, rid_t, unsigned, unsigned);
extern void send_data_msg(unsigned, unsigned, Region *, unsigned, Region *);
#endif

/* crl.c
 */
extern void *safe_malloc(unsigned);
extern void  safe_free(void *);

/* proto.c
 */
extern State state_HomeExclusive;
extern State state_HomeExclusiveRip;
extern State state_HomeExclusiveWip;
extern State state_HomeShared;
extern State state_HomeSharedRip;
extern State state_HomeIip;
extern State state_HomeIipSpecial;
extern State state_HomeInvalid;
extern State state_RemoteInvalid;
extern State state_RemoteInvalidReq;
extern State state_RemoteShared;
extern State state_RemoteSharedRlock;
extern State state_RemoteSharedReq;
extern State state_RemoteSharedRip;
extern State state_RemoteExclusive;
extern State state_RemoteExclusiveWlock;
extern State state_RemoteExclusiveRip;
extern State state_RemoteModified;
extern State state_RemoteModifiedRip;
extern State state_RemoteModifiedWip;

/* ptr.c
 */
extern Pointer *ptr_free_list;
extern Pointer *ptr_insert(HomeRegion *, unsigned, Region *, unsigned);
extern void     ptr_delete(HomeRegion *rgn, unsigned, unsigned);

/* rtable.c
 */
extern void    init_rtable(void);
#if defined(TCPUNIX)
extern void    exit_rtable(void);
#endif
extern rid_t   allocate_rgn_id(void);
extern Region *rtable_lookup(rid_t);
extern void    rtable_insert(Region *);
extern void    rtable_delete(Region *);

/* sync.c
 */
extern void init_sync(void);
#if defined(TCPUNIX)
extern void exit_sync(void);
#endif

/* urc.c
 */
extern void urc_insert(Region *);
extern void urc_delete(Region *);

/* stats.c
 */
#if defined(STATISTICS)

#define StatCntrCreate               (0)
#define StatCntrDestroy              (1)
#define StatCntrMap                  (2)
#define StatCntrMapRemote            (3)
#define StatCntrMapMiss              (4)
#define StatCntrUnmap                (5)
#define StatCntrStartRead            (6)
#define StatCntrStartReadRemote      (7)
#define StatCntrStartReadMiss        (8)
#define StatCntrStartReadBlock       (9)
#define StatCntrEndRead              (10)
#define StatCntrStartWrite           (11)
#define StatCntrStartWriteRemote     (12)
#define StatCntrStartWriteMiss       (13)
#define StatCntrStartWriteBlock      (14)
#define StatCntrEndWrite             (15)
#define StatCntrFlush                (16)
#define StatCntrMsgRgnInfoReq        (17)
#define StatCntrMsgRgnInfoAck        (18)
#define StatCntrProtoMsg             (19)
#define StatCntrProtoMsgQueued       (20)
#define StatCntrMsgSharedReq         (21)
#define StatCntrMsgSharedAckData     (22)
#define StatCntrMsgExclusiveReq      (23)
#define StatCntrMsgExclusiveAckData  (24)
#define StatCntrMsgModifyReq         (25)
#define StatCntrMsgModifyAck         (26)
#define StatCntrMsgModifyAckData     (27)
#define StatCntrMsgRInvalidate       (28)
#define StatCntrMsgWInvalidate       (29)
#define StatCntrMsgInvalidateAck     (30)
#define StatCntrMsgInvalidateAckData (31)
#define StatCntrMsgRelease           (32)
#define StatCntrMsgReleaseData       (33)
#define StatCntrMsgFlush             (34)
#define StatCntrMsgFlushData         (35)
#define StatCntrRtableLookup         (36)
#define StatCntrRtableLookupLoop     (37)
#define NumStatCntrs                 (38)

extern unsigned crl_stat_cntrs[NumStatCntrs];

#endif

#endif
