/*
 * crl.c
 * kirk johnson
 * january 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: crl.c,v 1.10 1995/08/22 21:27:49 tuna Exp $
 */

 /*
  * This file contains functions for use by applications.  All the functions
  * listed in the user documentation and crl.h are contained here.  For
  * example, all the region functions and global synchronization functions
  * are defined here.
  */

#include "crl_int.h"

static Region *allocate_remote_region(rid_t);

static int crl_started = 0;

unsigned crl_self_addr;
unsigned crl_num_nodes;
#if defined(TCPUNIX)
char *group;
#endif


#if defined(TCPUNIX)
void crl_init(char *group_name)
#else
void crl_init(void)
#endif
{
  if (crl_started == 1)
  {
    fprintf(stderr, "CRL has already been initialized!\n");
    exit(1);
  }

  crl_started = 1;

#if defined(TCPUNIX)
  group = (char *) malloc(sizeof(char) * (strlen(group_name) + 1));
  assert(group != NULL);
  strcpy(group, group_name);
#endif

#if defined(CM5)
  crl_self_addr = CMMD_self_address();
  crl_num_nodes = CMMD_partition_size();
#elif defined(ALEWIFE)
  crl_self_addr = MY_PID;
  crl_num_nodes = NPROCS;
#elif defined(TCPUNIX)
  tcp_setup();
#endif

  assert(crl_num_nodes <= MaxNodes);

  init_rtable();
  init_comm();
  init_sync();

#if defined(STATISTICS)
  crl_stats_reset();
#endif

  rgn_barrier();
}

#if defined(TCPUNIX)
void crl_exit(void)
{
  int retval;

  shuttingdown = 1;

  /* After every node hits this barrier, then all messages should have
   * been sent/received, so we can close connections and remove signals,
   * etc.
   */
  rgn_barrier();

  /* Turn off signal handling */
  signal(SIGIO, SIG_DFL);

  retval = pvm_barrier(group, crl_num_nodes);
  if (retval < 0)
  {
    fprintf(stderr, "%s:%d pvm_barrier returned %d\n",
	    __FILE__, __LINE__, retval);
    exit(1);
  }

  exit_rtable();
  exit_comm();
  exit_sync();
  tcp_shutdown();
  free(group);
  crl_started = 0;
  shuttingdown = 0;
}
#endif

rid_t rgn_create(unsigned size)
{
  rid_t       rgn_id;
  HomeRegion *rgn;

  rgn_id = allocate_rgn_id();

  /* round allocated size of user data area up to the next doubleword
   */
  rgn = (HomeRegion *) safe_malloc(MetaDataSize + ((size + 0x7) & (~0x7)));
  assert(rgn != NULL);
  bzero((char *) rgn, MetaDataSize);

  rgn->core.state  = &state_HomeExclusive;
  rgn->core.rgn_id = rgn_id;
  rgn->core.vers   = 0;
  rgn->core.size   = size;

  rgn->blocked_msgs.head = NULL;
  rgn->blocked_msgs.tail = NULL;

  rtable_insert((Region *) rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrCreate] += 1;
#endif

  return rgn_id;
}


void rgn_destroy(rid_t rgn_id)
{
  /* nothing yet */

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrDestroy] += 1;
#endif
}


#if (ALEWIFE_PROFILING != ProfileNone)
static void *_rgn_map(rid_t rgn_id)
#else
void *rgn_map(rid_t rgn_id)
#endif
{
  Region *rgn;
  void   *rslt;

  rgn = rtable_lookup(rgn_id);

  if (rgn == NULL)
  {
    rgn = allocate_remote_region(rgn_id);
    rtable_insert(rgn);

#if defined(STATISTICS)
    crl_stat_cntrs[StatCntrMapMiss] += 1;
#endif
  }
  else
  {
    if ((rgn->map_cnt == 0) &&
	(!IsHomeState(rgn->state->state)))
      urc_delete(rgn);
  }

  rgn->map_cnt += 1;

#if defined(STATISTICS)
  if (!IsHomeState(rgn->state->state))
    crl_stat_cntrs[StatCntrMapRemote] += 1;
  crl_stat_cntrs[StatCntrMap] += 1;
#endif

  rslt = UserFromMeta(rgn);

  return rslt;
}


#if (ALEWIFE_PROFILING != ProfileNone)

void *rgn_map(rid_t rgn_id)
{
  void *rslt;

  start_statcount(0);
  rslt = _rgn_map(rgn_id);
  stop_statcount(0);

  return rslt;
}

#endif


#if (ALEWIFE_PROFILING != ProfileNone)
static void _rgn_unmap(void *x)
#else
void rgn_unmap(void *x)
#endif
{
  Region  *rgn;
  unsigned cnt;

  rgn = MetaFromUser(x);

  cnt = (rgn->map_cnt -= 1);
  if ((cnt == 0) &&
      (!IsHomeState(rgn->state->state)))
    urc_insert(rgn);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrUnmap] += 1;
#endif
}


#if (ALEWIFE_PROFILING != ProfileNone)

void rgn_unmap(void *x)
{
  start_statcount(0);
  _rgn_unmap(x);
  stop_statcount(0);
}

#endif


rid_t rgn_rid(void *x)
{
  return MetaFromUser(x)->rgn_id;
}


unsigned rgn_size(void *x)
{
  return MetaFromUser(x)->size;
}


#if (ALEWIFE_PROFILING == ProfileAll)
static void _rgn_start_read(void *x)
#else
void rgn_start_read(void *x)
#endif
{
#if defined(ALEWIFE)
  user_atomic_request();
#else  
  incoming_queue.enabled = 1;
#endif
  CallEvent(CallStartRead, MetaFromUser(x));

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartRead] += 1;
#endif
}


#if (ALEWIFE_PROFILING == ProfileAll)

void rgn_start_read(void *x)
{
  start_statcount(0);
  _rgn_start_read(x);
  stop_statcount(0);
}

#endif


#if (ALEWIFE_PROFILING == ProfileAll)
static void _rgn_end_read(void *x)
#else
void rgn_end_read(void *x)
#endif
{
#if defined(ALEWIFE)
  user_atomic_request();
#else
  incoming_queue.enabled = 1;
#endif
  CallEvent(CallEndRead, MetaFromUser(x));

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrEndRead] += 1;
#endif
}


#if (ALEWIFE_PROFILING == ProfileAll)

void rgn_end_read(void *x)
{
  start_statcount(0);
  _rgn_end_read(x);
  stop_statcount(0);
}

#endif


#if (ALEWIFE_PROFILING == ProfileAll)
static void _rgn_start_write(void *x)
#else
void rgn_start_write(void *x)
#endif
{
#if defined(ALEWIFE)
  user_atomic_request();
#else
  incoming_queue.enabled = 1;
#endif
  CallEvent(CallStartWrite, MetaFromUser(x));

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrStartWrite] += 1;
#endif
}


#if (ALEWIFE_PROFILING == ProfileAll)

void rgn_start_write(void *x)
{
  start_statcount(0);
  _rgn_start_write(x);
  stop_statcount(0);
}

#endif


#if (ALEWIFE_PROFILING == ProfileAll)
static void _rgn_end_write(void *x)
#else
void rgn_end_write(void *x)
#endif
{
#if defined(ALEWIFE)
  user_atomic_request();
#else
  incoming_queue.enabled = 1;
#endif
  CallEvent(CallEndWrite, MetaFromUser(x));

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrEndWrite] += 1;
#endif
}


#if (ALEWIFE_PROFILING == ProfileAll)

void rgn_end_write(void *x)
{
  start_statcount(0);
  _rgn_end_write(x);
  stop_statcount(0);
}

#endif


#if (ALEWIFE_PROFILING == ProfileAll)
static void _rgn_flush(void *x)
#else
void rgn_flush(void *x)
#endif
{
  incoming_queue.enabled = 1;
  CallEvent(CallFlush, MetaFromUser(x));

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrFlush] += 1;
#endif
}


#if (ALEWIFE_PROFILING == ProfileAll)

void rgn_flush(void *x)
{
  unsigned _mask;

  /* because rgn_flush() from urc_insert() [which could have been
   * called from rgn_unmap(), we need to be a bit more careful here
   * about enabling/disabling the statistics counter
   */
  _mask = CReg->StatArray[0].Mask;
  CReg->StatArray[0].Mask = ENABLE_STATCNT_MASK;

  _rgn_flush(x);

  CReg->StatArray[0].Mask = _mask;
}

#endif


void *safe_malloc(unsigned nbytes)
{
  void *rslt;

#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  rslt = (void *) malloc(nbytes);

#if defined(CM5)
  if (reenable) CMAML_enable_interrupts();
#elif defined(TCPUNIX)
  crl_sigsetmask(oldmask);
#endif

  return rslt;
}


void safe_free(void *ptr)
{
#if defined(CM5)
  int reenable;
  reenable = CMAML_disable_interrupts();
#elif defined(TCPUNIX)
  int oldmask;
  oldmask = crl_sigblock(sigmask(SIGIO));
#endif

  free(ptr);

#if defined(CM5)
  if (reenable) CMAML_enable_interrupts();
#elif defined(TCPUNIX)
  crl_sigsetmask(oldmask);
#endif
}


static Region *allocate_remote_region(rid_t rgn_id)
{
  unsigned      home;
  unsigned      size;
  RgnInfo       info;
  RemoteRegion *rgn;

  home = get_region_info(rgn_id, &info);
  sanity(home != crl_self_addr);
  size = info.size;

  /* round allocated size of user data area up to the next doubleword
   */
  rgn = (RemoteRegion *) safe_malloc(MetaDataSize + ((size + 0x7) & (~0x7)));
  assert(rgn != NULL);
  bzero((char *) rgn, MetaDataSize);

  rgn->core.state  = &state_RemoteInvalid;
  rgn->core.rgn_id = rgn_id;
  rgn->core.vers   = info.vers;
  rgn->core.size   = size;

  rgn->home      = home;
  rgn->home_addr = info.home_addr;

  return (Region *) rgn;
}
