/*
 * cm5_comm.c
 * january 1995
 * kirk johnson
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: cm5_comm.c,v 1.4 1995/08/22 21:27:49 tuna Exp $
 */

#include "crl_int.h"

#define QueueChunkSize    (16)
#define NumReservedRports (32)

ActMsgQueue incoming_queue;
ProtMsg    *prot_msg_free;

static int      num_available_rports;
static int      available_rports[NumReservedRports];
static RgnQueue rport_queue;

static void    rgn_recv_data_done(int, Region *);
static void    rgn_send_data(unsigned, unsigned, int);
static void    rgn_send_data_done(Region *);
static int     rport_queue_empty(void);
static void    rport_queue_put(Region *);
static Region *rport_queue_get(void);


void init_comm(void)
{
  int     i;
  int     rport;
  ActMsg *new;

  /* allocate message buffers
   */
  new = (ActMsg *) safe_malloc(sizeof(ActMsg) * QueueChunkSize);
  assert(new != NULL);

  /* link them together 
   */
  for (i=0; i<(QueueChunkSize-1); i++)
    new[i].next = &(new[i+1]);
  new[i].next = &(new[0]);

  /* initialize incoming_queue
   */
  incoming_queue.enabled = 0;
  incoming_queue.head    = new;
  incoming_queue.tail    = new;

  /* protocol message free list is initially empty
   */
  prot_msg_free = NULL;

  /* allocate rports
   */
  for (i=0; i<NumReservedRports; i++)
  {
    rport = CMAML_allocate_rport();
    sanity(rport != -1);
    available_rports[i] = rport;
  }
  num_available_rports = NumReservedRports;

  /* rport queue is initially empty
   */
  rport_queue.head = NULL;
  rport_queue.tail = NULL;
}


void actmsg_enqueue(void *proc, unsigned a0, unsigned a1,
		    unsigned a2, unsigned a3)
{
  int          i;
  ActMsgQueue *q;
  ActMsg      *m;
  ActMsg      *next;

  /* actmsg_enqueue() should only be called
   * when interrupts are disabled
   */
  sanity(CMAML_interrupt_status() == FALSE);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsgQueued] += 1;
#endif

  q = &incoming_queue;
  m = q->tail;

  next = m->next;
  if (next == q->head)
  {
    /* allocate more message buffers
     */
    next = (ActMsg *) malloc(sizeof(ActMsg) * QueueChunkSize);
    assert(next != NULL);

    /* link them in 
     */
    m->next = next;
    for (i=0; i<(QueueChunkSize-1); i++)
      next[i].next = &(next[i+1]);
    next[i].next = q->head;
  }
  q->tail = next;

  m->proc = proc;
  m->arg0 = a0;
  m->arg1 = a1;
  m->arg2 = a2;
  m->arg3 = a3;
}


/* drain the incoming message queue; conditionally
 * disable message queueing when done
 */
void actmsg_poll(unsigned new_enable)
{
  int         reenable;
  int         enabled;
  ActMsgQueue *q;
  ActMsg      *m;
  void       (*f)(unsigned, unsigned, unsigned, unsigned);

  reenable = 0;
  enabled  = CMAML_interrupt_status();

  q = &incoming_queue;

  while (1)
  {
    /* walk through the message queue invoking handlers
     */
    m = q->head;

    if (m != q->tail)
    {
      if (enabled)
      {
	reenable = CMAML_disable_interrupts();
	enabled  = 0;
      }

      do
      {
	sanity(q->enabled);
	f = m->proc;
	f(m->arg0, m->arg1, m->arg2, m->arg3);
	m = m->next;
	q->head = m;
      } while (m != q->tail);
    }

    /* turn message queueing off. if (m == q->tail) is still true, no
     * messages arrived and queued themselves sneakily between the
     * time we exited the inner loop and when message queuing got
     * turned off, so we can safely exit the outer loop (note clever
     * [hah!] use of volatile to make sure the compiled code does the
     * right thing) 
     */
    *((volatile unsigned *) &(q->enabled)) = new_enable;
    if (m == *((ActMsg * volatile *) &(q->tail)))
      break;

    /* no such luck, so lather, rinse, repeat
     */
    q->enabled = 1;
  }

  if (reenable) CMAML_enable_interrupts();
}


/* like actmsg_poll(), but assumes interrupts are already disabled
 */
void actmsg_poll_no_intr(unsigned new_enable)
{
  ActMsgQueue *q;
  ActMsg      *m;
  void       (*f)(unsigned, unsigned, unsigned, unsigned);

  sanity(CMAML_interrupt_status() == FALSE);

  q = &incoming_queue;

  /* walk through the message queue invoking handlers
   */
  m = q->head;

  while (m != q->tail)
  {
    sanity(q->enabled);
    f = m->proc;
    f(m->arg0, m->arg1, m->arg2, m->arg3);
    m = m->next;
    q->head = m;
  }

  q->enabled = new_enable;
  sanity(q->head == q->tail);
}



void retry_blocked_msgs(HomeRegion *rgn)
{
  ProtMsgQueue *q;
  ProtMsg      *m;
  ProtMsg      *m2;

  /* since retry_blocked_msgs() calls protocol message handlers, make sure
   * interrupts are disabled and the incoming message queue is enabled
   */
  sanity(CMAML_interrupt_status() == FALSE);
  sanity(incoming_queue.enabled);

  q = &(rgn->blocked_msgs);
  sanity(ProtMsgQueuePeek(q) != NULL);

  while (1)
  {
    m = ProtMsgQueuePeek(q);

    /* stop if there aren't any more blocked messages
     */
    if (m == NULL) break;

    /* stop if the first message is still blocked
     */
    if (MsgEvent(m->type, (Region *) rgn, m->src, m->vers) == MessageBlocked)
      break;

    /* message was successfully processed, so remove it from the queue,
     * double check that it's the same message we got when we peeked,
     * and free it
     */
    ProtMsgQueueGet(m2, q);
    sanity(m == m2);
    ProtMsgFree(m2);
  }
}


static void rgn_msg(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
  int      rtn;
  ProtMsg *msg;

  /* since rgn_msgs() calls protocol message handlers, make sure interrupts
   * are disabled and the incoming message queue is enabled
   */
  sanity(CMAML_interrupt_status() == FALSE);
  sanity(incoming_queue.enabled);

  rtn = MsgEvent(type, rgn, src, vers);

  if (rtn == MessageBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    ProtMsgAlloc(msg);
    msg->type = type;
    msg->src  = src;
    msg->vers = vers;
    ProtMsgQueuePut(&((HomeRegion *) rgn)->blocked_msgs, msg);
  }
  else if (rtn == RetryBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    if (ProtMsgQueuePeek(&(((HomeRegion *) rgn)->blocked_msgs)) != NULL)
      retry_blocked_msgs((HomeRegion *) rgn);
  }
}


static void rgn_inv(unsigned type, rid_t rgn_id, unsigned src, unsigned vers)
{
  Region *rgn;

  rgn = rtable_lookup(rgn_id);

  /* if rtable_lookup() returns NULL, the region in question has been
   * flushed and removed from the rtable; this invalidate message must
   * have been sent from the home node before the flush message
   * arrived there, so it can be safely ignored.
   */
  if (rgn != NULL)
    rgn_msg(type, rgn, src, vers);
}


void rgn_msg_stub(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    actmsg_enqueue(rgn_msg, type, (unsigned) rgn, src, vers);
  }
  else
  {
    incoming_queue.enabled = 1;
    rgn_msg(type, rgn, src, vers);
    actmsg_poll_no_intr(0);
  }
}


void rgn_inv_stub(unsigned type, rid_t rgn_id, unsigned src, unsigned vers)
{
#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    actmsg_enqueue(rgn_inv, type, rgn_id, src, vers);
  }
  else
  {
    incoming_queue.enabled = 1;
    rgn_inv(type, rgn_id, src, vers);
    actmsg_poll_no_intr(0);
  }
}


void rgn_data_rts(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
  int      idx;
  int      rport;
  unsigned size;
  void    *data;

  /* save handler information
   */
  sanity(rgn->recv_in_progress == 0);
  rgn->recv_in_progress = 1;
  rgn->recv_type = type;
  rgn->recv_src  = src;
  rgn->recv_vers = vers;

  if (num_available_rports == 0)
  {
    /* queue rport request
     */
    rport_queue_put(rgn);
  }
  else
  {
    /* grab an rport
     */
    idx = num_available_rports - 1;
    num_available_rports = idx; 
    rport = available_rports[idx];

    /* prepare rport
     */
    size = rgn->size;
    data = UserFromMeta(rgn);

    CMAML_set_rport_addr         (rport, data);
    CMAML_set_rport_byte_counter (rport, size);
    CMAML_set_rport_total_nbytes (rport, size);
    CMAML_set_rport_handler      (rport, (void *) rgn_recv_data_done);
    CMAML_set_rport_handler_arg2 (rport, (void *) rgn);

    /* notify sender
     */
    sanity(src < crl_num_nodes);
    sanity(CMAML_interrupt_status() == FALSE);
    CMAML_rpc(src, rgn_send_data,
	      rgn->rgn_id, crl_self_addr, rport);
  }
}


static void rgn_recv_data_done(int rport, Region *rgn)
{
  unsigned type;
  unsigned src;
  unsigned vers;
  int      idx;
  unsigned size;
  void    *data;

  /* invoke handler
   */
  sanity(rgn->recv_in_progress == 1);
  type = rgn->recv_type;
  src  = rgn->recv_src;
  vers = rgn->recv_vers;
  rgn->recv_in_progress = 0;
  rgn_msg_stub(type, rgn, src, vers);

  if (rport_queue_empty())
  {
    /* free rport
     */
    idx = num_available_rports;
    num_available_rports = idx + 1;
    available_rports[idx] = rport;
  }
  else
  {
    /* pop region from the queue of those waiting for rports
     */
    rgn = rport_queue_get();
    sanity(rgn != NULL);
    sanity(num_available_rports == 0);

    /* prepare rport
     */
    size = rgn->size;
    data = UserFromMeta(rgn);

    CMAML_set_rport_addr         (rport, data);
    CMAML_set_rport_byte_counter (rport, size);
    CMAML_set_rport_total_nbytes (rport, size);
    CMAML_set_rport_handler      (rport, (void *) rgn_recv_data_done);
    CMAML_set_rport_handler_arg2 (rport, (void *) rgn);

    /* notify sender
     */
    sanity(rgn->recv_src < crl_num_nodes);
    sanity(CMAML_interrupt_status() == FALSE);
    CMAML_rpc(rgn->recv_src, rgn_send_data,
	      rgn->rgn_id, crl_self_addr, rport);
  }
}


static void rgn_send_data(unsigned rgn_id, unsigned dst, int rport)
{
  Region  *rgn;
  unsigned size;
  void    *data;
  int      rtn;

  rgn = rtable_lookup(rgn_id);
  sanity(rgn != NULL);
  sanity(rgn->send_cnt > 0);

  size = rgn->size;
  data = UserFromMeta(rgn);

  sanity(dst < crl_num_nodes);
  rtn = CMAML_scopy(dst, rport, 0, 0, data, size,
		    rgn_send_data_done, rgn);
  sanity(rtn != -1);
}


static void rgn_send_data_done(Region *rgn)
{
  rgn->send_cnt -= 1;
}


static int rport_queue_empty(void)
{
  return (rport_queue.head == NULL);
}


static void rport_queue_put(Region *rgn)
{
  RgnQueue *q;

  sanity(rgn->queue_link == NULL);

  q = &rport_queue;

  if (q->head == NULL)
    q->head = rgn;
  else
    q->tail->queue_link = (void *) rgn;

  q->tail = rgn;
}


static Region *rport_queue_get(void)
{
  RgnQueue *q;
  Region   *rslt;

  q = &rport_queue;
  rslt = q->head;

  if (rslt != NULL)
  {
    q->head          = rslt->queue_link;
    rslt->queue_link = NULL;
  }

  return rslt;
}


static void rgn_info_ack(unsigned vers, unsigned size, void *addr, RgnInfo *rtn)
{
  rtn->vers      = vers - 1;
  rtn->size      = size;
  rtn->home_addr = addr;
  rtn->valid     = 1;
}


static void rgn_info_req(unsigned src, rid_t rgn_id, RgnInfo *rtn)
{
  Region *rgn;

  rgn = rtable_lookup(rgn_id);
  sanity(rgn != NULL);

  sanity(src < crl_num_nodes);
  CMAML_reply(src, rgn_info_ack, rgn->vers, rgn->size, rgn, rtn); 

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRgnInfoAck] += 1;
#endif
}


unsigned get_region_info(rid_t rgn_id, RgnInfo *info)
{
  int      reenable;
  unsigned home;
  unsigned self;

  /* figure out where home node is
   */
  home = RegionIdNode(rgn_id);
  sanity(home < crl_num_nodes);

  /* this procedure should only be called with region ids for remote
   * regions, so crash and burn if the home is the local node
   */
  self = crl_self_addr;
  sanity(home != self);

  /* query home node for info
   */
  info->valid = 0;
  reenable = CMAML_disable_interrupts();
  CMAML_request(home, rgn_info_req, self, rgn_id, info);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRgnInfoReq] += 1;
#endif

  CMAML_poll_while_lt(&(info->valid), 1);
  if (reenable) CMAML_enable_interrupts();

  return home;
}
