/*
 * alw_comm.c
 * january 1995
 * kirk johnson
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: alw_comm.c,v 1.4 1995/08/22 21:27:49 tuna Exp $
 */

#include <primops.h>
#include "crl_int.h"

#define FLUSH_MSG_BUFS
#define USE_GLOBAL_TRANSITION

#define QueueChunkSize (16)

#if (ALEWIFE_PROFILING != ProfileNone)

#define ProfMsgPrologue_enable()                \
 unsigned _mask;                                \
 _mask = CReg->StatArray[0].Mask;               \
 CReg->StatArray[0].Mask = ENABLE_STATCNT_MASK; \
 _msg_intr[(_mask == 0) ? 0 : 1] += 1

#define ProfMsgPrologue_disable()               \
 unsigned _mask;                                \
 _mask = CReg->StatArray[0].Mask;               \
 CReg->StatArray[0].Mask = DISABLE_STATCNT_MASK

#define ProfMsgEpilogue() \
 CReg->StatArray[0].Mask = _mask

unsigned _msg_intr[2];

#endif

ActMsgQueue incoming_queue;
ProtMsg    *prot_msg_free;


void init_comm(void)
{
  int     i;
  ActMsg *new;

  /* allocate message buffers
   */
  new = (ActMsg *) safe_malloc(sizeof(ActMsg) * QueueChunkSize);
  assert(new != NULL);

  /* link them together 
   */
  for (i=0; i<(QueueChunkSize-1); i++)
    new[i].next = &(new[i+1]);
  new[i].next = &(new[0]);

  /* initialize incoming_queue
   */
  incoming_queue.enabled = 0;
  incoming_queue.head    = new;
  incoming_queue.tail    = new;

  /* protocol message free list is initially empty
   */
  prot_msg_free = NULL;
}


void actmsg_enqueue(void *proc, unsigned a0, unsigned a1,
		    unsigned a2, unsigned a3)
{
  int          i;
  ActMsgQueue *q;
  ActMsg      *m;
  ActMsg      *next;

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsgQueued] += 1;
#endif

  q = &incoming_queue;
  m = q->tail;

  next = m->next;
  if (next == q->head)
  {
    /* allocate more message buffers
     */
    next = (ActMsg *) malloc(sizeof(ActMsg) * QueueChunkSize);
    assert(next != NULL);

    /* link them in 
     */
    m->next = next;
    for (i=0; i<(QueueChunkSize-1); i++)
      next[i].next = &(next[i+1]);
    next[i].next = q->head;
  }
  q->tail = next;

  m->proc = proc;
  m->arg0 = a0;
  m->arg1 = a1;
  m->arg2 = a2;
  m->arg3 = a3;
}


/* drain the incoming message queue; conditionally
 * disable message queueing when done
 */
void actmsg_poll(unsigned new_enable)
{
  ActMsgQueue *q;
  ActMsg      *m;
  void       (*f)(unsigned, unsigned, unsigned, unsigned);

  q = &incoming_queue;

  while (1)
  {
    /* walk through the message queue invoking handlers
     */
    m = q->head;

    while (m != q->tail)
    {
      sanity(q->enabled);
      f = m->proc;
      f(m->arg0, m->arg1, m->arg2, m->arg3);
      m = m->next;
      q->head = m;
    }

    /* disable message delivery and check if the queue is still
     * empty. if so, break out of the loop, set q-enabled to
     * new_enable, reenable message delivery, and return. if not,
     * reenable message deliver and go around the big while loop
     * again.
     */
    user_atomic_request();
    if (m == *((ActMsg * volatile *) &(q->tail)))
      break;
    user_atomic_release();
  }

  q->enabled = new_enable;
  user_atomic_release();
}


void retry_blocked_msgs(HomeRegion *rgn)
{
  ProtMsgQueue *q;
  ProtMsg      *m;
  ProtMsg      *m2;

  /* since retry_blocked_msgs() calls protocol message handlers, make sure
   * the incoming message queue is enabled
   */
#if defined(USE_GLOBAL_TRANSITION)
  sanity(incoming_queue.enabled);
#endif

  q = &(rgn->blocked_msgs);
  sanity(ProtMsgQueuePeek(q) != NULL);

  while (1)
  {
    m = ProtMsgQueuePeek(q);

    /* stop if there aren't any more blocked messages
     */
    if (m == NULL) break;

    /* stop if the first message is still blocked
     */
    if (MsgEvent(m->type, (Region *) rgn, m->src, m->vers) == MessageBlocked)
      break;

    /* message was successfully processed, so remove it from the queue,
     * double check that it's the same message we got when we peeked,
     * and free it
     */
    ProtMsgQueueGet(m2, q);
    sanity(m == m2);
    ProtMsgFree(m2);
  }
}


static void rgn_msg(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
  int      rtn;
  ProtMsg *msg;

  /* since rgn_msg() calls protocol message handlers, make sure the
   * incoming message queue is enabled
   */
#if defined(USE_GLOBAL_TRANSITION)
  sanity(incoming_queue.enabled);
#endif

  rtn = MsgEvent(type, rgn, src, vers);

  if (rtn == MessageBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    ProtMsgAlloc(msg);
    msg->type = type;
    msg->src  = src;
    msg->vers = vers;
    ProtMsgQueuePut(&((HomeRegion *) rgn)->blocked_msgs, msg);
  }
  else if (rtn == RetryBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    if (ProtMsgQueuePeek(&(((HomeRegion *) rgn)->blocked_msgs)) != NULL)
      retry_blocked_msgs((HomeRegion *) rgn);
  }
}


static void rgn_inv(unsigned type, rid_t rgn_id, unsigned src, unsigned vers)
{
  Region *rgn;

  rgn = rtable_lookup(rgn_id);

  /* if rtable_lookup() returns NULL, the region in question has been
   * flushed and removed from the rtable; this invalidate message must
   * have been sent from the home node before the flush message
   * arrived there, so it can be safely ignored.
   */
  if (rgn != NULL)
    rgn_msg(type, rgn, src, vers);
}


void rgn_msg_stub(void)
{
  unsigned type;
  unsigned src;
  unsigned vers;
  Region  *rgn;

#if (ALEWIFE_PROFILING == ProfileAll)
  ProfMsgPrologue_enable();
#elif (ALEWIFE_PROFILING == ProfileMap)
  ProfMsgPrologue_disable();
#endif

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    /* extract message args
     */
    type = (unsigned) ipi_in_reg(2);
    rgn  = (Region *) ipi_in_reg(3);
    src  = (unsigned) ipi_in_reg(4);
    vers = (unsigned) ipi_in_reg(5);
    ipicst(-1, -1);

    actmsg_enqueue(rgn_msg, type, (unsigned) rgn, src, vers);
  }
  else
  {
    /* extract message args
     */
    type = (unsigned) ipi_in_reg(2);
    rgn  = (Region *) ipi_in_reg(3);
    src  = (unsigned) ipi_in_reg(4);
    vers = (unsigned) ipi_in_reg(5);
    ipicst(-1, -1);

#if defined(USE_GLOBAL_TRANSITION)
    incoming_queue.enabled = 1;
    user_active_global();
#endif

    rgn_msg(type, rgn, src, vers);

#if defined(USE_GLOBAL_TRANSITION)
    actmsg_poll(0);
#endif
  }

#if (ALEWIFE_PROFILING != ProfileNone)
  ProfMsgEpilogue();
#endif
}


void rgn_inv_stub(void)
{
  unsigned type;
  unsigned src;
  unsigned vers;
  rid_t    rgn_id;

#if (ALEWIFE_PROFILING == ProfileAll)
  ProfMsgPrologue_enable();
#elif (ALEWIFE_PROFILING == ProfileMap)
  ProfMsgPrologue_disable();
#endif

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    /* extract message args
     */
    type   = (unsigned) ipi_in_reg(2);
    rgn_id = (rid_t)    ipi_in_reg(3);
    src    = (unsigned) ipi_in_reg(4);
    vers   = (unsigned) ipi_in_reg(5);
    ipicst(-1, -1);

    actmsg_enqueue(rgn_inv, type, rgn_id, src, vers);
  }
  else
  {
    /* extract message args
     */
    type   = (unsigned) ipi_in_reg(2);
    rgn_id = (rid_t)    ipi_in_reg(3);
    src    = (unsigned) ipi_in_reg(4);
    vers   = (unsigned) ipi_in_reg(5);
    ipicst(-1, -1);

#if defined(USE_GLOBAL_TRANSITION)
    incoming_queue.enabled = 1;
    user_active_global();
#endif

    rgn_inv(type, rgn_id, src, vers);

#if defined(USE_GLOBAL_TRANSITION)
    actmsg_poll(0);
#endif
  }

#if (ALEWIFE_PROFILING != ProfileNone)
  ProfMsgEpilogue();
#endif
}


void send_data_msg(unsigned node, unsigned type, Region *dst,
		   unsigned vers, Region *src)
{
  int      i;
  unsigned ndwords;
  void    *src_buf;
  void    *dst_buf;

  ndwords = (src->size + 7) >> 3;
  src_buf = UserFromMeta(src);
  dst_buf = UserFromMeta(dst);

  /* flush send buffer from memory system
   */
#if defined(FLUSH_MSG_BUFS)
  for (i=((ndwords<<3)-8); i>0; i-=16)
    hardflush2(src_buf, i);
  hardflush(src_buf);
#endif

  user_do_on_dmanofix(node, rgn_data_msg_stub,
		      type, dst, crl_self_addr, vers,
		      src_buf, ndwords);
}


static void rgn_data_msg(unsigned type, Region *rgn, unsigned src, unsigned vers)
{
  int      rtn;
  ProtMsg *msg;

  /* since rgn_data_msg() calls protocol message handlers, make sure
   * the incoming message queue is enabled
   */
#if defined(USE_GLOBAL_TRANSITION)
  sanity(incoming_queue.enabled);
#endif

  /* don't call protocol message handler until data storeback is done
   */
  wait_for_storeback();

  rtn = MsgEvent(type, rgn, src, vers);

  if (rtn == MessageBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    ProtMsgAlloc(msg);
    msg->type = type;
    msg->src  = src;
    msg->vers = vers;
    ProtMsgQueuePut(&((HomeRegion *) rgn)->blocked_msgs, msg);
  }
  else if (rtn == RetryBlocked)
  {
    sanity(IsHomeState(rgn->state->state));

    if (ProtMsgQueuePeek(&(((HomeRegion *) rgn)->blocked_msgs)) != NULL)
      retry_blocked_msgs((HomeRegion *) rgn);
  }
}


void rgn_data_msg_stub(void)
{
  int      i;
  unsigned type;
  unsigned src;
  unsigned vers;
  unsigned ndwords;
  Region  *rgn;
  void    *dst_buf;

#if (ALEWIFE_PROFILING == ProfileAll)
  ProfMsgPrologue_enable();
#elif (ALEWIFE_PROFILING == ProfileMap)
  ProfMsgPrologue_disable();
#endif

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrProtoMsg] += 1;
#endif

  if (incoming_queue.enabled)
  {
    /* extract scalar args from head of message
     */
    type = (unsigned) ipi_in_reg(2);
    rgn  = (Region *) ipi_in_reg(3);
    src  = (unsigned) ipi_in_reg(4);
    vers = (unsigned) ipi_in_reg(5);

    dst_buf = UserFromMeta(rgn);
    ndwords = (rgn->size + 7) >> 3;

    /* flush recv buffer from memory system
     */
#if defined(FLUSH_MSG_BUFS)
    for (i=((ndwords<<3)-8); i>0; i-=16)
      hardflush2(dst_buf, i);
    hardflush(dst_buf);
#endif

    /* initiate data storeback
     */
    CReg->StoreAddr = (unsigned) dst_buf;
    ipicstnofix(-1, 3);

    actmsg_enqueue(rgn_data_msg, type, (unsigned) rgn, src, vers);
  }
  else
  {
    /* extract scalar args from head of message
     */
    type = (unsigned) ipi_in_reg(2);
    rgn  = (Region *) ipi_in_reg(3);
    src  = (unsigned) ipi_in_reg(4);
    vers = (unsigned) ipi_in_reg(5);

    dst_buf = UserFromMeta(rgn);
    ndwords = (rgn->size + 7) >> 3;

    /* flush recv buffer from memory system
     */
#if defined(FLUSH_MSG_BUFS)
    for (i=((ndwords<<3)-8); i>0; i-=16)
      hardflush2(dst_buf, i);
    hardflush(dst_buf);
#endif

    /* initiate data storeback
     */
    CReg->StoreAddr = (unsigned) dst_buf;
    ipicstnofix(-1, 3);

#if defined(USE_GLOBAL_TRANSITION)
    incoming_queue.enabled = 1;
    user_active_global();
#endif

    rgn_data_msg(type, rgn, src, vers);

#if defined(USE_GLOBAL_TRANSITION)
    actmsg_poll(0);
#endif
  }

#if (ALEWIFE_PROFILING != ProfileNone)
  ProfMsgEpilogue();
#endif
}


handler rgn_info_ack(unsigned vers, unsigned size, void *addr, RgnInfo *rtn)
{
#if (ALEWIFE_PROFILING != ProfileNone)
  /* we know a priori that we're always profiling when a rgn_info_ack
   * message arrives, so no need to fiddle with the counters
   */
  _msg_intr[1] += 1;
#endif

  rtn->vers      = vers - 1;
  rtn->size      = size;
  rtn->home_addr = addr;
  rtn->valid     = 1;
}


handler rgn_info_req(unsigned src, rid_t rgn_id, RgnInfo *rtn)
{
  Region *rgn;

#if (ALEWIFE_PROFILING != ProfileNone)
  ProfMsgPrologue_enable();
#endif

  rgn = rtable_lookup(rgn_id);
  sanity(rgn != NULL);

  sanity(src < crl_num_nodes);
  user_do_on(src, rgn_info_ack, rgn->vers, rgn->size, rgn, rtn); 

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRgnInfoAck] += 1;
#endif

#if (ALEWIFE_PROFILING != ProfileNone)
  ProfMsgEpilogue();
#endif
}


unsigned get_region_info(rid_t rgn_id, RgnInfo *info)
{
  unsigned home;
  unsigned self;

  /* figure out where home node is
   */
  home = RegionIdNode(rgn_id);
  sanity(home < crl_num_nodes);

  /* this procedure should only be called with region ids for remote
   * regions, so crash and burn if the home is the local node
   */
  self = crl_self_addr;
  sanity(home != self);

  /* query home node for info
   */
  info->valid = 0;
  user_do_on(home, rgn_info_req, self, rgn_id, info);

#if defined(STATISTICS)
  crl_stat_cntrs[StatCntrMsgRgnInfoReq] += 1;
#endif

  while (*((volatile unsigned *) &(info->valid)) == 0);

  return home;
}
