/*
 * tsp.c
 * deborah wallach
 * october 1994
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: tsp.c,v 1.5 1995/08/22 21:27:49 tuna Exp $
 */

#include <stdio.h>
#include <stdlib.h>

#if defined(CM5)
#include <cm/cmmd.h>
#define assert(x)                                  \
 do {                                              \
  if (!(x))                                        \
    CMMD_error("pn %d: failed assertion, %s:%d\n", \
	       crl_self_addr, __FILE__, __LINE__); \
 } while (0)
#elif defined(ALEWIFE)
#include <parallel.h>
#include <assert.h>
#else
#include <sys/time.h>
#include <sys/types.h>
#include <assert.h>
#endif

#include "crl.h"

/* #define DISABLE_INTERRUPTS */

#define NRTOWNS     (12)
#define INF         (100000)
/* #define INITIAL_MIN (245) */
#define INITIAL_MIN (INF)
#define MAXJOBS     (7920)
/* #define MAXJOBS (990) */
/* #define MAXJOBS (110) */
#define MAXHOPS     (4)
#define MASTER      (0)
#define MAXSLAVE    (128)

#define TRUE  (1)
#define FALSE (0)

struct dist {
  int totown;
  int dis;
} distance[NRTOWNS][NRTOWNS] = {
#include "12_cities.h"
};

int visited[NRTOWNS];
int path[NRTOWNS];
int my_address;
int num_nodes;
int nslaves;

int njobs = 0;

void tsp(int, int);
int present(int, int);
void distributor(int, int);
int slave();  /* returns # jobs solved */
void terminate();
void print_path(int *, int);

int main_continuation();

rid_t  boot_rid;
rid_t *boot_g;

#define SHARED (7)

/* int pathqueue[MAXJOBS][MAXHOPS]; */
/* shared, read-only (after creation) queue */
rid_t   pathqueue_rid;
int     *pathqueue_g;

/* count of size of queue so far */
rid_t   creator_pq_ptr_rid;
int     *creator_pq_ptr_g;

/* count of how much of queue has been read */
rid_t   slave_pq_ptr_rid;
int     *slave_pq_ptr_g;

/* sync variable - true when queue filled (ie creator read-only) */
rid_t   all_jobs_created_rid;
int     *all_jobs_created_g;

/* sync variable - counts # slaves totally finished */
rid_t 	nslave_done_rid;
int 	*nslave_done_g;

/* keeps track of minimum value found so far */
rid_t 	min_rid;
int 	*min_g;
int     cached_min;

/* sums jobs from each slave */
rid_t 	total_rid;
int 	*total_g;

void print_path(int *apath, int cities)
{
  int i;

  printf("%d: hops (%d) path : ", my_address, cities);
  for (i = 0; i < cities; i++) {
    printf("%d ", apath[i]);
  }
  printf("\n");
}

/* ----------- tsp functions ------------ */

int present(int e, int l)
{
  int i;

  for (i = 0; i <= l; i++) {
    if (path[i] == e) return 1;
  }
  return 0;
}

void distributor(int hops, int len)
{
  int e, me, i;

  if (len >= *min_g) {
    return;
  }
  visited[hops]++;
  if (hops == MAXHOPS) {
    assert(*creator_pq_ptr_g < MAXJOBS);
    for (e=0; e < MAXHOPS; e++) {
      *(pathqueue_g+MAXHOPS*(*creator_pq_ptr_g)+e) = path[e+1];
      /* pathqueue[*creator_pq_ptr_g][e] = path[e+1]; */
    }
    (*creator_pq_ptr_g)++;
    njobs++;
  } else {
    me = path[hops];
    for (i = 0; i < NRTOWNS; i++) {
      e = distance[me][i].totown;
      if (!present(e,hops)) {
	path[hops+1] = e;
	distributor(hops + 1, len + distance[me][i].dis);
      }
    }
  }
}

void create_jobs()
{
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_poll();
#endif
  rgn_start_read(min_g);
  rgn_start_write(pathqueue_g);
  rgn_start_write(creator_pq_ptr_g);
  distributor(0, 0);
  rgn_end_read(min_g);
  rgn_end_write(pathqueue_g);
  rgn_end_write(creator_pq_ptr_g);
  rgn_start_write(all_jobs_created_g);
  *all_jobs_created_g = TRUE;
  rgn_end_write(all_jobs_created_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_poll();
#endif
}


int compute_length(int *apath, int cities)
{
  int i;
  int len = 0;
  int me;
  int to;
  int j;

  for (i = 0; i < cities -1; i++) {
    me = apath[i];
    to = apath[i+1];
    for (j = 0; j < NRTOWNS; j++) {
      if (distance[me][j].totown == to) {
	len += distance[me][j].dis;
	break;
      }
    }
  }

  return len;
}

void tsp(int l, int len)
{
  int e, me, i;

  /* if (len >= *min_g) return;  /* pruning */
  if (len >= cached_min) return;  /* pruning */
  visited[l]++;
  if (l == NRTOWNS-1) {
    if (len < cached_min) {
      rgn_start_write(min_g);
      if (len < *min_g) { /* in case it changed */
	*min_g = len;
      }
      cached_min = *min_g;
      rgn_end_write(min_g);
    }
  } else {
    me = path[l];
    for (i = 0; i < NRTOWNS; i++) {
      if (me < 0 || me >= NRTOWNS) {
	fprintf(stderr, "me is wrong value %d\n", me);
	assert(0);
      }
      e = distance[me][i].totown;
      if (!present(e,l)) {
	path[l+1] = e;
	tsp(l+1, len + distance[me][i].dis);
      }
    }
  }
}


int slave()
{
  int length;
  int my_copy_slave_pq_ptr;
  int jobs = 0;

  rgn_start_read(all_jobs_created_g);
  while (*all_jobs_created_g == FALSE) {
    rgn_end_read(all_jobs_created_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
    CMAML_poll();
#endif
    rgn_start_read(all_jobs_created_g);
  }
  rgn_end_read(all_jobs_created_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_poll();
#endif

  rgn_start_read(creator_pq_ptr_g);

  rgn_start_write(slave_pq_ptr_g);
  while (*creator_pq_ptr_g > *slave_pq_ptr_g) {
    my_copy_slave_pq_ptr = *slave_pq_ptr_g;
    (*slave_pq_ptr_g)++;
    rgn_end_write(slave_pq_ptr_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
    CMAML_poll();
#endif

    rgn_start_read(pathqueue_g);
    path[0] = 0;
    path[1] = *(pathqueue_g+MAXHOPS*(my_copy_slave_pq_ptr)+0);
    path[2] = *(pathqueue_g+MAXHOPS*(my_copy_slave_pq_ptr)+1);
    path[3] = *(pathqueue_g+MAXHOPS*(my_copy_slave_pq_ptr)+2);
    path[4] = *(pathqueue_g+MAXHOPS*(my_copy_slave_pq_ptr)+3);
    rgn_end_read(pathqueue_g);

    length = compute_length(path, MAXHOPS+1);
    jobs++;
    rgn_start_read(min_g);
    cached_min = *min_g;
    rgn_end_read(min_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
    CMAML_poll();
#endif
    tsp(MAXHOPS, length);
    rgn_start_write(slave_pq_ptr_g);
  }

  rgn_end_write(slave_pq_ptr_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_poll();
#endif

  /* update total count of jobs solved */
  rgn_start_write(total_g);
  *(total_g) += jobs;
  rgn_end_write(total_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_poll();
#endif

  /* final sync - update count of finished slaves */
  rgn_start_write(nslave_done_g);
  (*nslave_done_g)++;
  rgn_end_write(nslave_done_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_poll();
#endif

  rgn_end_read(creator_pq_ptr_g);
  return(jobs);
}

void terminate()
{
  rgn_start_read(nslave_done_g);
  while (*nslave_done_g < nslaves) {
    rgn_end_read(nslave_done_g);
#if defined(CM5) && defined(DISABLE_INTERRUPTS)
    CMAML_poll();
#endif
    rgn_start_read(nslave_done_g);
  }
  rgn_end_read(nslave_done_g);
}


#if defined(TCPUNIX)
extern char *GROUP;
int main2(int argc, char **argv)
#else
int main(int argc, char **argv)
#endif
{

#if defined(CM5)
  CMMD_fset_io_mode(stdout, CMMD_independent);
  CMMD_fset_io_mode(stderr, CMMD_independent);
#endif

#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMMD_sync_with_nodes();
  CMAML_disable_interrupts();
#endif

  if (argc != 2) {
    fprintf(stderr, "Usage: %s <nslaves>\n", argv[0], argv[1]);
    assert(0);
  }

  nslaves = atoi(argv[1]);

#if defined(ALEWIFE)
  do_in_parallel(main_continuation, nslaves);
#else
  main_continuation(nslaves);
#endif

  return 0;
} /* main.c */


int main_continuation(int nslaves)
{
  int partition_size;
  double time;
#if defined(ALEWIFE)
  double timer_start, timer_stop;
#elif defined(TCPUNIX) || defined(NULLCRL)
  struct timeval start, finish;
#endif
  int jobs;

  partition_size = nslaves + 1;

#if defined(CM5)
  CMMD_sync_with_nodes();
  CMMD_reset_partition_size(partition_size);
  if(CMMD_self_address() >= partition_size) {
    exit(0);
  }
#endif

#if defined(TCPUNIX)
  crl_init(GROUP);
#else
  crl_init();
#endif

  assert(crl_num_nodes == partition_size);
  my_address = crl_self_addr;

  jobs = 0;

  if (my_address == MASTER) {
    boot_rid = rgn_create(sizeof(rid_t) * SHARED);
    rgn_bcast_send(sizeof(rid_t), &boot_rid);
  }
  else {
    rgn_bcast_recv(sizeof(rid_t), &boot_rid);
  }

  boot_g = rgn_map(boot_rid);

  if (my_address == MASTER) {
    pathqueue_rid = rgn_create(MAXJOBS*MAXHOPS*sizeof(int));
    creator_pq_ptr_rid = rgn_create(sizeof(int));
    slave_pq_ptr_rid = rgn_create(sizeof(int));
    all_jobs_created_rid = rgn_create(sizeof(int));
    nslave_done_rid = rgn_create(sizeof(int));
    min_rid = rgn_create(sizeof(int));
    total_rid = rgn_create(sizeof(int));

    rgn_start_write(boot_g);
    boot_g[0] = pathqueue_rid;
    boot_g[1] = creator_pq_ptr_rid;
    boot_g[2] = slave_pq_ptr_rid;
    boot_g[3] = all_jobs_created_rid;
    boot_g[4] = nslave_done_rid;
    boot_g[5] = min_rid;
    boot_g[6] = total_rid;
    rgn_end_write(boot_g);

    pathqueue_g = (int *) rgn_map(pathqueue_rid);
    creator_pq_ptr_g = (int *) rgn_map(creator_pq_ptr_rid);
    slave_pq_ptr_g = (int *) rgn_map(slave_pq_ptr_rid);
    all_jobs_created_g = (int *) rgn_map(all_jobs_created_rid);
    nslave_done_g = (int *) rgn_map(nslave_done_rid);
    min_g = (int *) rgn_map(min_rid);
    total_g = (int *) rgn_map(total_rid);


    rgn_start_write(creator_pq_ptr_g);
    *creator_pq_ptr_g = 0;
    rgn_end_write(creator_pq_ptr_g);
    rgn_start_write(slave_pq_ptr_g);
    *slave_pq_ptr_g = 0;
    rgn_end_write(slave_pq_ptr_g);
    rgn_start_write(all_jobs_created_g);
    *all_jobs_created_g = 0;
    rgn_end_write(all_jobs_created_g);
    rgn_start_write(nslave_done_g);
    *nslave_done_g = 0;
    rgn_end_write(nslave_done_g);
    rgn_start_write(min_g);
    *min_g = INITIAL_MIN;
    rgn_end_write(min_g);
    rgn_start_write(total_g);
    *total_g = 0;
    rgn_end_write(total_g);

  }

  rgn_barrier();

  if (my_address != MASTER) {

    rgn_start_read(boot_g);
    pathqueue_rid = boot_g[0];
    creator_pq_ptr_rid = boot_g[1];
    slave_pq_ptr_rid = boot_g[2];
    all_jobs_created_rid = boot_g[3];
    nslave_done_rid = boot_g[4];
    min_rid = boot_g[5];
    total_rid = boot_g[6];
    rgn_end_read(boot_g);

    pathqueue_g = (int *) rgn_map(pathqueue_rid);
    creator_pq_ptr_g = (int *) rgn_map(creator_pq_ptr_rid);
    slave_pq_ptr_g = (int *) rgn_map(slave_pq_ptr_rid);
    all_jobs_created_g = (int *) rgn_map(all_jobs_created_rid);
    nslave_done_g = (int *) rgn_map(nslave_done_rid);
    min_g = (int *) rgn_map(min_rid);
    total_g = (int *) rgn_map(total_rid);

  }

  rgn_barrier();

#if defined(ALEWIFE)
  timer_start = get_time();
#elif defined(CM5)
  CMMD_node_timer_clear(0);
  CMMD_node_timer_start(0);
#else
  gettimeofday(&start,NULL);
#endif

  if (my_address == MASTER) {
    /* master */
    create_jobs();
    if (nslaves > 0) {
      terminate();
    }
    else {
      jobs = slave();
    }

#if defined(ALEWIFE)
    /* assuming 20 MHz clock on alewife */
    timer_stop = get_time();
    time = (timer_stop-timer_start)/ 20e6;
#elif defined(CM5)
    CMMD_node_timer_stop(0);
    time = CMMD_node_timer_elapsed(0);
#else
    gettimeofday(&finish, NULL);
    time = (finish.tv_sec - start.tv_sec) +
      ((finish.tv_usec - start.tv_usec) * 1e-6);
#endif

    rgn_start_read(min_g);
    rgn_start_read(total_g);
    printf("All slaves (%d) terminated in %f sec (#jobs %d); the solution is %d\n",
	   nslaves, time, *total_g, *min_g);
    assert(njobs == *total_g);
    rgn_end_read(min_g);
    rgn_end_read(total_g);
    /* printf("njobs = %d\n", njobs); */
    printf("%d solved %d jobs\n", my_address, jobs);
  } else if (my_address < partition_size) {
    /* slave */
    jobs = slave();
    printf("%d solved %d jobs\n", my_address, jobs);
  } else {
    /* wait until everybody is done */
    assert(0);
  }
  rgn_barrier();

  return 0;
}
