/*
 * synth.c
 * kirk johnson
 * august 1994
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: synth.c,v 1.8 1995/08/22 21:27:49 tuna Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#if defined(CM5)
#include <cm/cmmd.h>
#define assert(x)                                  \
 do {                                              \
  if (!(x))                                        \
    CMMD_error("pn %d: failed assertion, %s:%d\n", \
	       crl_self_addr, __FILE__, __LINE__); \
 } while (0)
#elif defined(ALEWIFE)
#include <parallel.h>
#include <assert.h>
#else
#include <assert.h>
#include <sys/time.h>
#endif

#include "crl.h"

#define TargetNumWrites   (1000)
#define AccTabSiz         (1<<10)
#define NumReadFracValues (10)

typedef struct
{
  unsigned short read;
  unsigned short index;
} Access;

void      command_line(int, char **);
void      synth_worker(int, int);
void      synth_test(unsigned, unsigned, Access *);
rid_t    *synth_test_setup(unsigned, unsigned);
void      synth_test_rids(unsigned, rid_t *, unsigned, Access *);
void      synth_test_rgns(unsigned, void **, unsigned, Access *);
unsigned  read_proc(void *, unsigned);
void      write_proc(void *, unsigned, unsigned);
Access   *gen_access_table(unsigned, double);
unsigned  fast_random(unsigned);
void      timer_clear_and_start(void);
void      timer_stop_and_print(unsigned);

double read_frac_values[NumReadFracValues] =
{ 0.00, 0.10, 0.30, 0.50, 0.70, 0.90, 0.92, 0.96, 0.98, 0.99 };

unsigned nrgns_arg;
unsigned rgnsiz_arg;


#if defined(TCPUNIX)
extern char *GROUP;
int main2(int argc, char *argv[])
#else
int main(int argc, char *argv[])
#endif
{
#if defined(CM5)
  CMMD_set_io_mode(0, CMMD_independent);
  CMMD_set_io_mode(1, CMMD_independent);
  CMMD_set_io_mode(2, CMMD_independent);
#endif

#if defined(CM5) && defined(DISABLE_INTERRUPTS)
  CMAML_disable_interrupts();
#endif

  command_line(argc, argv);

#if defined(ALEWIFE)
  do_in_parallel(synth_worker, nrgns_arg, rgnsiz_arg);
#else
  synth_worker(nrgns_arg, rgnsiz_arg);
#endif

  return 0;
}


void command_line(int argc, char **argv)
{
  if (argc != 3)
  {
#if defined(CM5)
    if (CMMD_self_address() == 0)
      fprintf(stderr, "usage: synth <nrgns> <rgnsiz>\n");
#else
    fprintf(stderr, "usage: synth <nrgns> <rgnsiz>\n");
#endif
    exit(1);
  }

  sscanf(argv[1], "%u", &nrgns_arg);
  assert(nrgns_arg > 0);
  sscanf(argv[2], "%u", &rgnsiz_arg);
  assert((rgnsiz_arg > 0) && ((rgnsiz_arg % sizeof(unsigned)) == 0));
}


void synth_worker(int nrgns, int rgnsiz)
{
  unsigned i;
  double   read_frac;
  Access  *acctab;

  nrgns_arg  = nrgns;
  rgnsiz_arg = rgnsiz;

#if defined(TCPUNIX)
  crl_init(GROUP);
#else
  crl_init();
#endif

  for (i=0; i<NumReadFracValues; i++)
  {
    read_frac = read_frac_values[i];

    if (crl_self_addr == 0)
    {
      printf("(nrgns %d) (rgnsiz %d) (read %.3f) ",
	     nrgns, rgnsiz, read_frac);

#if defined(CM5)
#if defined(DISABLE_INTERRUPTS)
      printf("[CM-5, interrupts disabled]\n");
#else
      printf("[CM-5, interrupts enabled]\n");
#endif
#elif defined(ALEWIFE)
      printf("[Alewife]\n");
#elif defined(TCPUNIX)
      printf("[TCP/UNIX]\n");
#elif defined(NULLCRL)
      printf("[null]\n");
#endif
    }

    rgn_barrier();

    acctab = gen_access_table(nrgns, read_frac);
    synth_test(nrgns, rgnsiz, acctab);
    safe_free(acctab);
  }
}


void synth_test(unsigned nrgns, unsigned rgnsiz, Access *acctab)
{
  unsigned i;
  rid_t   *the_rids;
  void   **the_rgns;

  the_rids = synth_test_setup(nrgns, rgnsiz);
  rgn_start_read(the_rids);

  rgn_barrier();

  synth_test_rids(nrgns, the_rids, rgnsiz, acctab);
  fflush(stdout);

  rgn_barrier();

  the_rgns = (void **) safe_malloc(sizeof(void *) * nrgns);
  assert(the_rgns != NULL);

  for (i=0; i<nrgns; i++)
    the_rgns[i] = (void *) rgn_map(the_rids[i]);

  rgn_barrier();

  synth_test_rgns(nrgns, the_rgns, rgnsiz, acctab);
  fflush(stdout);

  rgn_barrier();

  for (i=0; i<nrgns; i++)
    rgn_unmap(the_rgns[i]);

  rgn_barrier();

  safe_free((void *) the_rgns);
}


rid_t *synth_test_setup(unsigned nrgns, unsigned rgnsiz)
{
  unsigned i;
  rid_t    top_rid;
  rid_t   *top_dat;
  rid_t    new_rid;
  void    *new_dat;

  /* allocate top
   */
  if (crl_self_addr == 0)
  {
    top_rid = rgn_create(sizeof(rid_t) * nrgns);
    rgn_bcast_send(sizeof(rid_t), &top_rid);
  }
  else
  {
    rgn_bcast_recv(sizeof(rid_t), &top_rid);
  }

  top_dat = (rid_t *) rgn_map(top_rid);

  /* allocate regions
   */
  for (i=crl_self_addr; i<nrgns; i+=crl_num_nodes)
  {
    new_rid = rgn_create(rgnsiz);
    new_dat = (void *) rgn_map(new_rid);
    rgn_start_write(new_dat);
    bzero((char *) new_dat, rgnsiz);
    rgn_end_write(new_dat);
    rgn_unmap(new_dat);

    rgn_start_write(top_dat);
    top_dat[i] = new_rid;
    rgn_end_write(top_dat);
  }

  /* wait until everybody is done
   */
  rgn_barrier();

  return top_dat;
}


void synth_test_rids(unsigned nrgns, rid_t *the_rids,
		     unsigned rgnsiz, Access *acctab)
{
  unsigned nreads;
  unsigned nwrites;
  Access  *access;
  rid_t    curr_rid;
  void    *curr_rgn;

  nreads  = 0;
  nwrites = 0;

  rgn_barrier();

  timer_clear_and_start();

  while (nwrites < TargetNumWrites)
  {
    access   = &(acctab[(nreads + nwrites) & (AccTabSiz-1)]);
    curr_rid = the_rids[access->index];
    curr_rgn = (void *) rgn_map(curr_rid);

    if (access->read)
    {
      rgn_start_read(curr_rgn);
      read_proc(curr_rgn, rgnsiz);
      rgn_end_read(curr_rgn);
      nreads += 1;
    }
    else
    {
      rgn_start_write(curr_rgn);
      write_proc(curr_rgn, rgnsiz, 0);
      rgn_end_write(curr_rgn);
      nwrites += 1;
    }

    rgn_unmap(curr_rgn);

#if defined(CM5) && defined(DISABLE_INTERRUPTS)
    CMAML_poll();
#endif
  }

  timer_stop_and_print(nreads + nwrites);
}


void synth_test_rgns(unsigned nrgns, void **the_rgns,
		     unsigned rgnsiz, Access *acctab)
{
  unsigned nreads;
  unsigned nwrites;
  Access  *access;
  void    *curr_rgn;

  nreads  = 0;
  nwrites = 0;

  rgn_barrier();

  timer_clear_and_start();

  while (nwrites < TargetNumWrites)
  {
    access   = &(acctab[(nreads + nwrites) & (AccTabSiz-1)]);
    curr_rgn = the_rgns[access->index];

    if (access->read)
    {
      rgn_start_read(curr_rgn);
      read_proc(curr_rgn, rgnsiz);
      rgn_end_read(curr_rgn);
      nreads += 1;
    }
    else
    {
      rgn_start_write(curr_rgn);
      write_proc(curr_rgn, rgnsiz, 0);
      rgn_end_write(curr_rgn);
      nwrites += 1;
    }

#if defined(CM5) && defined(DISABLE_INTERRUPTS)
    CMAML_poll();
#endif
  }

  timer_stop_and_print(nreads + nwrites);
}


unsigned read_proc(void *data, unsigned size)
{
  unsigned i;
  unsigned rslt;

  rslt = 0;
  for (i=0; i<size; i+=sizeof(unsigned))
    rslt += *((unsigned *) ((char *) data + i));

  return rslt;
}


void write_proc(void *data, unsigned size, unsigned val)
{
  int i;

  for (i=0; i<size; i+=sizeof(unsigned))
    *((unsigned *) ((char *) data + i)) = val;
}


Access *gen_access_table(unsigned nrgns, double read)
{
  unsigned i, j, k;
  unsigned self;
  unsigned state;
  Access   swap;
  Access  *rslt;

  self  = crl_self_addr;
  state = fast_random(self + 1);

  rslt = (Access *) safe_malloc(sizeof(Access) * AccTabSiz);
  assert(rslt != NULL);

  j = (unsigned) ((read * AccTabSiz) + 0.5);
  for (i=0; i<AccTabSiz; i++)
  {
    rslt[i].read  = (i < j);
    rslt[i].index = (i + self) % nrgns;
  }

  for (i=0; i<5; i++)
  {
    for (j=0; j<AccTabSiz; j++)
    {
      do
      {
	state = fast_random(state);
	k = (unsigned) (((double) state / 2147483647) * AccTabSiz);
      } while (k == j);

      swap    = rslt[k];
      rslt[k] = rslt[j];
      rslt[j] = swap;
    }
  }

  return rslt;
}


unsigned fast_random(unsigned state)
{
  unsigned lo, hi;
  int      test;

  hi = state / ((unsigned) 127773);
  lo = state % ((unsigned) 127773);
  test = (((unsigned) 16807) * lo) - (((unsigned) 2836) * hi);
  if (test > 0)
    state = test;
  else
    state = test + ((unsigned) 2147483647);

  return state;
}


#if defined(ALEWIFE) || defined(TCPUNIX) || defined(NULLCRL)
static double _timer_start;
#endif

void timer_clear_and_start(void)
{
#if defined(CM5)
  CMMD_node_timer_clear(0);
  CMMD_sync_with_nodes();
  CMMD_node_timer_start(0);
#elif defined(ALEWIFE)
  mp_spin_barrier();
  _timer_start = get_time();
#else
  struct timeval tp;
  gettimeofday(&tp, NULL);
  _timer_start = tp.tv_sec + (tp.tv_usec * 1e-6);
#endif
}


void timer_stop_and_print(unsigned count)
{
  double t1;
  double t2;
  double avg;
  double std;

  /* get elapsed time
   */
#if defined(CM5)
  CMMD_node_timer_stop(0);
  t1 = CMMD_node_timer_elapsed(0); /* seconds */
#elif defined(ALEWIFE)
  t1 = get_time();
  t1 = (t1 - _timer_start) * 1e-6; /* Mcycles */
#else
  struct timeval tp;
  gettimeofday(&tp,NULL);
  t1  = tp.tv_sec + (tp.tv_usec * 1e-6);
  t1 -= _timer_start;		/* seconds */
#endif

  /* convert elapsed time to rate
   */
  t1 = (count / t1); 

  /* compute sum and sum of squares
   */
  t2 = t1 * t1;
  t1 = rgn_reduce_dadd(t1);
  t2 = rgn_reduce_dadd(t2);

  /* compute mean and standard deviation
   */
  avg = t1 / crl_num_nodes;
  std = sqrt((t2 / crl_num_nodes) - (avg * avg));

  /* report to user
   */
  if (crl_self_addr == 0)
    printf(" %.3f +- %.3f\n", avg, std);
}
