/*
 * pvm_app.c
 * Sandeep Gupta
 * August 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: pvm_app.c,v 1.8 1995/08/22 21:27:49 tuna Exp $
 */

/*
 * This file contains the sample main() function for starting applications
 * with the TCP/UNIX implementation of CRL using PVM.
 */

#include <stdio.h>
#include <stdlib.h>

#if !defined(TCPUNIX)
#define TCPUNIX
#endif

#include "pvm3.h"
#include "crl.h"

char *GROUP = "crl_pvm_test";

main(int argc, char *argv[])
{
  int inum,tid;
  int numnodes;
  int retval;
  int parent_tid;    /* contains tid of parent process */
  int i;

  tid = pvm_mytid();  /* obtain own pid, and start pvm */

  if((parent_tid = pvm_parent()) == PvmNoParent) { /* This is the initial process */
    int slave_tid;
    int nhost, nprocs, narch;
    int i;
    struct pvmhostinfo *hostp;
    int numt;

    pvm_catchout(stdout);  /* redirect output of children to parent stdout */

    pvm_config(&nhost,&narch,&hostp);   /* get virtual machine configuration */

    nprocs = nhost;  /* Set number of processes to number of hosts */

    pvm_initsend(PvmDataRaw);  /* Send raw data since all architectures are
                                  compatible */

    /* Spawn each process is round robin style */
    for(i=0;i<nprocs;i++) {
      printf("host %d is on %s\n",i,(hostp+(i%nhost))->hi_name);
      numt = pvm_spawn(argv[0],argv+1,PvmTaskHost,(hostp+(i%nhost))->hi_name,1,&slave_tid);
      if(numt < 1) {
        printf("pvm_spawn returned %d, slave_tid is %d\n",numt,slave_tid);
        exit(1);
      }

      /* Send number of processors to the spawned process */
      retval = pvm_psend(slave_tid,0,&nprocs,1,PVM_INT);
      if(retval < 0) {
        printf("pvm_psend returned %d\n",retval);
        exit(1);
      }
    }

    /* Automatically wait until all spawns have exited because of pvm_catchout */
    pvm_exit();
    printf("Parent exiting\n");
    exit(0);
  }

  /* Join the group and get instance number */
  inum = pvm_joingroup(GROUP);

  /* Receive the number of CRL nodes */
  retval = pvm_recv(parent_tid,0);
  if(retval < 0) {
    printf("pvm_recv returned %d\n",retval);
    exit(1);
  }
  retval = pvm_upkint(&numnodes,1,1);
  if(retval < 0) {
    printf("pvm_upkint returned %d\n",retval);
    exit(1);
  }

  printf("Process with tid %d has inum %d and numnodes %d\n",tid,inum,numnodes);
  fflush(stdout);

  /* Wait for all processes to join the group */
  retval = pvm_barrier(GROUP, numnodes);
  if(retval < 0) {
    printf("pvm_barrier returned %d\n",retval);
    exit(1);
  }

  /* Code which uses CRL */
  main2(argc,argv);

  /* Make sure all nodes are done before exiting */
  retval = pvm_barrier(GROUP, numnodes);
  if(retval < 0) {
    printf("pvm_barrier returned %d\n",retval);
    exit(1);
  }

  /* Leave the group */
  retval = pvm_lvgroup(GROUP);
  if(retval < 0) {
    printf("pvm_lvgroup returned %d\n",retval);
    exit(1);
  }

  printf("Slave exiting\n");
  pvm_exit();
}
