/*
 * linreg.c
 * kirk johnson
 * february 1995
 *
 * Copyright (C) 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation. The author makes no
 * representations about the suitability of this software for any
 * purpose. It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * RCS $Id: linreg.c,v 1.5 1995/08/22 21:27:49 tuna Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "linreg.h"
#include "crl.h"

static void      mat_invert(int, double **, double **);
static void      mat_lu(int, double **, double **, double **);
static void      lmat_invert(int, double **, double **);
static void      umat_invert(int, double **, double **);
static void      mat_mul(int, double **, double **, double **);
static void      mat_copy(int n, double **, double **);
static double  **mat_alloc(int);
static void      mat_free(double **);


void linear_regression(int n, double *x, double *y, LinReg *rslt)
{
  int      i;
  double   sum_x;
  double   sum_y;
  double   sum_x2;
  double   sum_xy;
  double   sum_y2;
  double   x_val;
  double   y_val;
  double **t;
  double **c;
  double   b[2];
  double   sst;
  double   ssr;
  double   sse;
  double   s_e;

  sum_x  = 0;
  sum_y  = 0;
  sum_x2 = 0;
  sum_xy = 0;
  sum_y2 = 0;

  for (i=0; i<n; i++)
  {
    x_val = x[i];
    y_val = y[i];

    sum_x  += x_val;
    sum_y  += y_val;
    sum_x2 += x_val * x_val;
    sum_xy += x_val * y_val;
    sum_y2 += y_val * y_val;
  }

  t = mat_alloc(2);
  c = mat_alloc(2);

  t[0][0] = n;
  t[0][1] = sum_x;
  t[1][0] = sum_x;
  t[1][1] = sum_x2;
  mat_invert(2, t, c);

  b[0] = (c[0][0] * sum_y) + (c[0][1] * sum_xy);
  b[1] = (c[1][0] * sum_y) + (c[1][1] * sum_xy);

  sst = sum_y2 - (sum_y * sum_y / n);
  sse = sum_y2 - ((b[0] * sum_y) + (b[1] * sum_xy));
  ssr = sst - sse;
  s_e = sqrt(sse / (n - 2));

  rslt->b0_val = b[0];
  rslt->b0_std = s_e * sqrt(c[0][0]);
  rslt->b1_val = b[1];
  rslt->b1_std = s_e * sqrt(c[1][1]);
  rslt->sst    = sst;
  rslt->sse    = sse;
  rslt->ssr    = ssr;

  mat_free(t);
  mat_free(c);
}


/* invert an n by n matrix using LU decomposition
 */
static void mat_invert(int n, double **x, double **y)
{
  double **l;
  double **u;
  double **l_inv;
  double **u_inv;

  l     = mat_alloc(n);
  u     = mat_alloc(n);
  l_inv = mat_alloc(n);
  u_inv = mat_alloc(n);

  mat_lu(n, x, l, u);
  lmat_invert(n, l, l_inv);
  umat_invert(n, u, u_inv);
  mat_mul(n, u_inv, l_inv, y);

  mat_free(l);
  mat_free(u);
  mat_free(l_inv);
  mat_free(u_inv);
}



/* find LU decomposition of a matrix
 */
static void mat_lu(int size, double **a, double **l, double **u)
{
  int     k, m, n;
  double  pivot;
  double *row_m;
  double *row_k;
  double  tmp;

  mat_copy(size, a, l);

  for (k=0; k<size; k++)
  {
    row_k = l[k];
    pivot = row_k[k];

    tmp = 1 / pivot;
    for (m=(k+1); m<size; m++)
      l[m][k] *= tmp;

    for (m=(k+1); m<size; m++)
    {
      row_m = l[m];

      tmp = row_m[k];
      for (n=(k+1); n<size; n++)
	row_m[n] -= tmp * row_k[n];
    }
  }

  for (k=0; k<size; k++)
    for (m=0; m<size; m++)
      if (k < m)
      {
	u[k][m] = l[k][m];
	l[k][m] = 0;
      }
      else if (k > m)
      {
	u[k][m] = 0;
      }
      else
      {
	u[k][m] = l[k][m];
	l[k][m] = 1;
      }
}


/* invert an n by n lower triangular matrix
 */
static void lmat_invert(int n, double **x, double **y)
{
  int    i;
  int    r, c;
  double val;

  for (r=0; r<n; r++)
    for (c=0; c<n; c++)
    {
      if (r > c)
      {
	val = 0;
	for (i=c; i<r; i++)
	  val -= x[r][i] * y[i][c];
	val /= x[r][r];
      }
      else if (r == c)
      {
	val = 1 / x[r][r];
      }
      else
      {
	val = 0;
      }

      y[r][c] = val;
    }
}


/* invert an n by n upper triangular matrix
 */
static void umat_invert(int n, double **x, double **y)
{
  int    i;
  int    r, c;
  double val;

  for (r=(n-1); r>=0; r--)
    for (c=(n-1); c>=0; c--)
    {
      if (r < c)
      {
	val = 0;
	for (i=c; i>r; i--)
	  val -= x[r][i] * y[i][c];
	val /= x[r][r];
      }
      else if (r == c)
      {
	val = 1 / x[r][r];
      }
      else
      {
	val = 0;
      }

      y[r][c] = val;
    }
}


/* multiply two n by n matrices
 */
static void mat_mul(int n, double **x, double **y, double **z)
{
  int    i;
  int    r, c;
  double val;

  for (r=0; r<n; r++)
    for (c=0; c<n; c++)
    {
      val = 0;
      for (i=0; i<n; i++)
	val += x[r][i] * y[i][c];
      z[r][c] = val;
    }
}


/* copy an n by n matrix
 */
static void mat_copy(int n, double **x, double **y)
{
  bcopy(x[0], y[0], sizeof(double) * n * n);
}


/* allocate an n by n matrix
 */
static double **mat_alloc(int n)
{
  int      i;
  double **rslt;

  rslt = (double **) safe_malloc(sizeof(double *) * n);
  assert(rslt != NULL);

  rslt[0] = (double *) safe_malloc(sizeof(double) * n * n);
  assert(rslt[0] != NULL);

  for (i=1; i<n; i++)
    rslt[i] = rslt[i-1] + n;

  return rslt;
}


/* free the storage allocated for a matrix
 */
static void mat_free(double **x)
{
  safe_free(x[0]);
  safe_free(x);
}
